<?php

/**
  * Example IP-to-country lookup
  *
  * This file containts functions that are intended as an
  * example of how to add information to the User
  * Information attribute list in SquirrelMail.  This example
  * includes sample code that attempts to look up the country
  * where the user is located based on IP-to-country mapping
  * data obtained externally.
  *
  * If you decide to use this code, it is your responsibility
  * to obtain said data and adapt this code to use it correctly.
  * Some freely available IP-to-country schemes are available
  * if you search for them, including:
  *
  * http://newestindustry.org/2005/11/08/geographic-ip-database-using-perl-php-and-mysql/
  * http://ip-to-country.webhosting.info/
  * http://www.maxmind.com/app/geolitecountry
  *
  * Country codes and names are best stored per ISO 3166,
  * which can be found here:
  *
  * http://www.iso.org/iso/country_codes
  *
  * The example database lookup herein uses Pear DB
  * (type "pear install DB" from a Linux command line), but
  * you can easily change it to use some other database API
  * or file-based lookup, etc.
  *
  * A very simple database schema that would work with this is:
  *
  * CREATE TABLE ip_data(
  * start_ip_number double NOT NULL,
  * end_ip_number double NOT NULL,
  * country_name varchar(50) DEFAULT NULL,
  * country_code char(2) NOT NULL,
  * PRIMARY KEY primary_key (start_ip_number, end_ip_number)
  * ) ENGINE=MyISAM;
  *
  * Copyright (c) 2008-2009 Paul Lesniewski <paul@squirrelmail.org>
  * Licensed under the GNU GPL. For full terms see the file COPYING.
  *
  */



// use first line instead of second one here to see any error messages
//if (!include_once('DB.php'))
if (!@include_once('DB.php'))
{
   echo '<strong><font color="red">DB.php could not be loaded. Please make sure you have Pear installed.  For more details see <a href="http://pear.php.net" target="_blank">http://pear.php.net</a></font></strong>';
   exit;
}



/**
  * Example IP-to-country lookup function
  *
  * @param array $user_info A list of existing user information
  *                         attributes, including the key-value
  *                         pairs "ip_address" and "username".
  *
  * @return array A list of attributes for use by the
  *               SquirrelMail User Information plugin.
  *               Unless an error occurs, in which case
  *               these values will be empty, "country_name"
  *               and "country_code" (per ISO 3166) will
  *               be added to the user information by this
  *               module.  If there is no country
  *               information corresponding to the IP
  *               address, the country name and code will
  *               both be empty.
  *
  * Copyright (c) 2008-2009 Paul Lesniewski <paul@squirrelmail.org>
  * Licensed under the GNU GPL. For full terms see the file COPYING.
  *
  */
function lookup_country_by_ip_addr($user_info)
{

// ==================== BEGIN CONFIGURATION ====================

   // Should database connection problems be silently ignored?
   // in high-use production environments, this is probably a
   // good idea unless it is mission-critical that this lookup
   // succeed.
   //
   // 0 = halt and display error message upon error
   // 1 = ignore errors
   //
   $quiet = 1;


   // Theoretically, any SQL database supported by Pear should be supported
   // here.  The DSN (data source name) must contain the information needed
   // to connect to your database backend.  A MySQL example is included below.
   // For more details about DSN syntax and list of supported database types,
   // please see:
   //   http://pear.php.net/manual/en/package.database.db.intro-dsn.php
   //
   $dsn = 'mysql://user:password@localhost/ip_data';


   // You can massage this query to suit your environment.  It should
   // be designed to return (only) the country code and then country
   // name in that order.  The string %1$u will be replaced with the IP
   // number before it is sent to the database
   //
   $sql = 'SELECT country_code, country_name FROM ip_data WHERE start_ip_number <= %1$u AND end_ip_number >= %1$u';

// ==================== END CONFIGURATION ====================


   $country_name = '';
   $country_code = '';


   $ip_address = $user_info['ip_address'];
   $ip_number = ip2long($ip_address);
   //printf("IP ADDRESS: $ip_address<br />IP NUMBER: %u<hr />", $ip_number);


   // get database connection
   //
   $DB = country_by_ip_get_database_connection($dsn, $quiet);


   // now do the lookup
   //
   if ($DB)
   {
      $sql = sprintf($sql, $ip_number);
      $result = $DB->getAll($sql);


      // check for database errors
      //
      if (!$quiet && DB::isError($result))
      {
         $msg = $result->getMessage();
         echo '<strong><font color="red">Could not query database: ' . $msg . '</a></font></strong>';
         exit;
      }


      // grab results
      //
      if (!empty($result[0][0]))
         $country_code = $result[0][0];
      if (!empty($result[0][1]))
         $country_name = $result[0][1];

   }


   // return attributes for use in the User Information plugin
   //
   return array(
      'country_name' => $country_name,
      'country_code' => $country_code,
   );

}



/**
  * Get a database connection
  *
  * If a connection has already been opened, return that,
  * otherwise, open a new connection.
  *
  * @param string  $dsn   The database connection parameters
  * @param boolean $quiet If TRUE, connection errors are
  *                       silently ignored and FALSE is returned;
  *                       otherwise an error is displayed if
  *                       one occurs and execution halts
  *                       (OPTIONAL; default = FALSE).
  *
  * @return object The database connection handle, or FALSE
  *                if a database connection could not be made.
  *
  */
function country_by_ip_get_database_connection($dsn, $quiet=FALSE)
{

   global $COUNTRY_BY_IP_DB;


   // make a new connection if needed; exit if failure
   //
   if (empty($COUNTRY_BY_IP_DB))
   {

      $COUNTRY_BY_IP_DB = DB::connect($dsn);
      if (!$quiet && DB::isError($COUNTRY_BY_IP_DB))
      {
         echo '<strong><font color="red">Could not make database connection</a></font></strong>';
         exit;
      }

      if (!DB::isError($COUNTRY_BY_IP_DB))
         $COUNTRY_BY_IP_DB->setFetchMode(DB_FETCHMODE_ORDERED);
      else
         return FALSE;

   }


   // return connection
   //
   return $COUNTRY_BY_IP_DB;

}



