<?php

/**
  * SquirrelMail SMTP Authentication Plugin
  *
  * Copyright (c) 2011-2012 Paul Lesniewski <paul@squirrelmail.org>
  * Copyright (c) 2011 Emmanuel Dreyfus <manu@netbsd.org>
  *
  * Licensed under the GNU GPL. For full terms see the file COPYING.
  *
  * @package plugins
  * @subpackage smtp_auth
  *
  */



/**
  * Initialize this plugin (load config values)
  *
  * @param boolean $debug When TRUE, do not suppress errors when including
  *                       configuration files (OPTIONAL; default FALSE)
  *
  * @return boolean FALSE if no configuration file could be loaded, TRUE otherwise
  *
  */
function smtp_auth_init($debug=FALSE)
{

   if ($debug)
   {
      if (!include_once(SM_PATH . 'config/config_smtp_auth.php'))
         if (!include_once(SM_PATH . 'plugins/smtp_auth/config.php'))
            return FALSE;
   }
   else
   {
      if (!@include_once(SM_PATH . 'config/config_smtp_auth.php'))
         if (!@include_once(SM_PATH . 'plugins/smtp_auth/config.php'))
            return FALSE;
   }


   return TRUE;

}



/**
  * Authenticate user to the SMTP server in order to send email
  *
  * @param array $args An array consisting of these key value pairs (these
  *                    are the keys): "auth_mech", "user", "pass", "host",
  *                    "port" and "stream" (which is the open stream to the
  *                    SMTP server)
  *
  * @return boolean TRUE if the user has been authenticated, FALSE otherwise
  *
  */
function smtp_auth_authenticate($args)
{

   // find authentication mechanism (refer to global SquirrelMail
   // configuration (which is passed in the plugin arguments) if
   // one is not given for this plugin)
   //
   global $smtp_auth_type;
   smtp_auth_init();
   if (empty($smtp_auth_type))
      $smtp_auth_type = $args['auth_mech'];


   // if no authentication type has been configured or the
   // specified mechanism is handled by SquirrelMail, bail
   //
   if (empty($smtp_auth_type) || $smtp_auth_type === 'none'
    || $smtp_auth_type === 'login' || $smtp_auth_type === 'plain'
    || $smtp_auth_type === 'cram-md5' || $smtp_auth_type === 'digest-md5')
      return FALSE;


   // figure out what module to use for authentication check
   //
   if ($smtp_auth_type === 'saml_sasl')
      return authenticate_saml_sasl($args);


   // call external validation routine
   //
   else
      return $smtp_auth_type($args);

}



/**
  * Authenticate the user against the SAML SASL mechanism
  *
  * @param array $args An array consisting of these key value pairs (these
  *                    are the keys): "auth_mech" (which has already been
  *                    handled and can be ignored), "user", "pass", "host",
  *                    "port" and "stream" (which is the open stream to the
  *                    SMTP server)
  *
  * @return boolean TRUE if the user has been authenticated, FALSE otherwise
  *
  */
function authenticate_saml_sasl($args)
{

   global $saml_assertion_environment_variable,
          $compress_saml_assertion;
   smtp_auth_init();


   // make sure we have what we need
   //
   $saml_assertion = NULL;
   if (empty($args['user'])
    || !sqgetGlobalVar($saml_assertion_environment_variable, $saml_assertion, SQ_SERVER)
    || empty($saml_assertion))
      return FALSE;
//TODO: could log an error so as not to leave the admin in the dark here


   // build authentication string
   //
   if ($compress_saml_assertion)
      $saml_assertion = base64_encode(gzcompress(base64_decode($saml_assertion)));
   $auth = base64_encode($args['user'] . "\0" . $saml_assertion);


   // send to server
   //
   $bytes_written = fwrite($args['stream'], "AUTH SAML\r\n");
   if (!$bytes_written || !check_smtp_response($args['stream']))
      return FALSE;

   $bytes_written = fwrite($args['stream'], "$auth\r\n");
   if (!$bytes_written || !check_smtp_response($args['stream']))
      return FALSE;


   // no errors means authentication succeeded
   //
   return TRUE;

}



/**
  * Reads a response from the SMTP server and verifies
  * whether or not the server is returning a bad/error
  * response.
  *
  * @param resource $stream The open connection to the SMTP server
  *
  * @return boolean TRUE if the server response was OK, or FALSE if
  *                 the server returned a bad/error response
  *
  */
function check_smtp_response($stream)
{

   $response = fgets($stream, 1024);

   $error_number = substr($response, 0, 3);

   while(substr($response, 0, 4) == ($error_number . '-'))
      $line = fgets($stream, 1024);


   // 400 responses (and up) are errors
   //
   if (((int)$error_number{0}) < 4)
      return TRUE;

   return FALSE;

}



/**
  * Validate that this plugin is configured correctly
  *
  * @return boolean Whether or not there was a
  *                 configuration error for this plugin.
  *
  */
function smtp_auth_check_configuration()
{

   // only need to check the SquirrelMail version for versions
   // under 1.5.2, which will do that check for us automatically
   //
   if (!check_sm_version(1, 5, 2))
   {

      if (!check_sm_version(1, 4, 23)
       || check_sm_version(1, 5, 0)
       || check_sm_version(1, 5, 1))
      {
         do_err('The SMTP Authentication plugin requires SquirrelMail version 1.4.23 or above', FALSE);
         return TRUE;
      }

   }


   // make sure plugin is correctly configured
   //
   if (!smtp_auth_init())
   {
      do_err('The SMTP Authentication plugin is not configured correctly', FALSE);
      return TRUE;
   }


   global $smtp_auth_type, $compress_saml_assertion;


   if ($smtp_auth_type == 'saml_sasl'
    && $compress_saml_assertion
    && !function_exists('gzcompress'))
   {
      do_err('When using the "saml_sasl" module for the SMTP Authentication plugin, you cannot enable $compress_saml_assertion without PHP Zlib support.  See http://php.net/manual/en/zlib.installation.php', FALSE);
      return TRUE;
   }

   return FALSE;

}



