<?php

/**
  * SquirrelMail Shared Calendar Plugin
  * Copyright (C) 2004-2005 Paul Lesneiwski <pdontthink@angrynerds.com>
  * This program is licensed under GPL. See COPYING for details
  *
  */


/**
  * Event class
  *
  */
class Event
{

   // iCal standard attributes
   //
   var $id;
   var $sequence;
   var $type;
   var $summary;
   var $description;
   var $comments;
   var $status;
   var $priority;
   var $createdOn;
   var $lastUpdatedOn;
   var $thisObjectsTimestamp;  // not a Property object
   var $startDateTime;
   var $endDateTime;
   var $due;
   var $duration;
   var $recurrenceRule;
   var $recurrenceDates;
   var $recurrenceExclusionRule;
   var $recurrenceExclusionDates;
   var $percentComplete;


   // non-standard/internal attributes
   //
   var $dom;
   var $parentCalendars = array();
   var $owners = array();
   var $readable_users = array();
   var $writeable_users = array();
   var $unknownAttributes = array();
   var $createdBy;
   var $lastUpdatedBy;


   // other
   //
   var $startDateCache = array();
   var $endDateCache = array();
   var $cachedOccurrences = array();
   var $cachedOccurrencesThruDate;
   var $cachedEndDateTime = '';



   /**
     * Event constructor
     *
     * @param mixed $id The ID of this event
     *                  May be specified as a string or a Property object.
     * @param mixed $sequence The edit sequence ID for this event
     *                        May be specified as an int or a Property object.
     * @param mixed $dom The domain this event belongs under
     *                   May be specified as a string or a Property object.
     * @param mixed $type The type of this event, which should
     *                    correspond to the event type constants
     *                    defined in {@link constants.php}
     *                    May be specified as a string or a Property object.
     * @param mixed $summary The name of this event
     *                       May be specified as a string or a Property object.
     * @param mixed $description The description of this event
     *                           May be specified as a string or a Property object.
     * @param mixed $comments Administrative notes, etc
     *                        May be specified as a string or a Property object.
     * @param mixed $status This event/todo's current status, which should
     *                      correspond to the event or todo status constants
     *                      defined in {@link constants.php}
     *                      May be specified as a string or a Property object.
     * @param mixed $priority The priority of this event, which should
     *                        correspond to the event priority constants
     *                        defined in {@link constants.php}
     *                        May be specified as an int or a Property object.
     * @param mixed $startDateTime The date and time the event starts
     *                             May be specified as a UTC-formatted timestamp
     *                             (or just a date stamp (for all day events))
     *                             string or a Property object.
     * @param mixed $endDateTime The date and time the event ends
     *                           May be specified as a UTC-formatted timestamp 
     *                           (or just a date stamp (for all day events))
     *                           string or a Property object.
     * @param mixed $due The date and time the todo/task comes due
     *                   May be specified as a UTC-formatted timestamp 
     *                   string or a Property object.
     * @param mixed $duration The length of this event/todo, given as a 
     *                        period/duration string per vCal/iCal spec.
     *                        May be specified as a string or a Property object.
     * @param mixed $recurrenceRule The ruleset that determines recurrence of 
     *                              this event.
     *                              May be specified as a string or a Property object.
     * @param mixed $recurrenceDates A list of date(s) that determine when this
     *                               event should recur.
     *                               May be specified as an array of UTC-formatted 
     *                               timestamps or a Property object.
     * @param mixed $recurrenceExclusionRule The ruleset that determines recurrence 
     *                                       exceptions for this event.
     *                                       May be specified as a string or a Property object.
     * @param mixed $recurrenceExclusionDates A list of date(s) that determine 
     *                                        when this event should not recur.
     *                                        May be specified as an array of UTC-formatted 
     *                                        timestamps or a Property object.
     * @param mixed $percentComplete The completion status for this item
     *                               (for todo/tasks only)
     *                               May be specified as an int or a Property object.
     * @param mixed $parentCalendars An array of calendar IDs for calendars upon which
     *                               this event falls
     *                               May be specified an array or a Property object.
     * @param mixed $createdBy The name of the user who created this event
     *                         May be specified as a string or a Property object.
     * @param mixed $createdOn The date/time this event was created (optional; 
     *                         defaults to today's date)
     *                         May be specified as a UTC-formatted timestamp 
     *                         string or a Property object.
     * @param mixed $lastUpdatedBy The name of the user who last updated this event
     *                             May be specified as a string or a Property object.
     * @param mixed $lastUpdatedOn The date/time this event was last updated
     *                             May be specified as a UTC-formatted timestamp 
     *                             string or a Property object.
     * @param mixed $owners The users who share ownership of this event
     *                      May be specified an array or a Property object.
     * @param mixed $readable_users The users who have read access to this event
     *                              May be specified an array or a Property object.
     * @param mixed $writeable_users The users who have write access to this event
     *                               May be specified an array or a Property object.
     * @param array $unknownAttributes Extra unknown attributes in an
     *                                 array keyed by attribute name, although
     *                                 the value MUST be the full iCal line
     *                                 describing the property, INCLUDING its
     *                                 name.  These properties are often
     *                                 derived from custom attributes from an
     *                                 imported iCal file
     *
     */
   function Event($id='', $sequence=0, $dom='', $type='INVALID',
                  $summary='', $description='', 
                  $comments='', $status='', 
                  $priority=SM_CAL_EVENT_PRIORITY_NORMAL, 
                  $startDateTime='', $endDateTime='', $due='', $duration='',
                  $recurrenceRule='', $recurrenceDates=array(), 
                  $recurrenceExclusionRule='', $recurrenceExclusionDates=array(), 
                  $percentComplete=0,
                  $parentCalendars=array(), $createdBy='', $createdOn='', 
                  $lastUpdatedBy='', $lastUpdatedOn='', $owners=array(), 
                  $readable_users=array(), $writeable_users=array(),
                  $unknownAttributes=array())
   {

      $this->thisObjectsTimestamp = time(); 

      if (is_object($id) && strtolower(get_class($id)) == 'property')
         $this->id               = $id;
      else
         $this->id               = new Property('UID', $id);

      if (is_object($sequence) && strtolower(get_class($sequence)) == 'property')
         $this->sequence         = $sequence;
      else
         $this->sequence         = new Property('SEQUENCE', $sequence);

      if (is_object($dom) && strtolower(get_class($dom)) == 'property')
         $this->dom              = $dom;
      else
         $this->dom              = new Property('X-SQ-EVTDOMAIN',
                                    strtr($dom, '@|_-.:/ \\', '________'));

      if (is_object($type) && strtolower(get_class($type)) == 'property')
         $this->type             = $type;
      else
         $this->type             = new Property('NO-ICAL-TYPE', $type);

      if (is_object($summary) && strtolower(get_class($summary)) == 'property')
         $this->summary          = $summary;
      else
         $this->summary          = new Property('SUMMARY', $summary);

      if (is_object($description) && strtolower(get_class($description)) == 'property')
         $this->description      = $description;
      else
         $this->description      = new Property('DESCRIPTION', $description);

      if (is_object($comments) && strtolower(get_class($comments)) == 'property')
         $this->comments         = $comments;
      else
         $this->comments         = new Property('COMMENT', $comments);

      if (is_object($status) && strtolower(get_class($status)) == 'property')
         $this->status           = $status;
      else
         $this->status           = new Property('STATUS', $status);

      if (is_object($priority) && strtolower(get_class($priority)) == 'property')
         $this->priority         = $priority;
      else
         $this->priority         = new Property('PRIORITY', $priority);

      if (is_object($startDateTime) && strtolower(get_class($startDateTime)) == 'property')
         $this->startDateTime         = $startDateTime;
      else if (strpos($startDateTime, 'T') !== FALSE)
         $this->startDateTime         = new Property('DTSTART', $startDateTime, 
                                        array(), SM_CAL_ICAL_PROPERTY_TYPE_DATETIME_UTC);
      else
         $this->startDateTime         = new Property('DTSTART', $startDateTime, 
                                        array(), SM_CAL_ICAL_PROPERTY_TYPE_DATE);

      if (is_object($endDateTime) && strtolower(get_class($endDateTime)) == 'property')
         $this->endDateTime         = $endDateTime;
      else if (strpos($endDateTime, 'T') !== FALSE)
         $this->endDateTime         = new Property('DTEND', $endDateTime,
                                      array(), SM_CAL_ICAL_PROPERTY_TYPE_DATETIME_UTC);
      else
         $this->endDateTime         = new Property('DTEND', $endDateTime,
                                      array(), SM_CAL_ICAL_PROPERTY_TYPE_DATE);

      if (is_object($due) && strtolower(get_class($due)) == 'property')
         $this->due         = $due;
      else
         $this->due         = new Property('DUE', $due,
                                      array(), SM_CAL_ICAL_PROPERTY_TYPE_DATETIME_UTC);

      if (is_object($duration) && strtolower(get_class($duration)) == 'property')
         $this->duration  = $duration;
      else
         $this->duration  = new Property('DURATION', $duration);

      if (is_object($recurrenceRule) && strtolower(get_class($recurrenceRule)) == 'property')
         $this->recurrenceRule  = $recurrenceRule;
      else
         $this->recurrenceRule  = Property::extractICalProperty('RRULE:' . $recurrenceRule, SM_CAL_ICAL_PROPERTY_TYPE_RRULE);
                                 //new Property('RRULE', $recurrenceRule, 
                                  //$recurrenceRule, SM_CAL_ICAL_PROPERTY_TYPE_RRULE);

      if (is_object($recurrenceDates) && strtolower(get_class($recurrenceDates)) == 'property')
         $this->recurrenceDates  = $recurrenceDates;
      else
         $this->recurrenceDates  = new Property('RDATE', $recurrenceDates,
                                 array(), SM_CAL_ICAL_PROPERTY_TYPE_DATETIME_UTC);

      if (is_object($recurrenceExclusionRule) && strtolower(get_class($recurrenceExclusionRule)) == 'property')
         $this->recurrenceExclusionRule  = $recurrenceExclusionRule;
      else
         $this->recurrenceExclusionRule  = Property::extractICalProperty('EXRULE:' . $recurrenceExclusionRule, SM_CAL_ICAL_PROPERTY_TYPE_RRULE);

      if (is_object($recurrenceExclusionDates) && strtolower(get_class($recurrenceExclusionDates)) == 'property')
         $this->recurrenceExclusionDates  = $recurrenceExclusionDates;
      else
         $this->recurrenceExclusionDates  = new Property('EXDATE', $recurrenceExclusionDates,
                                 array(), SM_CAL_ICAL_PROPERTY_TYPE_DATETIME_UTC);

      if (is_object($percentComplete) && strtolower(get_class($percentComplete)) == 'property')
         $this->percentComplete  = $percentComplete;
      else
         $this->percentComplete  = new Property('PERCENT-COMPLETE', $percentComplete);

      if (is_object($parentCalendars) && strtolower(get_class($parentCalendars)) == 'property')
         $this->parentCalendars  = $parentCalendars;
      else
         $this->parentCalendars  = new Property('X-SQ-EVTPARENTCALENDARS', $parentCalendars);

      if (is_object($createdBy) && strtolower(get_class($createdBy)) == 'property')
         $this->createdBy        = $createdBy;
      else
         $this->createdBy        = new Property('X-SQ-EVTCREATOR', $createdBy);

      if (is_object($createdOn) && strtolower(get_class($createdOn)) == 'property')
         $this->createdOn        = $createdOn;
      else
         $this->createdOn        = new Property('CREATED', 
                                 (empty($createdOn) ? gmdate('Ymd\THis\Z') : $createdOn),
                                 array(), SM_CAL_ICAL_PROPERTY_TYPE_DATETIME_UTC);

      if (is_object($lastUpdatedBy) && strtolower(get_class($lastUpdatedBy)) == 'property')
         $this->lastUpdatedBy    = $lastUpdatedBy;
      else
         $this->lastUpdatedBy    = new Property('X-SQ-EVTLASTUPDATOR', 
                                 (empty($lastUpdatedBy) ? $this->createdBy() : $lastUpdatedBy));

      if (is_object($lastUpdatedOn) && strtolower(get_class($lastUpdatedOn)) == 'property')
         $this->lastUpdatedOn    = $lastUpdatedOn;
      else
         $this->lastUpdatedOn    = new Property('LAST-MODIFIED', 
                    (empty($lastUpdatedOn) ? $this->createdOn->getRawValue() : $lastUpdatedOn),
                    array(), SM_CAL_ICAL_PROPERTY_TYPE_DATETIME_UTC);

      if (is_object($owners) && strtolower(get_class($owners)) == 'property')
         $this->owners           = $owners;
      else
         $this->owners           = new Property('X-SQ-EVTOWNERS', $owners);

      if (is_object($readable_users) && strtolower(get_class($readable_users)) == 'property')
         $this->readable_users   = $readable_users;
      else
         $this->readable_users   = new Property('X-SQ-EVTREADABLEUSERS', $readable_users);

      if (is_object($writeable_users) && strtolower(get_class($writeable_users)) == 'property')
         $this->writeable_users  = $writeable_users;
      else
         $this->writeable_users  = new Property('X-SQ-EVTWRITEABLEUSERS', $writeable_users);

      $this->unknownAttributes = $unknownAttributes;



      global $domain, $username, $useDomainInCalID;


      // insert default values if not given above
      //
      // note that user is only given read access by default;
      // caller must explicitly give any more than that
      //
      $o = $this->getOwners();
      $w = $this->getWriteableUsers();
      $r = $this->getReadableUsers();
      if (empty($o) && empty($w) && empty($r))
         $this->readable_users->setValue(array($username));

      $c = $this->createdBy();
      if (empty($c))
      $this->setCreator($username);


      // um, we actually want events to have the same permissions
      // that their parent calendars have... per-event permissioning
      // is not currently useful; this is the least costly place
      // to duplicate parent calendar permissions, short of removing
      // the checks in the code for event permissions
      //
      $this->resetPermissionsToParent();


      $d = $this->dom->getValue();
      if (empty($d))
         $this->dom->setValue(strtr($domain, '@|_-.:/ \\', '________'));

      $i = $this->getID();
      if (empty($i))
         $this->id->setValue('sm_cal_evt_' . gmdate('Ymd\THis\Z') 
                           . ($useDomainInCalID ? '_' . $this->getDomain() : ''));



      // make sure RDATE and EXDATE values are always ordered arrays
      // 
      $rdates = $this->getRecurrenceDates();
      if (empty($rdates)) $this->recurrenceDates->resetValue(array());
      else if (!is_array($rdates)) $this->recurrenceDates->resetValue(array($rdates));
      else
      {

         // is this a PERIOD or regular date/time?
         //
         if ($this->recurrenceDates->getType() == SM_CAL_ICAL_PROPERTY_TYPE_PERIOD)
         {

            // values are always arrays (start/end of period),
            // but multiple periods will be stored as arrays of
            // arrays (start/end, start/end, start/end, etc)
            //
            // we only need to resort if there is more than
            // one period, which we can determine by seeing
            // if we have nested values here (if not, make
            // sure we make it nested anyway)
            //
            if (!is_array($rdates[0]))
               $this->recurrenceDates->resetValue(array($rdates));

            // sort nested arrays based on the period's starting 
            // (and secondly ending) timestamp(s)
            //
            else
            {
               usort($rdates, 'sortPeriods'); // see functions.php for function sortPeriods()
               $this->recurrenceDates->resetValue($rdates);
            }

         }


         // regular date or date/time - just make sure they are sorted
         //
         else
         {
            sort($rdates);
            $this->recurrenceDates->resetValue($rdates);
         }

      }

      $exdates = $this->getRecurrenceExclusionDates();
      if (empty($exdates)) $this->recurrenceExclusionDates->resetValue(array());
      else if (!is_array($exdates)) $this->recurrenceExclusionDates->resetValue(array($exdates));
      else
      {
         sort($exdates);
         $this->recurrenceExclusionDates->resetValue($exdates);
      }



      return $this->recurrenceDates->getValue();


   }



// ---------- PRIVATE ----------


 
   /**
     * Find the next possible event occurrence for a given interval
     * in the recurrence rule.
     *
     * Also considers if there are any recurrence dates that come
     * first.
     *
     * @param array $recurrenceRule The rule to follow for the given 
     *                              recurrence interval
     * @param array $recurrenceDates The set of recurrence dates 
     *                               for the event (passed by ref)
     * @param int $currentOccurrenceDate Timestamp for the current
     *                                   event iteration, after which
     *                                   to start looking for the next
     *                                   occurrence (optional, if given
     *                                   as empty string, the first
     *                                   possible occurrence is returned)
     * @param int $currentIntervalBasis The timestamp of the base date
     *                                  that defines this interval
     * @param int $maxDate The timestamp indicating the date past which
     *                     we needn't keep iterating (optional; default
     *                     is the constant MAX_RECURRENCE_DATE, as 
     *                     defined in {@link constants.php})
     *                     NOTE that to avoid time-of-day problems,
     *                     we typically iterate one day past this date
     *
     * @return mixed If a regular occurrence was found following the 
     *               given recurrence rule, that timestamp is returned;
     *               If a date was instead found from amongst the given
     *               recurrence dates, that timestamp is returned inside
     *               a one-element array;  If no occurrences are found
     *               in this particular interval, 0 (zero) is returned;
     *               If we exceed either $maxDate or the UNTIL clause
     *               of the given recurrence rule, -1 is returned.
     *
     * @access private
     *
     */
   function inspectIntraIntervalOccurrences($recurrenceRule, &$recurrenceDates, 
                                            $currentOccurrenceDate='', 
                                            $currentIntervalBasis, 
                                            $maxDate=MAX_RECURRENCE_DATE)
   {

      global $WEEKDAYS, $week_start_day;


      // no recurrence rule?  just bail
      //
      if (empty($recurrenceRule)) return 0;


      // add a day to maxDate
      //
      $maxDate += SM_CAL_DAY_SECONDS;


// just shorthand notes... not technically all correct... see RFC for that...
// BYSECOND=[0-59]{1,*}
// BYMINUTE=[0-59]{1,*}
// BYHOUR=[0-23]{1,*}
// BYDAY=[[+/-][1-53]][MO/TU/WE/TH/FR/SA/SU]
//    leading number only allowable when freq is monthly or yearly
// BYMONTHDAY=[+/-][1.31]{1,*}
// BYYEARDAY=[+/-][1-366]{1,*}
// BYWEEKNO=[+/-][1-53]{1,*}
//   only valid with YEARLY freq
// BYMONTH=[1-12]{1,*}
// BYSETPOS=[+/-][1-366]{1,*}
//   if given, one other BY**** MUST be given as well (well, ok, we can just ignore it if not)
// WKST=[MO/TU/WE/TH/FR/SA/SU]{1}
//   default = MO, only applicable to ((FREQ == WEEKLY && INTERVAL > 1 && !empty(BYDAY))
//                                  || (FREQ == YEARLY && !empty(BYWEEKNO)))


      list($year, $month, $day, $hour, $minute, $second, $dayOfWeek, 
           $weekNumber, $numberOfDaysInMonth)
       = explode('-', date('Y-n-j-G-i-s-w-W-t', $currentIntervalBasis));

      // can be 53 and 366 some years (but note that z is zero-based)
      //
      list($numberOfWeeksInYear, $numberOfDaysInYear)
       = explode('-', date('W-z', mktime(12, 0, 0, 12, 31, $year))); 
      $numberOfDaysInYear++;

      if (!isset($recurrenceRule['WKST'])) // don't check if empty, since zero is valid
         $weekStart = $week_start_day;
      else
         $weekStart = $recurrenceRule['WKST'];
      $weekEnd = $weekStart - 1;
      if ($weekEnd == -1) $weekEnd = 6;
      $occurrences = array();


      // build an array of occurrences within the given interval
      //
      switch ($recurrenceRule['FREQ'])
      {

         //---------------------------------------------
         //
         case SM_CAL_ICAL_EVENT_RECURRENCE_FREQ_YEARLY:

            // BYMONTHLY
            //
            foreach ($recurrenceRule['BYMONTH'] as $MON)
               $occurrences[] = mktime($hour, $minute, $second, $MON, $day, $year);



            // BYWEEKNO
            //
            // Use date of "WKST" for each week as our occurrence set
            //
            $weeknoOccurrences = array();
            foreach ($recurrenceRule['BYWEEKNO'] as $WEEKNO)
            {
               if ($WEEKNO < 0)
                  $WEEKNO = $numberOfWeeksInYear + $WEEKNO + 1;
               $weeknoOccurrences[] = datefromweeknr($year, $WEEKNO, $weekStart);
            }
            $temp_occurrences = array();
            if (empty($occurrences)) $occurrences = $weeknoOccurrences;

            // unless BYMONTHLY was given, in which case we limit
            // our set to those BYWEEKNO occurrences that fall in the
            // months given by BYMONTHLY
            //
            else if (!empty($weeknoOccurrences))
            {
               foreach ($weeknoOccurrences as $timestamp)
               {
                  foreach ($occurrences as $occurStamp)
                  {
                     if (date('n', $timestamp) == date('n', $occurStamp))
                     {
                        $temp_occurrences[] = $timestamp;
                        break;
                     }
                  }
               }
               $occurrences = $temp_occurrences;
            }



            // BYYEARDAY
            //
            // Straight-forward...
            //
            $yearDayOccurrences = array();
            foreach ($recurrenceRule['BYYEARDAY'] as $YEARDAY)
            {
               if ($YEARDAY < 0)
                  $YEARDAY = $numberOfDaysInYear + $YEARDAY + 1;
               $yearDayOccurrences[] = mktime($hour, $minute, $second, 1, $YEARDAY, $year);
            }
            $temp_occurrences = array();
            if (empty($occurrences)) $occurrences = $yearDayOccurrences;

            // unless one of the above was given too...
            //
            else if (!empty($yearDayOccurrences))
            {
               foreach ($yearDayOccurrences as $timestamp)
               {
                  foreach ($occurrences as $occurStamp)
                  {

                     // when only BYMONTH was given, limit occurrence set
                     // to those year days that fall in the given months
                     //
                     if ((!empty($recurrenceRule['BYMONTH']) && empty($recurrenceRule['BYWEEKNO'])
                      && date('n', $timestamp) == date('n', $occurStamp))
   
                     // when only BYWEEKNO was given, limit occurrence set
                     // to the year days that come in the BYWEEKNO weeks
                     //
                      || (empty($recurrenceRule['BYMONTH']) && !empty($recurrenceRule['BYWEEKNO'])
                      && $timestamp >= $occurStamp && $timestamp < $occurStamp + SM_CAL_WEEK_SECONDS)
   
                     // when both BYMONTHLY *and* BYWEEKNO are given, limit occurrence set
                     // to the year days that come in the BYWEEKNO weeks only in the
                     // BYMONTHLY months.  whew.
                     //
                      || (!empty($recurrenceRule['BYMONTH']) && !empty($recurrenceRule['BYWEEKNO'])
                      && $timestamp >= $occurStamp && $timestamp < $occurStamp + SM_CAL_WEEK_SECONDS
                      && date('n', $timestamp) == date('n', $occurStamp)))
                     {
                        $temp_occurrences[] = $timestamp;
                        break;
                     }
                  }
               }
               $occurrences = $temp_occurrences;
            }



            // BYMONTHDAY
            //
            // Straight-forward...
            //
            $monthDayOccurrences = array();
            if (!empty($recurrenceRule['BYMONTHDAY'])) for ($m = 1; $m < 13; $m++)
            {
               $numberOfDaysInMo = date('t', mktime(12, 0, 0, $m, 15, $year));
               foreach ($recurrenceRule['BYMONTHDAY'] as $DAY)
               {
                  if ($DAY < 0) 
                     $DAY = $numberOfDaysInMo + $DAY + 1;
                  $monthDayOccurrences[] = mktime($hour, $minute, $second, $m, $DAY, $year);
               }
            }
            $temp_occurrences = array();
            if (empty($occurrences)) $occurrences = $monthDayOccurrences;

            // unless one of the above was given too...
            //
            else if (!empty($monthDayOccurrences))
            {
               foreach ($monthDayOccurrences as $timestamp)
               {
                  foreach ($occurrences as $occurStamp)
                  {

                     // when only BYMONTH was given, limit occurrence set
                     // to those month days that fall in the given months
                     //
                     if ((!empty($recurrenceRule['BYMONTH']) && empty($recurrenceRule['BYWEEKNO'])
                      && empty($recurrenceRule['BYYEARDAY'])
                      && date('n', $timestamp) == date('n', $occurStamp))
   
                     // when only BYWEEKNO was given, limit occurrence set
                     // to the month days that come in the BYWEEKNO weeks
                     //
                      || (empty($recurrenceRule['BYMONTH']) && !empty($recurrenceRule['BYWEEKNO'])
                      && empty($recurrenceRule['BYYEARDAY'])
                      && $timestamp >= $occurStamp && $timestamp < $occurStamp + SM_CAL_WEEK_SECONDS)
   
                     // when both BYMONTHLY *and* BYWEEKNO are given, limit occurrence set
                     // to the month days that come in the BYWEEKNO weeks only in the
                     // BYMONTHLY months.  whew.
                     //
                      || (!empty($recurrenceRule['BYMONTH']) && !empty($recurrenceRule['BYWEEKNO'])
                      && empty($recurrenceRule['BYYEARDAY'])
                      && $timestamp >= $occurStamp && $timestamp < $occurStamp + SM_CAL_WEEK_SECONDS
                      && date('n', $timestamp) == date('n', $occurStamp))
   
                     // when BYYEARDAY was given - no matter what 
                     // other fields are given, limit occurrence set
                     // to the month days that intersect those year 
                     // days already selected
                     //
                      || (!empty($recurrenceRule['BYYEARDAY'])
                      && $timestamp == $occurStamp))
                     {
                        $temp_occurrences[] = $timestamp;
                        break;
                     }
                  }
               }
               $occurrences = $temp_occurrences;
            }



            // BYDAY
            //
            // pick all days in the year that are in BYDAY,
            // optionally limited by the count given in front of BYDAY values
            //
            $dayOccurrences = array();
            $dayOfYearCount = array(0 => array(), 1 => array(), 2 => array(), 
                                    3 => array(), 4 => array(), 5 => array(), 
                                    6 => array());
            $byDayRule = array();
            foreach ($recurrenceRule['BYDAY'] as $DAY)
            {
               if (!array_key_exists(strtoupper($DAY), $WEEKDAYS))
               {
                  $dayCount = substr($DAY, 0, strlen($DAY) - 2);
                  $dy = $WEEKDAYS[substr($DAY, strlen($DAY) - 2)];
               }
               else
               {
                  $dayCount = 0;
                  $dy = $WEEKDAYS[$DAY];
               }
//LEFT OFF HERE: make sure zero array indexes don't screw something up
               $byDayRule[$dy] = $dayCount;
            }
            if (!empty($recurrenceRule['BYDAY'])) for ($d = 1; $d <= $numberOfDaysInYear; $d++)
            {
               $timestamp = mktime($hour, $minute, $second, 1, $d, $year);
               $dyOfYr = date('w', $timestamp);
               $dayOfYearCount[$dyOfYr][] = $timestamp;
               foreach ($byDayRule as $DAY => $dayCount)
               {
                  if ($DAY == $dyOfYr 
                   && ($dayCount === 0 || $dayCount == sizeof($dayOfYearCount[$DAY])))
                     $dayOccurrences[] = $timestamp;
               }
            }
            // get negative offsets
            //
            foreach ($byDayRule as $DAY => $dayCount)
            {
               if ($dayCount < 0)
               {
                  $dayOccurrences[] 
                   = $dayOfYearCount[$DAY][sizeof($dayOfYearCount[$DAY]) + $dayCount];
               }
            }
            $temp_occurrences = array();
            if (empty($occurrences)) $occurrences = $dayOccurrences;

            // unless one of the above was given too...
            //
            else if (!empty($dayOccurrences))
            {
               foreach ($dayOccurrences as $timestamp)
               {
                  foreach ($occurrences as $occurStamp)
                  {

                     // when only BYMONTH was given, limit occurrence set
                     // to those days that fall in the given months
                     //
                     if ((!empty($recurrenceRule['BYMONTH']) && empty($recurrenceRule['BYWEEKNO'])
                      && empty($recurrenceRule['BYYEARDAY']) && empty($recurrenceRule['BYMONTHDAY'])
                      && date('n', $timestamp) == date('n', $occurStamp))
   
                     // when only BYWEEKNO was given, limit occurrence set
                     // to the days that come in the BYWEEKNO weeks
                     //
                      || (empty($recurrenceRule['BYMONTH']) && !empty($recurrenceRule['BYWEEKNO'])
                      && empty($recurrenceRule['BYYEARDAY']) && empty($recurrenceRule['BYMONTHDAY'])
                      && $timestamp >= $occurStamp && $timestamp < $occurStamp + SM_CAL_WEEK_SECONDS)
   
                     // when both BYMONTHLY *and* BYWEEKNO are given, limit occurrence set
                     // to the days that come in the BYWEEKNO weeks only in the
                     // BYMONTHLY months.  whew.
                     //
                      || (!empty($recurrenceRule['BYMONTH']) && !empty($recurrenceRule['BYWEEKNO'])
                      && empty($recurrenceRule['BYYEARDAY']) && empty($recurrenceRule['BYMONTHDAY'])
                      && $timestamp >= $occurStamp && $timestamp < $occurStamp + SM_CAL_WEEK_SECONDS
                      && date('n', $timestamp) == date('n', $occurStamp))
   
                     // when either BYYEARDAY or BYMONTHDAY was 
                     // given - no matter what other fields are 
                     // given, limit occurrence set to the 
                     // days that intersect those year days 
                     // and month days already selected
                     //
                      || ((!empty($recurrenceRule['BYYEARDAY']) 
                      || !empty($recurrenceRule['BYMONTHDAY']))
                      && $timestamp == $occurStamp))
                     {
                        $temp_occurrences[] = $timestamp;
                        break;
                     }
                  }
               }
               $occurrences = $temp_occurrences;
            }


            
            // BYHOUR
            // 
            // Straight-forward...
            // 
            $hourlyOccurrences = array();
            if (!empty($recurrenceRule['BYHOUR'])) for ($d = 1; $d <= $numberOfDaysInYear; $d++)
            {
               foreach ($recurrenceRule['BYHOUR'] as $h)
                  $hourlyOccurrences[] = mktime($h, $minute, $second, 1, $d, $year);
            }
            $temp_occurrences = array();
            if (empty($occurrences)) $occurrences = $hourlyOccurrences;

            // unless one of the above was given too...
            //
            else if (!empty($hourlyOccurrences))
            {
               foreach ($hourlyOccurrences as $timestamp)
               {
                  foreach ($occurrences as $occurStamp)
                  {

                     // when only BYMONTH was given, limit occurrence set
                     // to those hours that fall in the given months
                     //
                     if ((!empty($recurrenceRule['BYMONTH']) && empty($recurrenceRule['BYWEEKNO'])
                      && empty($recurrenceRule['BYYEARDAY']) && empty($recurrenceRule['BYMONTHDAY'])
                      && empty($recurrenceRule['BYDAY'])
                      && date('n', $timestamp) == date('n', $occurStamp))
   
                     // when only BYWEEKNO was given, limit occurrence set
                     // to the hours that come in the BYWEEKNO weeks
                     //
                      || (empty($recurrenceRule['BYMONTH']) && !empty($recurrenceRule['BYWEEKNO'])
                      && empty($recurrenceRule['BYYEARDAY']) && empty($recurrenceRule['BYMONTHDAY'])
                      && empty($recurrenceRule['BYDAY'])
                      && $timestamp >= $occurStamp && $timestamp < $occurStamp + SM_CAL_WEEK_SECONDS)
   
                     // when both BYMONTHLY *and* BYWEEKNO are given, limit occurrence set
                     // to the hours that come in the BYWEEKNO weeks only in the
                     // BYMONTHLY months.  whew.
                     //
                      || (!empty($recurrenceRule['BYMONTH']) && !empty($recurrenceRule['BYWEEKNO'])
                      && empty($recurrenceRule['BYYEARDAY']) && empty($recurrenceRule['BYMONTHDAY'])
                      && empty($recurrenceRule['BYDAY'])
                      && $timestamp >= $occurStamp && $timestamp < $occurStamp + SM_CAL_WEEK_SECONDS
                      && date('n', $timestamp) == date('n', $occurStamp))
   
                     // when BYYEARDAY or BYMONTHDAY or BYDAY were
                     // given - no matter what other fields are
                     // given, limit occurrence set to the 
                     // hours that fall in those year and month days
                     // already selected
                     //
                      || ((!empty($recurrenceRule['BYYEARDAY'])
                      || !empty($recurrenceRule['BYMONTHDAY'])
                      || !empty($recurrenceRule['BYDAY']))
                      && date('d', $timestamp) == date('d', $occurStamp)))
                     {
                        $temp_occurrences[] = $timestamp;
                        break;
                     }
                  }
               }
               $occurrences = $temp_occurrences;
            }



            // BYMINUTE
            //
            // Straight-forward, if inefficient...
            //
            $minuteOccurrences = array();
            if (!empty($recurrenceRule['BYMINUTE'])) for ($d = 1; $d <= $numberOfDaysInYear; $d++)
            {
               for ($h = 0; $h < 24; $h++)
                  foreach ($recurrenceRule['BYMINUTE'] as $mi)
                     $minuteOccurrences[] = mktime($h, $mi, $second, 1, $d, $year);
            }  
            $temp_occurrences = array();
            if (empty($occurrences)) $occurrences = $minuteOccurrences;
                   
            // unless one of the above was given too...
            //
            else if (!empty($minuteOccurrences))
            {
               foreach ($minuteOccurrences as $timestamp)
               {
                  foreach ($occurrences as $occurStamp)
                  {
            
                     // when only BYMONTH was given, limit occurrence set
                     // to those minutes that fall in the given months
                     //
                     if ((!empty($recurrenceRule['BYMONTH']) && empty($recurrenceRule['BYWEEKNO'])
                      && empty($recurrenceRule['BYYEARDAY']) && empty($recurrenceRule['BYMONTHDAY'])
                      && empty($recurrenceRule['BYDAY']) && empty($recurrenceRule['BYHOUR'])
                      && date('n', $timestamp) == date('n', $occurStamp))
                     
                     // when only BYWEEKNO was given, limit occurrence set
                     // to the minutes that come in the BYWEEKNO weeks
                     //
                      || (empty($recurrenceRule['BYMONTH']) && !empty($recurrenceRule['BYWEEKNO'])
                      && empty($recurrenceRule['BYYEARDAY']) && empty($recurrenceRule['BYMONTHDAY'])
                      && empty($recurrenceRule['BYDAY']) && empty($recurrenceRule['BYHOUR'])
                      && $timestamp >= $occurStamp && $timestamp < $occurStamp + SM_CAL_WEEK_SECONDS)
                      
                     // when both BYMONTHLY *and* BYWEEKNO are given, limit occurrence set
                     // to the minutes that come in the BYWEEKNO weeks only in the 
                     // BYMONTHLY months.  whew.
                     // 
                      || (!empty($recurrenceRule['BYMONTH']) && !empty($recurrenceRule['BYWEEKNO'])
                      && empty($recurrenceRule['BYYEARDAY']) && empty($recurrenceRule['BYMONTHDAY'])
                      && empty($recurrenceRule['BYDAY']) && empty($recurrenceRule['BYHOUR'])
                      && $timestamp >= $occurStamp && $timestamp < $occurStamp + SM_CAL_WEEK_SECONDS
                      && date('n', $timestamp) == date('n', $occurStamp))
                      
                     // when BYYEARDAY or BYMONTHDAY or BYDAY 
                     // were given - no matter what other 
                     // fields are given (except BYHOUR), limit 
                     // occurrence set to 
                     // the minutes that fall in those year and 
                     // month days already selected
                     // 
                      || ((!empty($recurrenceRule['BYYEARDAY'])
                      || !empty($recurrenceRule['BYMONTHDAY'])
                      || !empty($recurrenceRule['BYDAY']))
                      && empty($recurrenceRule['BYHOUR'])
                      && date('d', $timestamp) == date('d', $occurStamp))
   
                     // when BYHOUR was given, no matter what other
                     // fields are given, limit occurrence set to the
                     // minutes that fall in those hours already selected
                     //
                      || (!empty($recurrenceRule['BYHOUR'])
                      && date('H', $timestamp) == date('H', $occurStamp)))
                     {
                        $temp_occurrences[] = $timestamp;
                        break;
                     }  
                  }  
               }  
               $occurrences = $temp_occurrences;
            }  
            


            // BYSECOND
            //
            // Straight-forward... (but really inefficient)
            //
            $secondOccurrences = array();
            if (!empty($recurrenceRule['BYSECOND'])) for ($d = 1; $d <= $numberOfDaysInYear; $d++)
            {
               for ($h = 0; $h < 24; $h++)
                  for ($mi = 0; $mi < 60; $mi++)
                     foreach ($recurrenceRule['BYSECOND'] as $s)
                        $secondOccurrences[] = mktime($h, $mi, $s, 1, $d, $year);
            }
            $temp_occurrences = array();
            if (empty($occurrences)) $occurrences = $secondOccurrences;

            // unless one of the above was given too...
            //
            else if (!empty($secondOccurrences))
            {
               foreach ($secondOccurrences as $timestamp)
               {
                  foreach ($occurrences as $occurStamp)
                  {

                     // when only BYMONTH was given, limit occurrence set
                     // to those seconds that fall in the given months
                     //
                     if ((!empty($recurrenceRule['BYMONTH']) && empty($recurrenceRule['BYWEEKNO'])
                      && empty($recurrenceRule['BYYEARDAY']) && empty($recurrenceRule['BYMONTHDAY'])
                      && empty($recurrenceRule['BYDAY']) && empty($recurrenceRule['BYHOUR'])
                      && empty($recurrenceRule['BYMINUTE'])
                      && date('n', $timestamp) == date('n', $occurStamp))
   
                     // when only BYWEEKNO was given, limit occurrence set
                     // to the seconds that come in the BYWEEKNO weeks
                     //
                      || (empty($recurrenceRule['BYMONTH']) && !empty($recurrenceRule['BYWEEKNO'])
                      && empty($recurrenceRule['BYYEARDAY']) && empty($recurrenceRule['BYMONTHDAY'])
                      && empty($recurrenceRule['BYDAY']) && empty($recurrenceRule['BYHOUR'])
                      && empty($recurrenceRule['BYMINUTE'])
                      && $timestamp >= $occurStamp && $timestamp < $occurStamp + SM_CAL_WEEK_SECONDS)
   
                     // when both BYMONTHLY *and* BYWEEKNO are given, limit occurrence set
                     // to the seconds that come in the BYWEEKNO weeks only in the
                     // BYMONTHLY months.  whew.
                     //
                      || (!empty($recurrenceRule['BYMONTH']) && !empty($recurrenceRule['BYWEEKNO'])
                      && empty($recurrenceRule['BYYEARDAY']) && empty($recurrenceRule['BYMONTHDAY'])
                      && empty($recurrenceRule['BYDAY']) && empty($recurrenceRule['BYHOUR'])
                      && empty($recurrenceRule['BYMINUTE'])
                      && $timestamp >= $occurStamp && $timestamp < $occurStamp + SM_CAL_WEEK_SECONDS
                      && date('n', $timestamp) == date('n', $occurStamp))
   
                     // when BYYEARDAY or BYMONTHDAY or BYDAY 
                     // were given - no matter what other fields 
                     // beside BYHOUR and BYMINUTE are given, limit occurrence 
                     // set to the seconds that fall in those year and 
                     // month days already selected
                     //
                      || ((!empty($recurrenceRule['BYYEARDAY'])
                      || !empty($recurrenceRule['BYMONTHDAY'])
                      || !empty($recurrenceRule['BYDAY']))
                      && empty($recurrenceRule['BYHOUR'])
                      && empty($recurrenceRule['BYMINUTE'])
                      && date('d', $timestamp) == date('d', $occurStamp))
   
                     // when BYHOUR was given, no matter what other
                     // fields are given beside BYMINUTE, limit occurrence 
                     // set to the seconds that fall in those hours already 
                     // selected
                     //
                      || (!empty($recurrenceRule['BYHOUR'])
                      && empty($recurrenceRule['BYMINUTE'])
                      && date('H', $timestamp) == date('H', $occurStamp))

                     // when BYMINUTE was given, no matter what other
                     // fields are given, limit occurrence 
                     // set to the seconds that fall in those minutes already 
                     // selected
                     //
                      || (!empty($recurrenceRule['BYMINUTE'])
                      && date('i', $timestamp) == date('i', $occurStamp)))
                     {
                        $temp_occurrences[] = $timestamp;
                        break;
                     }
                  }
               }
               $occurrences = $temp_occurrences;
            }



            break;



         //---------------------------------------------
         //
         case SM_CAL_ICAL_EVENT_RECURRENCE_FREQ_MONTHLY:

            // BYMONTHDAY
            //
            // Straight-forward...
            //
            foreach ($recurrenceRule['BYMONTHDAY'] as $DAY)
               {
                  if ($DAY < 0) 
                     $DAY = $numberOfDaysInMonth + $DAY + 1;
                  $occurrences[] = mktime($hour, $minute, $second, $month, $DAY, $year);
               }



            // BYDAY
            //
            // pick all days in the month that are in BYDAY,
            // optionally limited by the count given in front of BYDAY values
            //
            $dayOccurrences = array();
            $dayOfMonthCount = array(0 => array(), 1 => array(), 2 => array(), 
                                     3 => array(), 4 => array(), 5 => array(), 
                                     6 => array());
            $byDayRule = array();
            // separate day count offset and actual day name
            //
            foreach ($recurrenceRule['BYDAY'] as $DAY)
            {
               if (!array_key_exists(strtoupper($DAY), $WEEKDAYS))
               {
                  $dayCount = substr($DAY, 0, strlen($DAY) - 2);
                  $dy = $WEEKDAYS[substr($DAY, strlen($DAY) - 2)];
               }
               else
               {
                  $dayCount = 0;
                  $dy = $WEEKDAYS[$DAY];
               }
//LEFT OFF HERE: make sure zero array indexes don't screw something up
               $byDayRule[$dy] = $dayCount;
            }
            // now deal with BYDAY rule itself
            //
            if (!empty($recurrenceRule['BYDAY'])) for ($d = 1; $d <= $numberOfDaysInMonth; $d++)
            {
               $timestamp = mktime($hour, $minute, $second, $month, $d, $year);
               $dyOfMo = date('w', $timestamp);
               $dayOfMonthCount[$dyOfMo][] = $timestamp;
               foreach ($byDayRule as $DAY => $dayCount)
               {
                  if ($DAY == $dyOfMo 
                   && ($dayCount === 0 || $dayCount == sizeof($dayOfMonthCount[$DAY])))
                     $dayOccurrences[] = $timestamp;
               }
            }
            // get negative offsets
            //
            foreach ($byDayRule as $DAY => $dayCount)
            {
               if ($dayCount < 0)
               {
                  $dayOccurrences[] 
                   = $dayOfMonthCount[$DAY][sizeof($dayOfMonthCount[$DAY]) + $dayCount];
               }
            }
            $temp_occurrences = array();
            if (empty($occurrences)) $occurrences = $dayOccurrences;

            // unless BYMONTHDAY was given too...
            //
            else if (!empty($dayOccurrences))
            {
               foreach ($dayOccurrences as $timestamp)
               {
                  foreach ($occurrences as $occurStamp)
                  {

                     // when BYMONTHDAY was given, limit
                     // occurrence set to the days that
                     // intersect those month days already
                     // selected
                     //
                     if ($timestamp == $occurStamp)
                     {
                        $temp_occurrences[] = $timestamp;
                        break;
                     }
                  }
               }
               $occurrences = $temp_occurrences;
            }


            
            // BYHOUR
            // 
            // Straight-forward...
            // 
            $hourlyOccurrences = array();
            if (!empty($recurrenceRule['BYHOUR'])) for ($d = 1; $d <= $numberOfDaysInMonth; $d++)
            {
               foreach ($recurrenceRule['BYHOUR'] as $h)
                  $hourlyOccurrences[] = mktime($h, $minute, $second, $month, $d, $year);
            }
            $temp_occurrences = array();
            if (empty($occurrences)) $occurrences = $hourlyOccurrences;

            // unless one of the above was given too...
            //
            else if (!empty($hourlyOccurrences))
            {
               foreach ($hourlyOccurrences as $timestamp)
               {
                  foreach ($occurrences as $occurStamp)
                  {

                     // when BYMONTHDAY or BYDAY were
                     // given, limit occurrence set to the 
                     // hours that fall in those days
                     // already selected
                     //
                     if (date('d', $timestamp) == date('d', $occurStamp))
                     {
                        $temp_occurrences[] = $timestamp;
                        break;
                     }
                  }
               }
               $occurrences = $temp_occurrences;
            }



            // BYMINUTE
            //
            // Straight-forward, if inefficient...
            //
            $minuteOccurrences = array();
            if (!empty($recurrenceRule['BYMINUTE'])) for ($d = 1; $d <= $numberOfDaysInMonth; $d++)
            {
               for ($h = 0; $h < 24; $h++)
                  foreach ($recurrenceRule['BYMINUTE'] as $mi)
                     $minuteOccurrences[] = mktime($h, $mi, $second, $month, $d, $year);
            }  
            $temp_occurrences = array();
            if (empty($occurrences)) $occurrences = $minuteOccurrences;
                   
            // unless one of the above was given too...
            //
            else if (!empty($minuteOccurrences))
            {
               foreach ($minuteOccurrences as $timestamp)
               {
                  foreach ($occurrences as $occurStamp)
                  {
            
                     // when BYMONTHDAY or BYDAY were
                     // given, limit occurrence set to the
                     // minutes that fall in those days
                     // already selected
                     // 
                     if (((!empty($recurrenceRule['BYMONTHDAY'])
                      || !empty($recurrenceRule['BYDAY']))
                      && empty($recurrenceRule['BYHOUR'])
                      && date('d', $timestamp) == date('d', $occurStamp))
   
                     // when BYHOUR was given, no matter what other
                     // fields are given, limit occurrence set to the
                     // minutes that fall in those hours already selected
                     //
                      || (!empty($recurrenceRule['BYHOUR'])
                      && date('H', $timestamp) == date('H', $occurStamp)))
                     {
                        $temp_occurrences[] = $timestamp;
                        break;
                     }  
                  }  
               }  
               $occurrences = $temp_occurrences;
            }  
            


            // BYSECOND
            //
            // Straight-forward... (but really inefficient)
            //
            $secondOccurrences = array();
            if (!empty($recurrenceRule['BYSECOND'])) for ($d = 1; $d <= $numberOfDaysInMonth; $d++)
            {
               for ($h = 0; $h < 24; $h++)
                  for ($mi = 0; $mi < 60; $mi++)
                     foreach ($recurrenceRule['BYSECOND'] as $s)
                        $secondOccurrences[] = mktime($h, $mi, $s, $month, $d, $year);
            }
            $temp_occurrences = array();
            if (empty($occurrences)) $occurrences = $secondOccurrences;

            // unless one of the above was given too...
            //
            else if (!empty($secondOccurrences))
            {
               foreach ($secondOccurrences as $timestamp)
               {
                  foreach ($occurrences as $occurStamp)
                  {

                     // when BYMONTHDAY or BYDAY were
                     // given, limit occurrence set to the
                     // minutes that fall in those days
                     // already selected
                     //
                     if (((!empty($recurrenceRule['BYMONTHDAY'])
                      || !empty($recurrenceRule['BYDAY']))
                      && empty($recurrenceRule['BYHOUR'])
                      && empty($recurrenceRule['BYMINUTE'])
                      && date('d', $timestamp) == date('d', $occurStamp))
   
                     // when BYHOUR was given, no matter what other
                     // fields are given except BYMINUTE, limit 
                     // occurrence set to the minutes that fall in 
                     // those hours already selected
                     //
                      || (!empty($recurrenceRule['BYHOUR'])
                      && empty($recurrenceRule['BYMINUTE'])
                      && date('H', $timestamp) == date('H', $occurStamp))
   
                     // when BYMINUTE was given, no matter what other
                     // fields are given, limit occurrence 
                     // set to the seconds that fall in those minutes already 
                     // selected
                     //
                      || (!empty($recurrenceRule['BYMINUTE'])
                      && date('i', $timestamp) == date('i', $occurStamp)))
                     {
                        $temp_occurrences[] = $timestamp;
                        break;
                     }
                  }
               }
               $occurrences = $temp_occurrences;
            }



            break;



         //---------------------------------------------
         //
         case SM_CAL_ICAL_EVENT_RECURRENCE_FREQ_WEEKLY:



            // BYDAY
            //
            // pick all days in the week that are in BYDAY,
            //
            for ($timestamp = $currentIntervalBasis; 
                 $timestamp < $currentIntervalBasis + SM_CAL_WEEK_SECONDS; 
                 $timestamp += SM_CAL_DAY_SECONDS)
            {
//LEFT OFF HERE: here is a possible efficiency improvement: this date() call 
//               and the whole loop look like they can be avoided when BYDAY is empty, yeah?
//               Are there any other similar improvements in the code elsewhere?
               $dyOfWk = date('w', $timestamp);

               foreach ($recurrenceRule['BYDAY'] as $DAY)
               {
                  if ($WEEKDAYS[$DAY] == $dyOfWk)
                     $occurrences[] = $timestamp;
               }

               // can't let the loop go past end of the week 
               // (which it will do on the first iteration if 
               // first occurrence isn't on WKST)
               // 
               if ($dyOfWk == $weekEnd)
                  break;

            }


            
            // BYHOUR
            // 
            // Straight-forward...
            // 
            $hourlyOccurrences = array();
            if (!empty($recurrenceRule['BYHOUR'])) for ($d = $day; $d < $day + 7; $d++)
            {
               foreach ($recurrenceRule['BYHOUR'] as $h)
                  $hourlyOccurrences[] = mktime($h, $minute, $second, $month, $d, $year);
            }
            $temp_occurrences = array();
            if (empty($occurrences)) $occurrences = $hourlyOccurrences;

            // unless BYDAY was also given...
            //
            else if (!empty($hourlyOccurrences))
            {
               foreach ($hourlyOccurrences as $timestamp)
               {
                  foreach ($occurrences as $occurStamp)
                  {

                     // when BYDAY was given, 
                     // limit occurrence set to the 
                     // hours that fall in those days
                     // already selected
                     //
                     if (date('d', $timestamp) == date('d', $occurStamp))
                     {
                        $temp_occurrences[] = $timestamp;
                        break;
                     }
                  }
               }
               $occurrences = $temp_occurrences;
            }



            // BYMINUTE
            //
            // Straight-forward, if inefficient...
            //
            $minuteOccurrences = array();
            if (!empty($recurrenceRule['BYMINUTE'])) for ($d = $day; $d < $day + 7; $d++)
            {
               for ($h = 0; $h < 24; $h++)
                  foreach ($recurrenceRule['BYMINUTE'] as $mi)
                     $minuteOccurrences[] = mktime($h, $mi, $second, $month, $d, $year);
            }  
            $temp_occurrences = array();
            if (empty($occurrences)) $occurrences = $minuteOccurrences;
                   
            // unless one of the above was given too...
            //
            else if (!empty($minuteOccurrences))
            {
               foreach ($minuteOccurrences as $timestamp)
               {
                  foreach ($occurrences as $occurStamp)
                  {
            
                     // when BYDAY was given,
                     // limit occurrence set to the
                     // minutes that fall in those days
                     // already selected
                     // 
                     if ((!empty($recurrenceRule['BYDAY'])
                      && empty($recurrenceRule['BYHOUR'])
                      && date('d', $timestamp) == date('d', $occurStamp))
   
                     // when BYHOUR was given, no matter what other
                     // fields are given, limit occurrence set to the
                     // minutes that fall in those hours already selected
                     //
                      || (!empty($recurrenceRule['BYHOUR'])
                      && date('H', $timestamp) == date('H', $occurStamp)))
                     {
                        $temp_occurrences[] = $timestamp;
                        break;
                     }  
                  }  
               }  
               $occurrences = $temp_occurrences;
            }  
            


            // BYSECOND
            //
            // Straight-forward... (but really inefficient)
            //
            $secondOccurrences = array();
            if (!empty($recurrenceRule['BYSECOND'])) for ($d = $day; $d < $day + 7; $d++)
            {
               for ($h = 0; $h < 24; $h++)
                  for ($mi = 0; $mi < 60; $mi++)
                     foreach ($recurrenceRule['BYSECOND'] as $s)
                        $secondOccurrences[] = mktime($h, $mi, $s, $month, $d, $year);
            }
            $temp_occurrences = array();
            if (empty($occurrences)) $occurrences = $secondOccurrences;

            // unless one of the above was given too...
            //
            else if (!empty($secondOccurrences))
            {
               foreach ($secondOccurrences as $timestamp)
               {
                  foreach ($occurrences as $occurStamp)
                  {

                     // when BYDAY was given,
                     // limit occurrence set to the
                     // minutes that fall in those days
                     // already selected
                     //
                     if ((!empty($recurrenceRule['BYDAY'])
                      && empty($recurrenceRule['BYHOUR'])
                      && empty($recurrenceRule['BYMINUTE'])
                      && date('d', $timestamp) == date('d', $occurStamp))
   
                     // when BYHOUR was given, no matter what other
                     // fields are given except BYMINUTE, limit 
                     // occurrence set to the minutes that fall in 
                     // those hours already selected
                     //
                      || (!empty($recurrenceRule['BYHOUR'])
                      && empty($recurrenceRule['BYMINUTE'])
                      && date('H', $timestamp) == date('H', $occurStamp))
   
                     // when BYMINUTE was given, no matter what other
                     // fields are given, limit occurrence 
                     // set to the seconds that fall in those minutes already 
                     // selected
                     //
                      || (!empty($recurrenceRule['BYMINUTE'])
                      && date('i', $timestamp) == date('i', $occurStamp)))
                     {
                        $temp_occurrences[] = $timestamp;
                        break;
                     }
                  }
               }
               $occurrences = $temp_occurrences;
            }



            break;



         //---------------------------------------------
         //
         case SM_CAL_ICAL_EVENT_RECURRENCE_FREQ_DAILY:



            // BYHOUR
            // 
            // Straight-forward...
            // 
            foreach ($recurrenceRule['BYHOUR'] as $h)
               $occurrences[] = mktime($h, $minute, $second, $month, $day, $year);



            // BYMINUTE
            //
            // Straight-forward, if inefficient...
            //
            $minuteOccurrences = array();
            if (!empty($recurrenceRule['BYMINUTE']))
            {
               for ($h = 0; $h < 24; $h++)
                  foreach ($recurrenceRule['BYMINUTE'] as $mi)
                     $minuteOccurrences[] = mktime($h, $mi, $second, $month, $day, $year);
            }  
            $temp_occurrences = array();
            if (empty($occurrences)) $occurrences = $minuteOccurrences;
                   
            // unless BYHOUR was given...
            //
            else if (!empty($minuteOccurrences))
            {
               foreach ($minuteOccurrences as $timestamp)
               {
                  foreach ($occurrences as $occurStamp)
                  {
            
                     // when BYHOUR was given, no matter what other
                     // fields are given, limit occurrence set to the
                     // minutes that fall in those hours already selected
                     //
                     if (date('H', $timestamp) == date('H', $occurStamp))
                     {
                        $temp_occurrences[] = $timestamp;
                        break;
                     }  
                  }  
               }  
               $occurrences = $temp_occurrences;
            }  
            


            // BYSECOND
            //
            // Straight-forward... (but really inefficient)
            //
            $secondOccurrences = array();
            if (!empty($recurrenceRule['BYSECOND']))
            {
               for ($h = 0; $h < 24; $h++)
                  for ($mi = 0; $mi < 60; $mi++)
                     foreach ($recurrenceRule['BYSECOND'] as $s)
                        $secondOccurrences[] = mktime($h, $mi, $s, $month, $day, $year);
            }
            $temp_occurrences = array();
            if (empty($occurrences)) $occurrences = $secondOccurrences;

            // unless one of the above was given too...
            //
            else if (!empty($secondOccurrences))
            {
               foreach ($secondOccurrences as $timestamp)
               {
                  foreach ($occurrences as $occurStamp)
                  {

                     // when BYHOUR was given, but not BYMINUTE,
                     // limit occurrence set to the minutes that 
                     // fall in those hours already selected
                     //
                     if ((!empty($recurrenceRule['BYHOUR'])
                      && empty($recurrenceRule['BYMINUTE'])
                      && date('H', $timestamp) == date('H', $occurStamp))
   
                     // when BYMINUTE was given, no matter what other
                     // fields are given, limit occurrence 
                     // set to the seconds that fall in those minutes already 
                     // selected
                     //
                      || (!empty($recurrenceRule['BYMINUTE'])
                      && date('i', $timestamp) == date('i', $occurStamp)))
                     {
                        $temp_occurrences[] = $timestamp;
                        break;
                     }
                  }
               }
               $occurrences = $temp_occurrences;
            }



            break;



         //---------------------------------------------
         //
         case SM_CAL_ICAL_EVENT_RECURRENCE_FREQ_HOURLY:



            // BYMINUTE
            //
            // Straight-forward, if inefficient...
            //
            foreach ($recurrenceRule['BYMINUTE'] as $mi)
               $occurrences[] = mktime($hour, $mi, $second, $month, $day, $year);



            // BYSECOND
            //
            // Straight-forward... (but really inefficient)
            //
            $secondOccurrences = array();
            if (!empty($recurrenceRule['BYSECOND']))
            {
               for ($mi = 0; $mi < 60; $mi++)
                  foreach ($recurrenceRule['BYSECOND'] as $s)
                     $secondOccurrences[] = mktime($hour, $mi, $s, $month, $day, $year);
            }
            $temp_occurrences = array();
            if (empty($occurrences)) $occurrences = $secondOccurrences;

            // unless BYMINUTE was given...
            //
            else if (!empty($secondOccurrences))
            {
               foreach ($secondOccurrences as $timestamp)
               {
                  foreach ($occurrences as $occurStamp)
                  {

                     // when BYMINUTE was given, 
                     // limit occurrence set to the 
                     // seconds that fall in those 
                     // minutes already selected
                     //
                     if (date('i', $timestamp) == date('i', $occurStamp))
                     {
                        $temp_occurrences[] = $timestamp;
                        break;
                     }
                  }
               }
               $occurrences = $temp_occurrences;
            }



            break;



         case SM_CAL_ICAL_EVENT_RECURRENCE_FREQ_MINUTELY:



            // BYSECOND
            //
            // Straight-forward... 
            //
            foreach ($recurrenceRule['BYSECOND'] as $s)
               $occurrences[] = mktime($hour, $minute, $s, $month, $day, $year);



            break;

      }



      // if we found zero occurrences, use the current interval basis itself
      //
      if (empty($occurrences)) $occurrences = array($currentIntervalBasis);



      // sort the occurrence dates
      //
      else sort($occurrences);



      // if BYSETPOS is specified, apply that one last rule here...
      //
      $newOccurrenceArray = array();
      foreach ($recurrenceRule['BYSETPOS'] as $setpos)
      {
         if ($setpos < 0)
            $setpos = sizeof($occurrences) + $setpos;
         $newOccurrenceArray[] = $occurrences[$setpos];
      }
      if (!empty($recurrenceRule['BYSETPOS']))
      {
         $occurrences = $newOccurrenceArray;
         sort($occurrences);
      }


      // now, run through our set of occurrences, comparing to:
      //   - the given current occurrence (if any)
      //   - the UNTIL date, if available
      //   - the maximum allowable recurrence date
      //
      $nextOccurrenceDate = 0;
      $maxPossibleOccurrenceDate = $occurrences[sizeof($occurrences) - 1];
      $pastEndDate = FALSE;
      foreach ($occurrences as $recur)
      {

         if (!empty($currentOccurrenceDate) && $recur <= $currentOccurrenceDate)
            continue;

         if (!empty($recurrenceRule['UNTIL']) && $recur > $recurrenceRule['UNTIL'])
         {
            $pastEndDate = TRUE;
            break;
         }

         if ($recur > $maxDate)
         {
            $pastEndDate = TRUE;
            break;
         }

         $nextOccurrenceDate = $recur;
         break;

      }



      // check if we have a date from the recurrence date set that comes first
      //
      $nextRDate = 0;
      if (sizeof($recurrenceDates))
      {

         // RFC says we can have periods here, but does not
         // explain what they mean in this context... does 
         // end of period override event end time for that
         // occurrence?  (grrr)  Or are we supposed to somehow
         // have multiple occurrences within the PERIOD???  (grrr)
         // until I find some better documentation, just using
         // the starting timestamp as another start date as
         // if this PERIOD was just a DATE/TIME stamp
         //
         if (is_array($recurrenceDates[0]))
           $nextRDate = $recurrenceDates[0][0];
         else 
           $nextRDate = $recurrenceDates[0];

      }


      // next recurrence same as next recurrence date?
      // remove from RDATE array but don't tell caller
      // that it is an RDATE
      //
      if ($nextOccurrenceDate > 0 && $nextRDate == $nextOccurrenceDate)
      {
         array_shift($recurrenceDates);
         return $nextOccurrenceDate;
      }

         
      // if we didn't find a next recurrence and exceeded
      // max date, use the next RDATE or return -1
      //
      if ($nextOccurrenceDate == 0 && $pastEndDate)
      {

         if ($nextRDate > 0 && $nextRDate <= $maxDate)
         {
            array_shift($recurrenceDates);
            return array($nextRDate);
         }

         return -1;

      }


      // if we didn't find a next recurrence, but haven't
      // exceeded max dates, check RDATEs or return 0
      //
      if ($nextOccurrenceDate == 0)
      {

         if ($nextRDate > 0 && $nextRDate <= $maxPossibleOccurrenceDate)
         {
            array_shift($recurrenceDates);
            return array($nextRDate);
         }

         return 0;

      }


      // otherwise, we *did* find a next recurrence, but look
      // to see if the next RDATE comes first
      //
      if ($nextRDate > 0 && $nextRDate < $nextOccurrenceDate)
      {
         array_shift($recurrenceDates);
         return array($nextRDate);
      }


      return $nextOccurrenceDate;

   }



   /**
     * Utility function for looping through recurring event occurrences
     *
     * NOTE: DO NOT return from this function without making sure 
     *       $this->cachedOccurrencesThruDate is OK.  Usually, at
     *       least when returning FALSE, this is sufficient:
     *       $this->cachedOccurrencesThruDate = $maxDate;
     *
     * @param string $callback A function that will be called
     *                         during each iteration; if this
     *                         callback function returns anything 
     *                         but FALSE, this function will cease 
     *                         and return that same value.  The
     *                         callback function should expect 
     *                         two parameters: the timestamp for 
     *                         the current event occurrence 
     *                         iteration and the $args array. 
     *                         Note that the callback function
     *                         must be a function in this class.
     * @param array $args An array of arguments to be passed to the
     *                    $callback function (optional)
     * @param timestamp $maxDate Date past which we don't 
     *                           need to iterate (optional; default
     *                           is the constant MAX_RECURRENCE_DATE,
     *                           as defined in {@link constants.php})
     *                           NOTE that to avoid time-of-day problems,
     *                           we typically iterate one day past this date
     * @param boolean $useExclusionRule If TRUE, the exclusion rule is 
     *                                  used instead of the regular
     *                                  recurrence rule. (optional;
     *                                  default=FALSE)
     *
     * @return mixed Any value returned from the $callback function, 
     *               or FALSE if nothing is returned after iterating
     *               all occurrences.
     *
     * @access private
     *
     */
   function iterateEventOccurrences($callback, $args=array(), $maxDate=MAX_RECURRENCE_DATE,
                                    $useExclusionRule=FALSE)
   {

      global $color, $week_start_day, $WEEKDAYS;


      // add a day to maxDate
      //
      $maxDate += SM_CAL_DAY_SECONDS;


      // which rule set do we use?
      //
      if (!$useExclusionRule)
      {
         $recurrenceRule = $this->getRecurrenceRule();
         $recurrenceDates = $this->getRecurrenceDates();

         if (empty($recurrenceRule) && empty($recurrenceDates))
         {
            plain_error_message('ERROR (iterateEventOccurrences): Recurring events must have recurrence rule or recurrence dates', $color);
            exit;
         }
      }
      else
      {
         $recurrenceRule = $this->getRecurrenceExclusionRule();
// NOTE: We check elsewhere for EXDATEs by just iterating 
// them, which is faster than including them here.  Make 
// sure this doesn't mess anything up if EXRULE/EXDATE is
// used from another context beside this one (see bottom
// of this function)
//         $recurrenceDates = $this->getRecurrenceExclusionDates();
         $recurrenceDates = array();

         if (empty($recurrenceRule) && empty($recurrenceDates))
            return FALSE;

      }



      // make sure we have what we need
      //
      $currentOccurrenceDate = $this->getStartDateTime();
      if (empty($currentOccurrenceDate))
      {
         plain_error_message('ERROR (iterateEventOccurrences): Recurring events must have start dates', $color);
         exit;
      }


      // have we already found occurrences up to the given
      // max date?  if so, use those so we don't have to 
      // reinvent the wheel - saves a LOT of cycles
      //
      // only use caching if not working with an exclusion rule
//TODO: should build an excluded date list cache too
      //
      if (!$useExclusionRule)
      {
         if (sizeof($this->cachedOccurrences) > 0 
          && ($this->cachedOccurrences[sizeof($this->cachedOccurrences) - 1] >= $maxDate
           || $this->cachedOccurrencesThruDate >= $maxDate))
         {
            $useCache = TRUE;
            $cacheCount = 0;
            $currentOccurrenceDate = $this->cachedOccurrences[0];
         }
         else  // build cache for next time
         {
            $useCache = FALSE;
            $this->cachedOccurrences = array();
            $this->cachedOccurrencesThruDate = 0;
         }
      }
      else 
      {
         $useCache = FALSE;
      }


      // ready?  go...
      //
      $currentIntervalBasis = $currentOccurrenceDate;
      $exclusionDates = $this->getRecurrenceExclusionDates();
      $nextOccurrenceDate = 0;
      $nextIntervalBasis = 0;
      $count = 1;
      $performCallback = TRUE;
      while (TRUE)
      {

         // go do callback 
         //
         if ($performCallback)
         {
            // build occurrence cache
            //
            if (!$useCache && !$useExclusionRule)
               $this->cachedOccurrences[] = $currentOccurrenceDate;

            $retValue = $this->$callback($currentOccurrenceDate, $args);

            if ($retValue !== FALSE)
               return $retValue;
         }
         $performCallback = TRUE;



         //
         // figure out next occurrence
         //



         // use cache if available
         //
         if ($useCache)
         {

            $cacheCount++;

            // no occurrences left in cache?  bail completely
            //
            if (!isset($this->cachedOccurrences[$cacheCount]))
               return FALSE;

            $currentOccurrenceDate = $this->cachedOccurrences[$cacheCount];
            continue;

         }


         // do we have more occurrences left in the current interval?
         //
         $foundInRDate = FALSE;
         $nextOccurrenceDate = $this->inspectIntraIntervalOccurrences($recurrenceRule, 
                               $recurrenceDates, $currentOccurrenceDate, 
                               $currentIntervalBasis, $maxDate);


         // surpassed allowable date range
         //
         if ($nextOccurrenceDate === -1)
         {
            $this->cachedOccurrencesThruDate = $maxDate;
            return FALSE;
         }


         
         // date found in $recurrenceDates?
         //
         else if (is_array($nextOccurrenceDate))
         {
            $foundInRDate = TRUE;
            $nextOccurrenceDate = $nextOccurrenceDate[0];
         }


         // need to increment our interval
         //
         else if ($nextOccurrenceDate === 0) while (!empty($recurrenceRule))
         {


            // increment by interval until we find a valid occurrence
            // 
            switch ($recurrenceRule['FREQ'])
            {

               case SM_CAL_ICAL_EVENT_RECURRENCE_FREQ_SECONDLY:
                  $nextIntervalBasis = $currentIntervalBasis + $recurrenceRule['INTERVAL'];
                  break;

               case SM_CAL_ICAL_EVENT_RECURRENCE_FREQ_MINUTELY:
                  $nextIntervalBasis = $currentIntervalBasis 
                                     + (60 * $recurrenceRule['INTERVAL']);
                  break;

               case SM_CAL_ICAL_EVENT_RECURRENCE_FREQ_HOURLY:
                  $nextIntervalBasis = $currentIntervalBasis 
                                     + (3600 * $recurrenceRule['INTERVAL']);
                  break;

               case SM_CAL_ICAL_EVENT_RECURRENCE_FREQ_DAILY:
                  $nextIntervalBasis = $currentIntervalBasis 
                                     + (SM_CAL_DAY_SECONDS * $recurrenceRule['INTERVAL']);
                  break;

               case SM_CAL_ICAL_EVENT_RECURRENCE_FREQ_WEEKLY:
// LEFT OFF HERE -- WKST: i think each interval jump needs to jump to week starting on WKST day
//                  But does this screw up a simply "once a week" event?
//                  Not sure if this is right (RFC is not clear), but we 
//                  move to first occurrence of WKST past today, then jump
//                  the requested number of weeks minus one
                  if (!isset($recurrenceRule['WKST'])) // don't check if empty, since zero is valid
                     $weekStart = $week_start_day;
                  else
                     $weekStart = $recurrenceRule['WKST'];
                  for ($x = $currentIntervalBasis + SM_CAL_DAY_SECONDS; 
                       $x <= $currentIntervalBasis + SM_CAL_WEEK_SECONDS; 
                       $x += SM_CAL_DAY_SECONDS)
                     if (date('w', $x) == $weekStart)
                     {
                        $nextIntervalBasis = $x;
                        break;
                     }
                  $nextIntervalBasis = $nextIntervalBasis 
                                     + (SM_CAL_WEEK_SECONDS * ($recurrenceRule['INTERVAL'] - 1));
// the old way - just jump one week (times interval):
//                  $nextIntervalBasis = $currentIntervalBasis 
//                                     + (SM_CAL_WEEK_SECONDS * $recurrenceRule['INTERVAL']);
                  break;

               case SM_CAL_ICAL_EVENT_RECURRENCE_FREQ_MONTHLY:

                  // incrementing by month is fuzzy; make sure we don't go past 
                  // last day of month when incrementing
                  //
                  list($mon, $dy, $yr, $hr, $min, $sec) 
                   = explode('-', date('n-j-Y-G-i-s', $currentIntervalBasis));
                  $mon += $recurrenceRule['INTERVAL'];
                  if ($mon > 12)
                  {
                     $overage = $mon - 12;
                     $yr += floor($overage / 12) + 1;
                     $mon = $overage - (floor($overage / 12) * 12);
                  }
                  $nextIntervalBasis = mktime(12, 0, 0, $mon, 15, $yr);
                  $lastDayOfMonth = date('t', $nextIntervalBasis);
                  if ($lastDayOfMonth < $dy) $dy = $lastDayOfMonth;
                  $nextIntervalBasis = mktime($hr, $min, $sec, $mon, $dy, $yr);
                  break;

               case SM_CAL_ICAL_EVENT_RECURRENCE_FREQ_YEARLY:

                  // incrementing by year is also fuzzy...
                  //
                  list($mon, $dy, $yr, $hr, $min, $sec) 
                   = explode('-', date('n-j-Y-G-i-s', $currentIntervalBasis));
                  $yr += $recurrenceRule['INTERVAL'];
                  $nextIntervalBasis = mktime(12, 0, 0, $mon, 15, $yr);
                  $lastDayOfMonth = date('t', $nextIntervalBasis);
                  if ($lastDayOfMonth < $dy) $dy = $lastDayOfMonth;
                  $nextIntervalBasis = mktime($hr, $min, $sec, $mon, $dy, $yr);
                  break;

               default:
                  plain_error_message('ERROR (iterateEventOccurrences): Bad recurrence frequency: ' . $recurrenceRule['FREQ'], $color);
                  exit;

            }


            // have we gone past allowable dates?
            // stop trying to find another interval
            //
            // note that the basis date for an interval
            // can be as much as a year past the actual
            // first occurrence in that interval, thus
            // the extended time check (54 weeks just to
            // be safe)
            //
            if ((isset($recurrenceRule['UNTIL']) && !empty($recurrenceRule['UNTIL']) 
             && ($nextIntervalBasis - (SM_CAL_WEEK_SECONDS * 54)) > $recurrenceRule['UNTIL'])
             || ($maxDate < ($nextIntervalBasis - (SM_CAL_WEEK_SECONDS * 54))))
            {
               $nextIntervalBasis = $currentIntervalBasis;
               break;
            }


            // now we have to check recurrence rule to make sure
            // this interval is allowable (the lack of break 
            // statements in this switch block is intentional and
            // crucial!)
            //
            switch ($recurrenceRule['FREQ'])
            {

               case SM_CAL_ICAL_EVENT_RECURRENCE_FREQ_SECONDLY:
                  if (!empty($recurrenceRule['BYSECOND'])
                   && !in_array(date('s', $nextIntervalBasis), $recurrenceRule['BYSECOND']))
                  {
                     $currentIntervalBasis = $nextIntervalBasis;
                     continue 2;
                  }

               case SM_CAL_ICAL_EVENT_RECURRENCE_FREQ_MINUTELY:

                  if (!empty($recurrenceRule['BYMINUTE'])
                   && !in_array(date('i', $nextIntervalBasis), $recurrenceRule['BYMINUTE']))
                  {
                     $currentIntervalBasis = $nextIntervalBasis;
                     continue 2;
                  }

               case SM_CAL_ICAL_EVENT_RECURRENCE_FREQ_HOURLY:

                  if (!empty($recurrenceRule['BYHOUR'])
                   && !in_array(date('H', $nextIntervalBasis), $recurrenceRule['BYHOUR']))
                  {
                     $currentIntervalBasis = $nextIntervalBasis;
                     continue 2;
                  }

               case SM_CAL_ICAL_EVENT_RECURRENCE_FREQ_DAILY:

//Note -- for yearly and monthly, this can have a leading number on it that shouldn't be 
//        there now, since we haven't gotten to yearly/monthly yet
                  if (!empty($recurrenceRule['BYDAY'])
                   && !in_array(date('w', $nextIntervalBasis), $recurrenceRule['BYDAY']))
                  {
                     $currentIntervalBasis = $nextIntervalBasis;
                     continue 2;
                  }

                  if (!empty($recurrenceRule['BYMONTHDAY']))
                  {

                     // checks positive and negative values of BYMONTHDAY
                     //
                     $OK = FALSE;
                     list($dayOfMonth, $daysInMon) 
                      = explode('-', date('j-t', $nextIntervalBasis));
                     foreach ($recurrenceRule['BYMONTHDAY'] as $DAY)
                     {

                        if ($DAY < 0)
                           $DAY = $daysInMon + $DAY + 1;

                        if ($DAY == $dayOfMonth)
                           $OK = TRUE;

                     }

                     // didn't find a match - can't use this date
                     //
                     if (!$OK)
                     {
                        $currentIntervalBasis = $nextIntervalBasis;
                        continue 2;
                     }

                  }

                  if (!empty($recurrenceRule['BYYEARDAY']))
                  {
                     
                     // checks positive and negative values of BYYEARDAY
                     //
                     $OK = FALSE;
                     list($dayOfYear, $yr) 
                      = explode('-', date('z-Y', $nextIntervalBasis));
                     $daysInYr = date('z', mktime(12, 0, 0, 12, 31, $yr)); 

                     // we need 1-based numbers, not zero-based numbers:
                     $dayOfYear++;
                     $daysInYr++;

                     foreach ($recurrenceRule['BYYEARDAY'] as $DAY)
                     {

                        if ($DAY < 0)
                           $DAY = $daysInYr + $DAY + 1;

                        if ($DAY == $dayOfYear)
                           $OK = TRUE;

                     }

                     // didn't find a match - can't use this date
                     // 
                     if (!$OK)
                     {
                        $currentIntervalBasis = $nextIntervalBasis;
                        continue 2;
                     }

                  }

//TODO: not entirely clear if these two cases should go above BYYEARDAY
               case SM_CAL_ICAL_EVENT_RECURRENCE_FREQ_WEEKLY:
               case SM_CAL_ICAL_EVENT_RECURRENCE_FREQ_MONTHLY:

//Note: BYWEEKNO is only applicable to YEARLY frequencies (per RFC), so we skip it here

                  if (!empty($recurrenceRule['BYMONTH'])
                   && !in_array(date('m', $nextIntervalBasis), $recurrenceRule['BYMONTH']))
                  {
                     $currentIntervalBasis = $nextIntervalBasis;
                     continue 2;
                  }

            }


            // if we got this far, we successfully found a 
            // new interval; look for an actual occurrence
            // in the interval
            //
            $currentIntervalBasis = $nextIntervalBasis;
            $nextOccurrenceDate = $this->inspectIntraIntervalOccurrences($recurrenceRule, 
                                  $recurrenceDates, '', 
                                  $currentIntervalBasis, $maxDate);

            // found a occurrence from RDATE; exit interval loop
            //
            if (is_array($nextOccurrenceDate))
            {
               $foundInRDate = TRUE;
               $nextOccurrenceDate = $nextOccurrenceDate[0];
               break;
            }

            // surpassed allowable date range exit completely
            //
            else if ($nextOccurrenceDate === -1)
            {
               $this->cachedOccurrencesThruDate = $maxDate;
               return FALSE;
            }

            // no occurrences found in this interval; get next interval
            //
            else if ($nextOccurrenceDate === 0)
               continue;

            // found a regular occurrence; exit interval loop
            //
            else
               break;


         } // end finding new interval



         // didn't find a new interval... all we have left is
         // to check remaining RDATES
         //
         if ($nextOccurrenceDate === 0)
         {

            if ($currentIntervalBasis == $nextIntervalBasis)
            {

               $rdate = array_shift($recurrenceDates);


               // if we couldn't find any more dates at all,
               // it's time to just give up and return
               //
               if (is_null($rdate))
               {
                  $this->cachedOccurrencesThruDate = $maxDate;
                  return FALSE;
               }


               $foundInRDate = TRUE;


               // PERIODs will be in the form of array(start/end)
               //
               if (is_array($rdate))
               {

                  // RFC says we can have periods here, but does not
                  // explain what they mean in this context... does 
                  // end of period override event end time for that
                  // occurrence?  (grrr)  Or are we supposed to somehow
                  // have multiple occurrences within the PERIOD???  (grrr)
                  // until I find some better documentation, just using
                  // the starting timestamp as another start date as
                  // if this PERIOD was just a DATE/TIME stamp
                  //
                  $nextOccurrenceDate = $rdate[0];

               }

               // regular timestamp
               //
               else
                  $nextOccurrenceDate = $rdate;

            }
            else
            {
               // should not be possible to get here, error if so
               // 
               plain_error_message('ERROR (iterateEventOccurrences): Unknown error', $color);
               exit;
            }
         }


         $currentOccurrenceDate = $nextOccurrenceDate;


         // check if the new occurrence is on the exclusion list
         //
         if (!$useExclusionRule)
         {

            // check EXDATEs
            //
            foreach ($exclusionDates as $exdate)
            {

               // if EXDATE type is just DATE (with no time), augment
               // with time info from DTSTART (original starting time)
               //
               if ($this->recurrenceExclusionDates->getType == SM_CAL_ICAL_PROPERTY_TYPE_DATE)
               {
                  list($h, $m, $s) = $this->startTime();
                  list($mo, $d, $y) = explode('-', date('n-j-Y', $exdate));
                  $exdate = mktime($h, $m, $s, $mo, $d, $y);
               }

               if ($exdate == $currentOccurrenceDate)
               {
                  $performCallback = FALSE;
                  break;
               }

            }
           
            // check EXRULE (only if EXDATE didn't already produce an exclusion)
            //
            if ($performCallback && $this->startsOnTimestamp($currentOccurrenceDate, TRUE))
               $performCallback = FALSE;

         }


         // no exclusion found and occurrence comes from RRULE (not RDATE)?
         // increment event counter
         //
         if ($performCallback && !$foundInRDate) $count++;


         // check that we haven't exceeded any limits
         //
         if (isset($recurrenceRule['COUNT']) && !empty($recurrenceRule['COUNT']) 
          && $count > $recurrenceRule['COUNT'])
         {
            $this->cachedOccurrencesThruDate = $maxDate;
            return FALSE;
         }
         if ($maxDate < $currentOccurrenceDate)
         {
            $this->cachedOccurrencesThruDate = $maxDate;
            return FALSE;
         }


      } // end main (endless) loop

   }



   /**
     * Dummy callback for use when forcing an event occurrence cache.
     *
     * Also constructs date-based cache information.
     *
     * This function is intended as a callback function for iterateEventOccurrences()
     *
     * @param timestamp $timestamp The date of the occurrence
     *
     * @return boolean FALSE
     *
     * @access private
     *
     */
   function forceEventOccurrenceCacheCallback($timestamp)
   {

      list($day, $month, $year) = explode(',', date('d,m,Y', $timestamp));
      $this->startDateCache[$year][intval($month)][intval($day)] = array(
         'timestamp' => $timestamp,
      );

      $endTime = $timestamp + $this->getEndDateTime() - $this->getStartDateTime();
      list($day, $month, $year) = explode(',', date('d,m,Y', $endTime));
      $this->endDateCache[$year][intval($month)][intval($day)] = array(
         'timestamp' => $endTime,
      );

      return FALSE;

   }



   /**
     * Determines if an event occurrence is on the given day.
     *
     * This function is intended as a callback function for iterateEventOccurrences()
     *
     * @param timestamp $timestamp The date of the occurrence
     * @param array $date The day being checked, this array contains four
     *                    integer entries: year, month and day as well as
     *                    the timestamp generated from the same values.
     *
     * @return boolean TRUE if this event occurrence is on the given day, FALSE otherwise.
     *
     * @access private
     *
     */
   function checkOccurrence($timestamp, $date)
   {

      return dayIsBetween($date[0], $date[1], $date[2],
                       $timestamp, 
//NOTE: we cheat by reaching into the startDateTime Property object directly,
//      which violates object integrity, but it gains a few tenths of a second 
//      (is that worth it?)
                       $timestamp + $this->getEndDateTime() - $this->startDateTime->value, 
// This is what it *should* be:
//                       $timestamp + $this->getEndDateTime() - $this->getStartDateTime(), 
                       $date[3]);

   }



   /**
     * Determines if an event occurrence is on the given timestamp.
     *
     * This function is intended as a callback function for iterateEventOccurrences()
     *
     * @param timestamp $timestamp The date of the occurrence
     * @param array $timestampToCheck The timestamp being checked; a one-element array
     *                                containing the timestamp
     *
     * @return boolean TRUE if this event occurrence is on the given timestamp, FALSE otherwise.
     *
     * @access private
     *
     */
   function checkOccurrenceForTimestamp($timestamp, $timestampToCheck)
   {

      if ($timestamp <= $timestampToCheck[0]
       && $timestampToCheck[0] <= $timestamp + $this->getEndDateTime() - $this->getStartDateTime())
         return TRUE;

      return FALSE;

   }



   /**
     * Determines if an event occurrence starts on the given timestamp.
     *
     * This function is intended as a callback function for iterateEventOccurrences()
     *
     * @param timestamp $timestamp The date of the occurrence
     * @param array $timestampToCheck The timestamp being checked; a one-element array
     *                                containing the timestamp
     *
     * @return boolean TRUE if this event occurrence starts on the 
     *                 given timestamp, FALSE otherwise.
     *
     * @access private
     *
     */
   function checkStartsOnTimestamp($timestamp, $timestampToCheck)
   {

      if ($timestamp == $timestampToCheck[0])
         return TRUE;

      return FALSE;

   }



   /**
     * Determines if an event occurrence starts on the given day.
     *
     * This function is intended as a callback function for iterateEventOccurrences()
     *
     * @param timestamp $timestamp The date of the occurrence
     * @param array $date The day being checked, this array contains three
     *                    integer entries: year, month and day plus an 
     *                    optional fourth entry that is the value of getdate()
     *                    for this date (provides speed boost).
     *
     * @return boolean TRUE if this event occurrence starts on the given day, FALSE otherwise.
     *
     * @access private
     *
     */
   function checkStartsOnDay($timestamp, $date)
   {

      $startInfo = getdate($timestamp);

      if (isset($date[3]) && !empty($date[3]))
         $dayInfo = $date[3];
      else
         $dayInfo = getdate(mktime(0, 0, 0, $date[1], $date[2], $date[0]));

      if ($startInfo['year'] == $date[0] && $startInfo['yday'] == $dayInfo['yday'])
         return TRUE;

      return FALSE;

   }



   /**
     * Determines if an event occurrence ends on the given day.
     *
     * This function is intended as a callback function for iterateEventOccurrences()
     *
     * @param timestamp $timestamp The date of the occurrence
     * @param array $date The day being checked, this array contains three
     *                    integer entries: year, month and day plus an 
     *                    optional fourth entry that is the value of getdate()
     *                    for this date (provides speed boost).
     *
     * @return boolean TRUE if this event occurrence ends on the given day, FALSE otherwise.
     *
     * @access private
     *
     */
   function checkEndsOnDay($timestamp, $date)
   {

      $endInfo = getdate($timestamp + $this->getEndDateTime() - $this->getStartDateTime());

      if (isset($date[3]) && !empty($date[3]))
         $dayInfo = $date[3];
      else
         $dayInfo = getdate(mktime(0, 0, 0, $date[1], $date[2], $date[0]));

      if ($endInfo['year'] == $date[0] && $endInfo['yday'] == $dayInfo['yday'])
         return TRUE;

      return FALSE;

   }



   /**
     * Finds the starting date for a given event occurrence
     *
     * This function is intended as a callback function for iterateEventOccurrences()
     *
     * @param timestamp $timestamp The date of the occurrence
     * @param array $date The day being checked, this array contains three
     *                    integer entries: year, month and day.
     *
     * @return mixed The timestamp of the event occurrence that matches
     *               the given day, FALSE otherwise.
     *
     * @access private
     *
     */
   function findOccurrenceStartDate($timestamp, $date)
   {

      if (dayIsBetween($date[0], $date[1], $date[2],
//NOTE: we cheat by reaching into the startDateTime Property object directly,
//      which violates object integrity, but it gains a few tenths of a second 
//      (is that worth it?)
                       $timestamp, $timestamp + $this->getEndDateTime() - $this->startDateTime->value))
// This is what it *should* be:
//                       $timestamp, $timestamp + $this->getEndDateTime() - $this->getStartDateTime()))
         return $timestamp;

      return FALSE;

   }



// ---------- PUBLIC ----------



   /**
     * Cache event occurrences
     *
     * It is highly recommended to use this function when first
     * instantiating a recurring event in order to cut down on 
     * having to run the recurrence engine, for example, for every 
     * day in a month.  In such a scenario, the best thing to do 
     * is call this method with a date a few days into the 
     * following month.
     *
     * Also supported is session-based cross-request caching,
     * which is also highly recommended.  Event times are stored
     * between HTTP requests, so the recursion engine only needs
     * to be run when an event is changed or when the requested
     * date is outside of the current cache (or when starting day
     * of week is changed).
     *
     * @param int $maxCacheDate The timestamp through which to 
     *                          force an occurrence cache
     * @param boolean $cacheInSession When TRUE, event times are
     *                                also stored in session.
     *                                (optional; default = TRUE)
     *
     * @access public
     *
     */
   function forceEventOccurrenceCache($maxCacheDate, $cacheInSession=TRUE)
   {

      // see if events are already stored in session if allowed
      //
      if ($cacheInSession)
      {
         global $calendarCache, $week_start_day;
         sqgetGlobalVar('calendarCache', $calendarCache, SQ_SESSION);
         if (isset($calendarCache[$this->getID()]) && !empty($calendarCache[$this->getID()]))
         {

            // cache is only valid if event has not been modified since it 
            // was stored in session and if the starting day of the week has
            // not changed
            //
            $eventLastModified = $calendarCache[$this->getID()]['lastUpdatedOn'];
            $lastWeekStartDay = $calendarCache[$this->getID()]['week_start_day'];
            if ($eventLastModified == $this->lastUpdatedOn()
             && $lastWeekStartDay == $week_start_day)
            {
               $this->startDateCache 
                = $calendarCache[$this->getID()]['startDateCache'];
               $this->endDateCache 
                = $calendarCache[$this->getID()]['endDateCache'];
               $this->cachedOccurrences 
                = $calendarCache[$this->getID()]['cachedOccurrences'];
               $this->cachedOccurrencesThruDate 
                = $calendarCache[$this->getID()]['cachedOccurrencesThruDate'];
            }
         }
      }


      // go figure out the cache dates
      //
      $this->iterateEventOccurrences('forceEventOccurrenceCacheCallback', 
                                     array(), $maxCacheDate);


      // if we need to store events in session, do so here
      // (overwrite anything that was there, because when
      // iterating above, it is entirely possible for the
      // cache to have had to been rebuilt due to different
      // max date)
      //
      if ($cacheInSession)
      {
         
         $calendarCache[$this->getID()]['startDateCache']
          = $this->startDateCache;
         $calendarCache[$this->getID()]['endDateCache']
          = $this->endDateCache;
         $calendarCache[$this->getID()]['cachedOccurrences']
          = $this->cachedOccurrences;
         $calendarCache[$this->getID()]['cachedOccurrencesThruDate']
          = $this->cachedOccurrencesThruDate;
         $calendarCache[$this->getID()]['lastUpdatedOn']
          = $this->lastUpdatedOn();
         $calendarCache[$this->getID()]['week_start_day']
          = $week_start_day;

         sqsession_register($calendarCache, 'calendarCache');

      }
//for testing - remove cache from session:
//sqsession_unregister('calendarCache');

   }



   /**
     * Reset Permissions To Match Parent Calendar
     *
     * Wipes any permissions for this event and resets
     * them to match whatever is in the parent calendar,
     * including copying the creator (of the last parent
     * calendar found).
     *
     * @param array $parents An array of the parent calendars,
     *                       used to avoid the need to look
     *                       up the parent calendars in the 
     *                       backend (which, if the event's
     *                       parent calendar(s) is(have) not
     *                       saved, will cause backend errors).
     *                       (optional, default empty (parents
     *                       are looked up in backend))
     *
     * @access public
     *
     */
   function resetPermissionsToParent($parents=array())
   {

      // loop thru all parent calendars, copy their owners,
      // readable and writeable users into this event
      //
      $o = array();
      $w = array();
      $r = array();
      $c = '';
      foreach ($this->getParentCalendars() as $parentID)
      {

         // check for parent in given array first
         //
         $lookup = TRUE;
         foreach ($parents as $parent)
            if ($parent->getID() == $parentID)
            {
               $lookup = FALSE;
               break;
            }

         if ($lookup)
            $parent = get_calendar($parentID, TRUE);


         // if there was no parent calendar found, the event and its
         // parent are not yet in the backend, so we'll just skip this
         // for now
         //
         // not ideal, but we hope the event will get its permissions
         // reset later... 
         //
         if ($parent === FALSE) continue;


         $o = array_unique(array_merge($o, $parent->getOwners()));
         $w = array_unique(array_merge($w, $parent->getWriteableUsers()));
         $r = array_unique(array_merge($r, $parent->getReadableUsers()));
         $c = $parent->createdBy();
      }
      $this->owners->setValue($o);
      $this->writeable_users->setValue($w);
      $this->readable_users->setValue($r);
      $this->setCreator($c);

   }



   /**
     * Get Event ID
     *
     * @return string This event's internal ID
     *
     * @access public
     *
     */
   function getID()
   {
      return $this->id->getValue();
   }



   /**
     * Set Event ID
     *
     * @param string $id The new internal ID to be assigned to this event
     *
     * @access public
     *
     */
   function setID($id)
   {
      $this->id->setValue($id);
   }



   /**
     * Get Sequence Number
     *
     * @return int This event's sequence number
     *
     * @access public
     *
     */
   function getSequence()
   {
      return $this->sequence->getValue();
   }



   /**
     * Increment Sequence Number
     *
     * @access public
     *
     */
   function incrementSequence()
   {
      $this->sequence->setValue($this->sequence->getValue() + 1);
   }



   /**
     * Get Event Title
     *
     * Returns event summary if available, otherwise a truncated
     * version of the description is returned, and if the description
     * is also empty, this event's UID is returned
     *
     * @return string This event's title
     *
     * @access public
     *
     */
   function getTitle()
   {
      $ret = $this->getSummary();
      if (empty($ret))
      {
         $ret = $this->getDescription();
         $ret = preg_replace("/(\r\n)|(\n)|(\r)/", '', $ret);
         $ret = substr($ret, 0, 10) . _("...");
         if (empty($ret))
            $ret = $this->getID();
      }
      return $ret;
   }



   /**
     * Get Event Summary
     *
     * @return string This event's summary
     *
     * @access public
     *
     */
   function getSummary()
   {
      return $this->summary->getValue();
   }



   /**
     * Get Event Description
     *
     * @return string This event's description
     *
     * @access public
     *
     */
   function getDescription()
   {
      return $this->description->getValue();
   }



   /**
     * Get Event Comments
     *
     * @return string This event's comments
     *
     * @access public
     *
     */
   function getComments()
   {
      return $this->comments->getValue();
   }



   /**
     * Get Event Domain
     *
     * @return string This event's domain
     *
     * @access public
     *
     */
   function getDomain()
   {
      return $this->dom->getValue();
   }



   /**
     * Get Event Status
     *
     * @return string This event's status, which should
     *                correspond to the event or todo
     *                status constants defined in {@link constants.php}
     *
     * @access public
     *
     */
   function getStatus()
   {
      return $this->status->getValue();
   }



   /**
     * Get Event Type
     *
     * @return string This event's type, which will
     *                correspond to the event type
     *                constants defined in {@link constants.php}
     *
     * @access public
     *
     */
   function getEventType()
   {
      return $this->type->getValue();
   }



   /**
     * Get Internal Start Date/Time
     *
     * @return int The timestamp representing this event's start date/time
     *
     * @access public
     *
     */
   function getStartDateTime()
   {

      return $this->startDateTime->getValue();

   }



   /** 
     * Determines if this event is an all-day/anniversary event
     *
     * @return boolean TRUE if this event is an all-day event, FALSE otherwise
     *
     * @access public
     *
     */
   function isAllDay()
   {
      return ($this->startDateTime->getType() == SM_CAL_ICAL_PROPERTY_TYPE_DATE);
   }



   /** 
     * Determines if this event is transparent for busy time searches
     *
     * @return boolean TRUE if this event is transparent, FALSE otherwise (opaque)
     *
     * @access public
     *
     */
   function isTransparent()
   {

      // all-day/anniversary type events are always transparent
      //
      if ($this->isAllDay()) return TRUE;


      // TODO: we don't yet support the TRANSP item for VEVENTS, but
      // from here on, we just return its value if it exists for this
      // event, and otherwise return false


      return FALSE;

   }



   /**
     * Get Internal End Date/Time
     *
     * @return int The timestamp representing this event's end date/time
     *
     * @access public
     *
     */
   function getEndDateTime()
   {

//      if ($this->cachedEndDateTime !== '') return $this->cachedEndDateTime;
      if (!empty($this->cachedEndDateTime)) return $this->cachedEndDateTime;
//LEFT OFF HERE: is it possible that any of the values used herein
//               to determine the return value will change in the
//               course of a page request and the cached value will
//               become invalid?  do we need to clear the cached
//               value wherever the dependencies herein get changed?


      // duration trumps all
      //
      $dur = $this->getDuration();
      if (!empty($dur))
      {
         $this->cachedEndDateTime = $dur + $this->getStartDateTime();
         return $this->cachedEndDateTime;
      }


      // Tasks/Todos have "due date" instead of end date
      //
      if ($this->isTask())
      {
         $this->cachedEndDateTime = $this->getDueDateTime();
         return $this->cachedEndDateTime;
      }


      $endValue = $this->endDateTime->getValue();


      // when start date is just a DATE, this is an 
      // "anniversary" (all-day) type event; end 
      // date/time will always be the end of a day
      // 
      if ($this->isAllDay())
      {

         // if no end date is available, it is the same day as the start
         //
         if (empty($endValue))
         {
            $endValue = $this->getStartDateTime();
         }

         // otherwise, we have to subtract a day from the end, since
         // end time is always non-inclusive
         //
         else
         {
            $endValue -= SM_CAL_DAY_SECONDS;
         }

         // set end time to last minute/second of the day
         //
         list($y, $m, $d) = explode('-', date('Y-n-j', $endValue));
         $endValue = mktime(23, 59, 59, $m, $d, $y);

      }


      // if no end date is given for regular (non-anniversary/all-day)
      // events, it gets set to the same time as the event start
      //
      else if (empty($endValue))
         $endValue = $this->startDateTime->getValue();


      $this->cachedEndDateTime = $endValue;
      return $endValue;

   }



   /**
     * Get Due Date
     *
     * @return int The timestamp representing this todo/taks's due date
     *
     * @access public
     *
     */
   function getDueDateTime()
   {
      return $this->due->getValue();
   }



   /**
     * Get Duration
     *
     * @return int The number of seconds corresponding to this event's duration
     *
     * @access public
     *
     */
   function getDuration()
   {
      return $this->duration->getValue();
   }



   /**
     * Get Recurrence Rule
     *
     * @return array The (parsed) recurrence rule for this event
     *
     * @access public
     *
     */
   function getRecurrenceRule()
   {
      return $this->recurrenceRule->getValue();
   }



   /**
     * Get Recurrence Days, if available
     *
     * @return mixed An array of the days of the week this event
     *               is set to occur on (each array element is a
     *               weekday constant as defined in {@link constants.php}.
     *               If none are set for this event, FALSE is returned.
     *
     * @access public
     *
     */
   function getRecurrenceDays()
   {

      $rrule = $this->getRecurrenceRule();

      if (!empty($rrule) && isset($rrule['BYDAY']) && !empty($rrule['BYDAY']))
      {
         global $WEEKDAYS;
         $returnArray = array();
//TODO: is this just a waste of CPU cycles?  we could just return the
//      array of day abbreviations (strings such as TU, WE, TH, etc)
//      and let the caller convert to day constants only if needed...??
         foreach ($rrule['BYDAY'] as $day)
            $returnArray[] = $WEEKDAYS[$day];
         return $returnArray;
      }

      return FALSE;

   }



   /**
     * Get Recurrence Interval, if available
     *
     * @return mixed The recurrence interval (integer) if
     *               recurrence rule is available and has
     *               INTERVAL clause; FALSE otherwise
     *
     * @access public
     *
     */
   function getRecurrenceInterval()
   {

      $rrule = $this->getRecurrenceRule();

      if (!empty($rrule) && isset($rrule['INTERVAL']) && !empty($rrule['INTERVAL']))
         return $rrule['INTERVAL'];

      return FALSE;

   }



   /**
     * Get Recurrence "Type"
     *
     * If this is a recurring event, give back the GUI-based 
     * recurrence type that corresponds to the RRULE's FREQ
     * clause, or return FALSE if this is a non-recurring event.
     *
     * @return mixed The GUI-based recurrence type, as defined
     *               by the event recurrence type constants 
     *               defined in {@link constants.php}, or FALSE
     *               if this is a one-time event
     *
     * @access public
     *
     */
   function getRecurrenceType()
   {

      global $color;

      if ($this->isOneTime())
         return FALSE;

      $rrule = $this->getRecurrenceRule();

      switch ($rrule['FREQ'])
      {

//TODO: some day, support hourly/minutely/secondly events
         case SM_CAL_ICAL_EVENT_RECURRENCE_FREQ_SECONDLY:
            plain_error_message('ERROR: The editing of events recurring SECONDLY not currently supported', $color);
            exit;

         case SM_CAL_ICAL_EVENT_RECURRENCE_FREQ_MINUTELY:
            plain_error_message('ERROR: The editing of events recurring MINUTELY not currently supported', $color);
            exit;

         case SM_CAL_ICAL_EVENT_RECURRENCE_FREQ_HOURLY:
            plain_error_message('ERROR: The editing of events recurring HOURLY not currently supported', $color);
            exit;

         case SM_CAL_ICAL_EVENT_RECURRENCE_FREQ_DAILY:
         case SM_CAL_ICAL_EVENT_RECURRENCE_FREQ_WEEKLY:
         case SM_CAL_ICAL_EVENT_RECURRENCE_FREQ_MONTHLY:
         case SM_CAL_ICAL_EVENT_RECURRENCE_FREQ_YEARLY:
            return $rrule['FREQ'];

      }

   }



   /**
     * Get Recurrence Date(s)
     *
     * NOTE: careful when getting value of RDATE, as it can be an 
     *       array of arrays (PERIOD start/end timestamps) *OR* 
     *       an array with just one PERIOD, *OR* an array of 
     *       regular timestamps!  Always check the type!
     *          $this->recurrenceDates->getType
     *          SM_CAL_ICAL_PROPERTY_TYPE_PERIOD
     *          or one of the date/time types
     *
     * @return mixed The recurrence date(s) for this event (int or array of ints)
     *
     * @access public
     *
     */
   function getRecurrenceDates()
   {
      return $this->recurrenceDates->getValue();
   }



   /**
     * Get Recurrence Exclusion Rule
     *
     * @return string The recurrence exclusion rule for this event
     *
     * @access public
     *
     */
   function getRecurrenceExclusionRule()
   {
      return $this->recurrenceExclusionRule->getValue();
   }



   /**
     * Get Recurrence Exclusion Date(s)
     *
     * @return mixed The recurrence exclusion date(s) for this event (int or array of ints)
     *
     * @access public
     *
     */
   function getRecurrenceExclusionDates()
   {
      return $this->recurrenceExclusionDates->getValue();
   }



   /**
     * Get Percent Complete
     *
     * @return int The percent value of completion for this todo/task
     *
     * @access public
     *
     */
   function getPercentComplete()
   {
      return $this->percentComplete->getValue();
   }



   /** 
     * Determines if this event is recurring
     *
     * @return boolean TRUE if this event is recurring, FALSE otherwise
     *
     * @access public
     *
     */
   function isRecurring()
   {
//TODO: does this logic hold up once we add todo/task types?
      return (!$this->isOneTime());
   }



   /** 
     * Determines if this event is one time (not recurring)
     *
     * @return boolean TRUE if this event is one-time, FALSE otherwise
     *
     * @access public
     *
     */
   function isOneTime()
   {
      $rrule = $this->getRecurrenceRule();
      $rdates = $this->getRecurrenceDates();
      return (empty($rrule) && empty($rdates));
   }



   /** 
     * Determines if this event is a task/todo item
     *
     * @return boolean TRUE if this event is a task/todo item, FALSE otherwise
     *
     * @access public
     *
     */
   function isTask()
   {
      return ($this->getEventType() == SM_EVENT_TYPE_TODO);
   }



   /**
     * Set Event Priority
     *
     * @param int $priority The new event priority to be assigned to this event
     *
     * @access public
     *
     */
   function setPriority($priority)
   {
      $this->priority->setValue($priority);
   }



   /**
     * Get Event Priority
     *
     * @return int This event's priority
     *
     * @access public
     *
     */
   function getPriority()
   {
      return $this->priority->getValue();
   }



   /**
     * Get Parent Calendars
     *
     * @return array A list of this event's parent calendar IDs
     *
     * @access public
     *
     */
   function getParentCalendars()
   {
      $p = $this->parentCalendars->getValue();
      if (empty($p))
         return array();
      else if (is_string($p))
         return array($p);
      else
         return $p;
   }



   /**
     * Reset Parent Calendars
     *
     * Clears all parent calendars - use with care!
     *
     * @access public
     *
     */
   function resetParentCalendars()
   {
      $this->parentCalendars->setValue(array());
   }



   /**
     * Add Parent Calendar
     *
     * Note that duplicates are automatically weeded out.
     *
     * @param string $id The ID of the parent calendar to be added
     *
     * @access public
     *
     */
   function addParentCalendar($id)
   {
      $p = $this->getParentCalendars();
      if (!in_array($id, $p)) $p[] = $id;
      $this->parentCalendars->setValue($p);
   }



   /**
     * Remove Parent Calendar
     *
     * @param string $id The ID of the parent calendar to be removed
     *
     * @access public
     *
     */
   function removeParentCalendar($id)
   {
      $p = $this->getParentCalendars();
      $p = array_diff($p, array($id));
      $this->parentCalendars->setValue($p);
   }



   /**
     * Get Creator Name
     *
     * @return string This event's creator
     *
     * @access public
     *
     */
   function createdBy()
   {
      return $this->createdBy->getValue();
   }



   /**
     * Set Username of User Who Created This Event
     *
     * @param string $user The name of the user who created this event
     *
     * @access public
     *
     */
   function setCreator($user)
   {
      return $this->createdBy->setValue($user);
   }



   /**
     * Get Creation Date
     *
     * @return timestamp This event's creation date
     *
     * @access public
     *
     */
   function createdOn()
   {
      return $this->createdOn->getValue();
   }



   /**
     * Get User That Last Updated Event
     *
     * @return string This event's last editor
     *
     * @access public
     *
     */
   function lastUpdatedBy()
   {
      return $this->lastUpdatedBy->getValue();
   }



   /**
     * Set Username of User Who Last Updated This Event
     *
     * @param string $user The name of the user updating this event
     *
     * @access public
     *
     */
   function setLastUpdator($user)
   {
      return $this->lastUpdatedBy->setValue($user);
   }



   /**
     * Get Last Update Date
     *
     * @return timestamp The date of this event's last update
     *
     * @access public
     *
     */
   function lastUpdatedOn()
   {
      return $this->lastUpdatedOn->getValue();
   }



   /**
     * Set Last Update Date
     *
     * @param string $timestamp The date this calendar was last updated
     *                          (should be a UTC-formatted date/time string)
     *
     * @access public
     *
     */
   function setLastUpdateDate($timestamp)
   {
      return $this->lastUpdatedOn->setValue($timestamp);
   }  
         
      
         
   /**
     * Get Event Owners
     *
     * @return array An array listing all event owners
     *
     * @access public
     *
     */
   function getOwners()
   {
      $o = $this->owners->getValue();
      if (empty($o))
         return array();
      else if (is_string($o))
         return array($o);
      else
         return $o;
   }



   /**
     * Get Readable Users
     *
     * @return array An array listing all users 
     *               who have read access to this event
     *
     * @access public
     *
     */
   function getReadableUsers()
   {
      $r = $this->readable_users->getValue();
      if (empty($r))
         return array();
      else if (is_string($r))
         return array($r);
      else
         return $r;
   }



   /**
     * Get Writeable Users
     *
     * @return array An array listing all users 
     *               who have write access to this event
     *
     * @access public
     *
     */
   function getWriteableUsers()
   {
      $w = $this->writeable_users->getValue();
      if (empty($w))
         return array();
      else if (is_string($w))
         return array($w);
      else
         return $w;
   }



   /**
     * Get Unknown Attributes
     *
     * @return array An array of all unknown attributes
     *
     * @access public
     *
     */
   function getUnknownAttributes()
   {
      return $this->unknownAttributes;
   }



   /**
//LEFT OFF HERE -- after my massive rewrite (added/removed some fields, make sure this
//LEFT OFF HERE -- validation stuff is OK -- surely there are some changes!
//LEFT OFF HERE ------- make sure if this is a TODO that status is a TODO status constant
//LEFT OFF HERE ------- make sure if this is a EVENT that status is a EVENT status constant
     * Check Validity of All Event Fields
     *
     * Determines if the data contained in this object
     * is sufficient for saving the event correctly.
     *
     * @return string An error message if there is any invalid
     *                field data, otherwise an empty string
     *                is returned.
     *
     * @access public
     *
     */
   function validateFields()
   {

//LEFT OFF HERE -- these are all checks good for one time and recurring events, but NOT for tasks/todos or holidays!
//LEFT OFF HERE -- these are all checks good for one time and recurring events, but NOT for tasks/todos or holidays!
//LEFT OFF HERE -- these are all checks good for one time and recurring events, but NOT for tasks/todos or holidays!
//LEFT OFF HERE -- these are all checks good for one time and recurring events, but NOT for tasks/todos or holidays!
      $d = $this->getDomain();
      if (empty($d)) 
         return _("Must have domain information");

      if (!$this->isTask()
       && $this->getEventType() != SM_EVENT_TYPE_EVENT)
         return _("Valid event type must be specified");

      global $RECURRENCE_TYPES;
      if ($this->isRecurring()
       && !in_array($this->getRecurrenceType(), array_keys($RECURRENCE_TYPES)))
         return _("Must specify valid recurrence type");

      if ($this->isRecurring())
      {
         $rrule = $this->getRecurrenceRule();
         if (isset($rrule['UNTIL']) && !empty($rrule['UNTIL'])
          && $rrule['UNTIL'] < $this->getEndDateTime() 
                                     - date('H', $this->getEndDateTime()) * 3600 
                                     - date('i', $this->getEndDateTime()) * 60)
            return _("Overall end of recurrence for recurring events must come after end of initial occurrence");
      }

      if ($this->isRecurring()
       && (!is_numeric($this->getRecurrenceInterval()) || $this->getRecurrenceInterval() < 1))
         return _("Must specify valid recurrence interval");

      $s = $this->getSummary();
      if (empty($s))
         return _("Event must have a summary/title");

      global $EVENT_PRIORITIES;
      if (!in_array($this->getPriority(), array_keys($EVENT_PRIORITIES)))
         return _("Must specify valid event priority");

      $st = $this->getStartDateTime();
      if (empty($st) || !is_numeric($st)) 
         return _("Event must have a valid start date/time");

      $et = $this->getEndDateTime();
      if (empty($et) || !is_numeric($et)) 
         return _("Event must have a valid end date/time");

      if ($et < $st)
         return _("End date/time must come after start date and time");

      $p = $this->getParentCalendars();
      if (empty($p) || !is_array($p)) 
         return _("At least one parent calendar must be given");

      $cb = $this->createdBy();
      if (empty($cb)) 
         return _("Event must have a creator");

      $co = $this->createdOn();
      if (empty($co)) 
         return _("Event must have a creation date");

      $o = $this->getOwners();
      if (empty($o) || !is_array($o)) 
         return _("Event must have at least one owner");


      // no problems found
      //
      return '';

   }



   /**
     * Determines if this event comes before 
     * or after the given event, for use with
     * sorting events.
     *
     * @param object $otherEvent The event to compare
     *                           to this one.
     *
     * @return int -1 if this event comes first,
     *             1 if this event comes second,
     *             or 0 if the events should be 
     *             considered to be "equal".
     *
     * @access public
     *
     */
   function compareTo($otherEvent)
   {

      // events of the same type?  just compare start dates
      //
      if (($this->isOneTime()
       && $otherEvent->isOneTime())
       || ($this->isRecurring()
       && $otherEvent->isRecurring()))
      {

         if ($this->getStartDateTime() < $otherEvent->getStartDateTime())
            return -1;

         if ($this->getStartDateTime() > $otherEvent->getStartDateTime())
            return 1;

         return 0;

      }


      // different types?  we always sort recurring events first
      //
//TODO: hmmm, maybe we want to do something smarter here?
      else if ($this->isOneTime()
       && $otherEvent->isRecurring())
         return 1;


      // different types?  we always sort recurring events first
      //
//TODO: hmmm, maybe we want to do something smarter here?
      else if ($this->isRecurring()
       && $otherEvent->isOneTime())
         return -1;


      else
      {
         global $color;
         plain_error_message('ERROR IN EVENT CLASS (compareTo): Cannot compare events without event type', $color);
         exit;
      }

   }



   /**
     * Returns the lenth of this event in number of
     * minutes
     *
     * @return int The number of minutes that this event lasts
     *
     * @access public
     *
     */
   function lengthInMinutes()
   {

      return (($this->getEndDateTime() - $this->getStartDateTime()) / 60);

   }



   /**
     * Returns the lenth of this event in number of
     * quarter hours
     *
     * @return int The number of quarter hours that 
     *             this event lasts
     *
     * @access public
     *
     */
   function lengthInQuarterHours()
   {

      // can't simply subtract start time from end time
      // because events that start, say at 1:59
      // and end at 2:16 will be shown as only 2 quarters
      // long, but for display purposes, the return value
      // has to be three in such a case
      //
      // so we extend start and end times to fill out
      // the quarter of the hour that they fall in first
      // and then do the math
      //
      $start = getdate($this->getStartDateTime());
      $startMinute = $start['minutes'];
      if ($startMinute < 15) $startMinute = 0;
      else if ($startMinute < 30) $startMinute = 15;
      else if ($startMinute < 45) $startMinute = 30;
      else $startMinute = 45;
      $start = mktime($start['hours'], $startMinute, $start['seconds'], 
                      $start['mon'], $start['mday'], $start['year']);

      $end = getdate($this->getEndDateTime());
      $endMinute = $end['minutes'];
      $endHour = $end['hours'];
      if ($endMinute > 45)
      {
         $endMinute = 0;
         $endHour++;
      }
      else if ($endMinute > 30) $endMinute = 45;
      else if ($endMinute > 15) $endMinute = 30;
      else if ($endMinute > 0) $endMinute = 15;
      $end = mktime($endHour, $endMinute, $end['seconds'], 
                    $end['mon'], $end['mday'], $end['year']);

      return round(($end - $start) / 60 / 15);

   }



   /**
     * Returns the quarter hour during which this 
     * event starts (0, 15, 30 or 45).
     *
     * This method helps abstract the fact that
     * recurring events might have start timestamps
     * on a completely different day but should be
     * seen as starting on other days at a similar
     * time.
     *
     * Note that regular events that occur on the given
     * day but start on an earlier day are seen as starting
     * at midnight.
     *
     * @param int $year The year of the day for which 
     *                  to check start time
     * @param int $month The month of the day for which 
     *                   to check start time
     * @param int $day The day for which to check start time
     *
     * @return mixed The quarter of the hour during which 
     *               this event starts (0, 15, 30 or 45), 
     *               or FALSE if this event does not have 
     *               a start time on the given day.
     *
     * @access public
     *
     */
   function startQuarterHour($year, $month, $day)
   {

      if (!$this->occursOnDay($year, $month, $day)) return FALSE;

      if (!$this->startsOnDay($year, $month, $day)) return 0;

      $startMinute = date('i', $this->getStartDateTime());
      if ($startMinute < 15) return 0;
      if ($startMinute < 30) return 15;
      if ($startMinute < 45) return 30;
      return 45;

   }



   /**
     * Returns the quarter hour during which this
     * event ends (0, 15, 30, 45 or 60 (meaning next 
     * hour, zero minute)).
     *
     * This method helps abstract the fact that
     * recurring events might have end timestamps
     * on a completely different day but should be
     * seen as ending on other days at a similar
     * time.
     *
     * Note that regular events that occur on the given
     * day but end on a later day are seen as ending
     * at 23:45.
     *
     * @param int $year The year of the day for which
     *                  to check end time
     * @param int $month The month of the day for which
     *                   to check end time
     * @param int $day The day for which to check end time
     *
     * @return mixed The quarter of the hour during which
     *               this event ends (0, 15, 30, 45, or 60),
     *               or FALSE if this event does not have
     *               a end time on the given day.
     *
     * @access public
     *
     */
   function endQuarterHour($year, $month, $day)
   {

      if (!$this->occursOnDay($year, $month, $day)) return FALSE;

      if (!$this->endsOnDay($year, $month, $day)) return 45;

      $endMinute = date('i', $this->getEndDateTime());
      if ($endMinute > 45) return 60;
      if ($endMinute > 30) return 45;
      if ($endMinute > 15) return 30;
      if ($endMinute > 0) return 15;
      return 0;

   }



   /**
     * Returns start date (date only, no time) in 
     * displayable format
     *
     * @param string $format The desired format of 
     *                       the returned date (optional)
     * @param int $year For recurring events, the year that
     *                  identifies the desired occurrence for
     *                  which to return formatted start date
     *                  (optional; defaults to initial start
     *                  date)
     * @param int $month For recurring events, the month that
     *                   identifies the desired occurrence for
     *                   which to return formatted start date
     *                   (optional; defaults to initial start
     *                   date) 
     * @param int $day For recurring events, the day that
     *                 identifies the desired occurrence for
     *                 which to return formatted start date
     *                 (optional; defaults to initial start
     *                 date) 
     *
     * @return string The desired date
     *
     * @access public
     *
     */
   function formattedStartDate($format='', $year=0, $month=0, $day=0)
   {

      if (empty($format))
         $format = 'M d, Y';


      if ($year == 0 || $month == 0) // days could be zero or negative   || $day == 0)
         return date_intl($format, $this->getStartDateTime());


      // 
      // otherwise need to check actual event occurrence
      // on the given day to determine this occurrence's 
      // date
      // 


      // regular one-time events are easy to figure out
      //
      if ($this->isOneTime())
      {

         return date_intl($format, $this->getStartDateTime());

      }


      // recurring events require more involved check
      //
      else if ($this->isRecurring())
      {

         $theDay = mktime(0, 0, 0, $month, $day, $year);


         // in case day is out of range, bring it back in
         //
         $correctedDate = getdate($theDay);
         $month = $correctedDate['mon'];
         $day = $correctedDate['mday'];
         $year = $correctedDate['year'];


         // original start date?  same as above 
         //
         if (dayIsBetween($year, $month, $day, $this->getStartDateTime(), $this->getEndDateTime()))
            return date_intl($format, $this->startDateTime);


         // otherwise, iterate through occurrences looking for match
         //
//LEFT OFF HERE --- the max date being passed in is midnight, but that's too early
         $occurrenceDate = $this->iterateEventOccurrences('findOccurrenceStartDate', 
                                                          array($year, $month, $day), 
                                                          $theDay);

         if ($occurrenceDate === FALSE)
         {
            global $color;
            plain_error_message('ERROR IN EVENT CLASS (formattedStartDate): Event does not occur on given date', $color);
            exit;
         }

         return date_intl($format, $occurrenceDate);

      }

   }



   /**
     * Returns end date (date only, no time) in
     * displayable format
     *
     * @param string $format The desired format of
     *                       the returned date (optional)
     * @param int $year For recurring events, the year that
     *                  identifies the desired occurrence for
     *                  which to return formatted end date
     *                  (optional; defaults to initial end
     *                  date)
     * @param int $month For recurring events, the month that
     *                   identifies the desired occurrence for
     *                   which to return formatted end date
     *                   (optional; defaults to initial end
     *                   date) 
     * @param int $day For recurring events, the day that
     *                 identifies the desired occurrence for
     *                 which to return formatted end date
     *                 (optional; defaults to initial end
     *                 date) 
     *
     * @return string The desired date
     *
     * @access public
     *
     */
   function formattedEndDate($format='', $year=0, $month=0, $day=0)
   {

      if (empty($format))
         $format = 'M d, Y';


      if ($year == 0 || $month == 0) // days could be zero or negative  || $day == 0)
         return date_intl($format, $this->getEndDateTime());


      //
      // otherwise need to check actual event occurrence
      // on the given day to determine this occurrence's
      // date
      //


      // regular one-time events are easy to figure out
      //
      if ($this->isOneTime())
      {

         return date_intl($format, $this->getEndDateTime());

      }


      // recurring events require more involved check
      //
      else if ($this->isRecurring())
      {

         $theDay = mktime(0, 0, 0, $month, $day, $year);


         // in case day is out of range, bring it back in
         //
         $correctedDate = getdate($theDay);
         $month = $correctedDate['mon'];
         $day = $correctedDate['mday'];
         $year = $correctedDate['year'];


         // original start date?  same as above
         //
         if (dayIsBetween($year, $month, $day, $this->getStartDateTime(), $this->getEndDateTime()))
            return date_intl($format, $this->getEndDateTime());


         // otherwise, iterate through occurrences looking for match
         //
//LEFT OFF HERE --- the max date being passed in is midnight, but that's too early
         $occurrenceDate = $this->iterateEventOccurrences('findOccurrenceStartDate',
                                                          array($year, $month, $day),
                                                          $theDay);

         if ($occurrenceDate === FALSE)
         {
            global $color;
            plain_error_message('ERROR IN EVENT CLASS (formattedEndDate): Event does not occur on given date', $color);
            exit;
         }

         return date_intl($format, $occurrenceDate + $this->getEndDateTime() - $this->getStartDateTime());

      }

   }



   /**
     * Returns end recurrence date (if available) in displayable format
     *
     * @param string $format The desired format of
     *                       the returned date (optional)
     *
     * @return mixed The desired date as a string, or FALSE 
     *               if the recurrence rule does not exist
     *               or does not contain an UNTIL clause
     *
     * @access public
     *
     */
   function formattedEndRecurrenceDate($format='M d, Y')
   {
//TODO: when GUI has a way to indicate end of exclusion rule (yeah, right, 
//      when will that really be useful?), need a similar function for 
//      formattedEndRecurrenceExclusionDate

      $rrule = $this->getRecurrenceRule();

      if (!empty($rrule) && isset($rrule['UNTIL']) && !empty($rrule['UNTIL']))
         return date_intl($format, $rrule['UNTIL']);

      return FALSE;

   }



   /**
     * Returns end recurrence count (if available)
     *
     * @return mixed The desired max recurrence count (integer), 
     *               or FALSE if the recurrence rule does not exist
     *               or does not contain a COUNT clause
     *
     * @access public
     *
     */
   function maxRecurrenceCount()
   {
//TODO: when GUI has a way to indicate end of exclusion rule (yeah, right, 
//      when will that really be useful?), need a similar function for 
//      maxRecurrenceExclusionCount

      $rrule = $this->getRecurrenceRule();

      if (!empty($rrule) && isset($rrule['COUNT']) && !empty($rrule['COUNT']))
         return $rrule['COUNT'];

      return FALSE;

   }



   /**
     * Returns start time (time only, no date)
     *
     * @return array A three-element array, the first 
     *               element being the hours (24-hour 
     *               clock), the second being the minutes,
     *               the third being the seconds.
     *
     * @access public
     *
     */
   function startTime()
   {

      $dateInfo = getdate($this->getStartDateTime());
      return array($dateInfo['hours'], $dateInfo['minutes'], $dateInfo['seconds']);

   }



   /**
     * Returns end time (time only, no date)
     *
     * @return array A two-element array, the first 
     *               element being the hours (24-hour 
     *               clock), the second being the minutes,
     *               the third being the seconds.
     *
     * @access public
     *
     */
   function endTime()
   {

      $dateInfo = getdate($this->getEndDateTime());
      return array($dateInfo['hours'], $dateInfo['minutes'], $dateInfo['seconds']);

   }



   /**
     * Returns start time (time only, no date) in 
     * displayable format
     *
     * @param string $format The desired format of
     *                       the returned time (optional)
     * @param int $twentyFourHourOverride This parameter may be used
     *                                    to specify that time should
     *                                    be returned in twenty-four
     *                                    hour format (if given as 1),
     *                                    twelve hour format (if given 
     *                                    as 0), or whatever the system
     *                                    default is (when given as -1)
     *                                    (an attempt will be made to
     *                                    find a global configuration 
     *                                    variable to determine which 
     *                                    is desired) (optional; when 
     *                                    not specified, defaults to 
     *                                    system configuration 
     *                                    setting, or twenty-four hour 
     *                                    format if no global configuration 
     *                                    value is found).  Note that this
     *                                    parameter has no effect when
     *                                    $format is given by the caller.
     *
     * @return string The desired time
     *
     * @access public
     *
     */
   function formattedStartTime($format='', $twentyFourHourOverride=-1)
   {

      if (empty($format))
      {
         global $hour_format;
         if ($twentyFourHourOverride == -1)
            $twentyFourHourOverride = ($hour_format == SMPREF_TIME_24HR);

         if ($twentyFourHourOverride == 0)
            $format = 'g:ia';
         else
            $format = 'H:i';
      }


      return date_intl($format, $this->getStartDateTime());
   
   }



   /**
     * Returns end time (time only, no date) in
     * displayable format
     * 
     * @param string $format The desired format of
     *                       the returned time (optional)
     * @param int $twentyFourHourOverride This parameter may be used
     *                                    to specify that time should
     *                                    be returned in twenty-four
     *                                    hour format (if given as 1),
     *                                    twelve hour format (if given
     *                                    as 0), or whatever the system
     *                                    default is (when given as -1)
     *                                    (an attempt will be made to
     *                                    find a global configuration
     *                                    variable to determine which
     *                                    is desired) (optional; when
     *                                    not specified, defaults to
     *                                    system configuration
     *                                    setting, or twenty-four hour
     *                                    format if no global configuration
     *                                    value is found).  Note that this
     *                                    parameter has no effect when
     *                                    $format is given by the caller.
     *
     * @return string The desired time
     * 
     * @access public
     *
     */
   function formattedEndTime($format='', $twentyFourHourOverride=-1)
   { 
     
      if (empty($format))
      {
         global $hour_format;
         if ($twentyFourHourOverride == -1)
            $twentyFourHourOverride = ($hour_format == SMPREF_TIME_24HR);

         if ($twentyFourHourOverride == 0)
            $format = 'g:ia';
         else
            $format = 'H:i';
      }


      return date_intl($format, $this->getEndDateTime());

   }



   /**
     * Returns the year, month and day this event stops
     * recurring (if available)
     *
     * @return mixed If no recurrence rule is available,
     *               or it does not have an UNTIL clause,
     *               FALSE is returned; otherwise a three-
     *               element array is returned, the first 
     *               element being the four-digit year,
     *               the second being the month, and 
     *               the last being the day.
     *
     * @access public
     *
     */
   function recurrenceEnd()
   {

      $rrule = $this->getRecurrenceRule();

      if (!empty($rrule) && isset($rrule['UNTIL']) && !empty($rrule['UNTIL']))
      {
         $dateInfo = getdate($rrule['UNTIL']);
         return array($dateInfo['year'], $dateInfo['mon'], $dateInfo['mday']);
      }

      return FALSE;

   }



   /**
     * Returns the year, month and day this event starts
     *
     * @return array A three-element array, the first 
     *               element being the four-digit year,
     *               the second being the month, and 
     *               the last being the day.
     *
     * @access public
     *
     */
   function startDate()
   {

      $dateInfo = getdate($this->getStartDateTime());
      return array($dateInfo['year'], $dateInfo['mon'], $dateInfo['mday']);

   }



   /**
     * Returns the year, month and day this event ends
     *
     * @return array A three-element array, the first 
     *               element being the four-digit year,
     *               the second being the month, and 
     *               the last being the day.
     *
     * @access public
     *
     */
   function endDate()
   {

      $dateInfo = getdate($this->getEndDateTime());
      return array($dateInfo['year'], $dateInfo['mon'], $dateInfo['mday']);

   }



   /**
     * Returns the hour of day this event starts 
     * on the given day (24-hour clock, no 
     * leading zeros).
     *
     * This method helps abstract the fact that 
     * recurring events might have start timestamps 
     * on a completely different day but should be 
     * seen as starting on other days at a similar 
     * time.
     *
     * Note that regular events that occur on the given 
     * day but start on an earlier day are seen as starting
     * at midnight.
     *
     * @param int $year The year of the day for which 
     *                  to check start time
     * @param int $month The month of the day for which 
     *                   to check start time
     * @param int $day The day for which to check start time
     *
     * @return mixed The hour of the day that this event 
     *               starts, or FALSE if this event does
     *               not have a start time on the given day.
     *
     * @access public
     *
     */
   function startHour($year, $month, $day)
   {

      if (!$this->occursOnDay($year, $month, $day)) return FALSE;

      if (!$this->startsOnDay($year, $month, $day)) return 0;

      return date('G', $this->getStartDateTime());

   }



   /**
     * Returns the hour of day this event ends
     * on the given day (24-hour clock, no
     * leading zeros).
     *
     * This method helps abstract the fact that
     * recurring events might have end timestamps
     * on a completely different day but should be
     * seen as ending on other days at a similar
     * time.
     *
     * Note that regular events that occur on the given
     * day but end on a later day are seen as ending
     * at 23:45.
     *
     * @param int $year The year of the day for which
     *                  to check end time
     * @param int $month The month of the day for which
     *                   to check end time
     * @param int $day The day for which to check end time
     *
     * @return mixed The hour of the day that this event
     *               ends, or FALSE if this event does
     *               not have a end time on the given day.
     *
     * @access public
     *
     */
   function endHour($year, $month, $day)
   {

      if (!$this->occursOnDay($year, $month, $day)) return FALSE;

      if (!$this->endsOnDay($year, $month, $day)) return 23;

      return date('G', $this->getEndDateTime());

   }



   /**
     * Determines if this event starts between 
     * the given timestamps.
     *
     * @param int $begin The beginning of the 
     *                   timeframe to check for 
     *                   event start.
     * @param int $end The end of the timeframe 
     *                   to check for event start.
     *
     * @return boolean TRUE if this event starts 
     *                 during the given timeframe, 
     *                 FALSE otherwise.
     *
     * @access public
     *
     */
   function startsBetween($begin, $end)
   {
echo "\n\n<hr><h3>Event->startsBetween(): coding not finished on this function!  finish it before you use it somewhere!</h3><hr>";exit;

      // regular one-time events are easy to figure out
      //
      if ($this->isOneTime())
      {

         return ($begin <= $this->getStartDateTime() && $this->getStartDateTime() <= $end);

      }


      // recurring events require more involved check
      //
      else if ($this->isRecurring())
      {
//LEFT OFF HERE
//LEFT OFF HERE
//LEFT OFF HERE
      }


      else
      {
         global $color;
         plain_error_message('ERROR IN EVENT CLASS (startsBetween): Cannot check event start time without event type', $color);
         exit;
      }

   }



   /**
     * Determines if this event has an occurrence on the
     * given date/time.
     *
     * @param int $timestamp The timestamp to check for occurrence
     * @param boolean $useExclusionRule When TRUE, recurring events
     *                                  are evaluated using the 
     *                                  exclusion recurrence rule (optional;
     *                                  default=FALSE)
     *
     * @return boolean TRUE if this event occurs on the given timestamp, FALSE otherwise.
     *
     * @access public
     *
     */
   function occursOnTimestamp($timestamp, $useExclusionRule=FALSE)
   {

      // regular one-time events are easy to figure out
      //
      if ($this->isOneTime())
      {

         return ($this->getStartDateTime() <= $timestamp 
              && $timestamp <= $this->getEndDateTime());

      }


      // recurring events require more involved check
      //
      else if ($this->isRecurring())
      {

         // always occurs on the original start date
         //
         if ($this->getStartDateTime() <= $timestamp 
          && $timestamp <= $this->getEndDateTime())
            return TRUE;


         // otherwise, iterate through occurrences looking for match
         //
         return $this->iterateEventOccurrences('checkOccurrenceForTimestamp',
                                               array($timestamp), $timestamp, 
                                               $useExclusionRule);

      }


      else
      {
         global $color;
         plain_error_message('ERROR IN EVENT CLASS (occursOnTimestamp): Cannot check event occurrence without event type', $color);
         exit;
      }

   }



   /**
     * Determines if this event has an occurrence on the
     * given day.
     *
     * @param int $year The year of the day to check for occurrence
     * @param int $month The month of the day to check for occurrence
     * @param int $day The day to check for occurrence
     *
     * @return boolean TRUE if this event occurs on the given day, FALSE otherwise.
     *
     * @access public
     *
     */
   function occursOnDay($year, $month, $day)
   {

      // regular one-time events are easy to figure out
      //
      if ($this->isOneTime())
      {

         return dayIsBetween($year, $month, $day, $this->getStartDateTime(), $this->getEndDateTime());

      }


      // recurring events require more involved check
      //
      else if ($this->isRecurring())
      {

         $theDay = mktime(0, 0, 0, $month, $day, $year);


         // can we use date cache?  this test doesn't test the contents
         // of $this->startDateCache directly, but trusts that it was built
         // correctly when the cachedOccurrences array was built
         //
         if (sizeof($this->cachedOccurrences) > 0
          && ($this->cachedOccurrences[sizeof($this->cachedOccurrences) - 1] >= $theDay
           || $this->cachedOccurrencesThruDate >= $theDay))
         {
            return isset($this->startDateCache[$year][intval($month)][intval($day)]['timestamp']);
         }


         else
         {

            // always occurs on the original start date 
            //
            if (dayIsBetween($year, $month, $day, $this->getStartDateTime(), $this->getEndDateTime(), $theDay))
               return TRUE;


            // otherwise, iterate through occurrences looking for match
            //
//LEFT OFF HERE --- the max date being passed in is midnight, but that's too early
            return $this->iterateEventOccurrences('checkOccurrence', 
                                                  array($year, $month, $day, $theDay), 
                                                  $theDay);

         }

      }


      else
      {
         global $color;
         plain_error_message('ERROR IN EVENT CLASS (occursOnDay): Cannot check event occurrence without event type', $color);
         exit;
      }

   }



   /**
     * Determines if this event starts on the given day.
     *
     * @param int $year The year of the day to check for starting.
     * @param int $month The month of the day to check for starting.
     * @param int $day The day to check for starting.
     *
     * @return boolean TRUE if this event starts on the given day, FALSE otherwise.
     *
     * @access public
     *
     */
   function startsOnDay($year, $month, $day)
   {

      $startInfo = getdate($this->getStartDateTime());
      $theDay = mktime(0, 0, 0, $month, $day, $year);
      $dayInfo = getdate($theDay);


      // regular one-time events are easy to figure out
      //
      if ($this->isOneTime())
      {

         return ($startInfo['year'] == $year && $startInfo['yday'] == $dayInfo['yday']);

      }


      // recurring events require more involved check
      //
      else if ($this->isRecurring())
      {

         // can we use date cache?  this test doesn't test the contents
         // of $this->startDateCache directly, but trusts that it was built
         // correctly when the cachedOccurrences array was built
         //
         if (sizeof($this->cachedOccurrences) > 0
          && ($this->cachedOccurrences[sizeof($this->cachedOccurrences) - 1] >= $theDay
           || $this->cachedOccurrencesThruDate >= $theDay))
         {
            return isset($this->startDateCache[$year][intval($month)][intval($day)]['timestamp']);
         }


         else
         {

            // always starts on the original start date
            //
            if ($startInfo['year'] == $year && $startInfo['yday'] == $dayInfo['yday'])
               return TRUE;


            // otherwise, iterate through occurrences looking for match
            //
//LEFT OFF HERE --- the max date being passed in is midnight, but that's too early
            return $this->iterateEventOccurrences('checkStartsOnDay',
                                                  array($year, $month, $day, $dayInfo),
                                                  $theDay);

         }

      }


      else
      {
         global $color;
         plain_error_message('ERROR IN EVENT CLASS (startsOnDay): Cannot check event start without event type', $color);
         exit;
      }

   }



   /**
     * Determines if this event starts on the given date/time.
     *
     * @param int $timestamp The timestamp to check for starting.
     * @param boolean $useExclusionRule When TRUE, recurring events
     *                                  are evaluated using the 
     *                                  exclusion recurrence rule (optional;
     *                                  default=FALSE)
     *
     * @return boolean TRUE if this event starts on the given timestamp, FALSE otherwise.
     *
     * @access public
     *
     */
   function startsOnTimestamp($timestamp, $useExclusionRule=FALSE)
   {

      // regular one-time events are easy to figure out
      //
      if ($this->isOneTime())
      {

         return ($timestamp == $this->getStartDateTime());

      }


      // recurring events require more involved check
      //
      else if ($this->isRecurring())
      {

         // always starts on the original start date
         //
         if ($timestamp == $this->getStartDateTime())
            return TRUE;


         // otherwise, iterate through occurrences looking for match
         //
         return $this->iterateEventOccurrences('checkStartsOnTimestamp',
                                               array($timestamp), $timestamp, $useExclusionRule);

      }


      else
      {
         global $color;
         plain_error_message('ERROR IN EVENT CLASS (startsOnTimestamp): Cannot check event start without event type', $color);
         exit;
      }

   }



   /**
     * Determines if this event ends on the given day.
     *
     * @param int $year The year of the day to check for ending.
     * @param int $month The month of the day to check for ending.
     * @param int $day The day to check for ending.
     *
     * @return boolean TRUE if this event ends on the given day, FALSE otherwise.
     *
     * @access public
     *
     */
   function endsOnDay($year, $month, $day)
   {

      $endInfo = getdate($this->getEndDateTime());
      $theDay = mktime(0, 0, 0, $month, $day, $year);
      $dayInfo = getdate($theDay);


      // regular one-time events are easy to figure out
      //
      if ($this->isOneTime())
      {

         return ($endInfo['year'] == $year && $endInfo['yday'] == $dayInfo['yday']);

      }


      // recurring events require more involved check
      //
      else if ($this->isRecurring())
      {

         // can we use date cache?  this test doesn't test the contents
         // of $this->endDateCache directly, but trusts that it was built
         // correctly when the cachedOccurrences array was built
         //
         if (sizeof($this->cachedOccurrences) > 0
          && ($this->cachedOccurrences[sizeof($this->cachedOccurrences) - 1] >= $theDay
           || $this->cachedOccurrencesThruDate >= $theDay))
         {
            return isset($this->endDateCache[$year][intval($month)][intval($day)]['timestamp']);
         }


         else
         {

            // always ends on the original end date
            //
            if ($endInfo['year'] == $year && $endInfo['yday'] == $dayInfo['yday'])
               return TRUE;


            // otherwise, iterate through occurrences looking for match
            //
//LEFT OFF HERE --- the max date being passed in is midnight, but that's too early
            return $this->iterateEventOccurrences('checkEndsOnDay',
                                                  array($year, $month, $day, $dayInfo),
                                                  $theDay);

         }

      }


      else
      {
         global $color;
         plain_error_message('ERROR IN EVENT CLASS (endsOnDay): Cannot check event end without event type', $color);
         exit;
      }

   }



   /**
     * Removes a user from this event, from the list
     * of owners, readable users, and writeable users
     *
     * @param string $user The user to be removed
     *
     * @access public
     *
     */
   function remove_user($user)
   {

      $this->owners->setValue(array_diff($this->getOwners(), array($user)));
      $this->readable_users->setValue(array_diff($this->getReadableUsers(), array($user)));
      $this->writeable_users->setValue(array_diff($this->getWriteableUsers(), array($user)));

   }



   /** 
     * Adds a new user to this event
     *
     * @param string $user The user name being added
     * @param string $accessLevel The access level being
     *                            granted to the new user,
     *                            which should correspond
     *                            to the event access constants
     *                            defined in {@link constants.php}
     * @access public
     *
     */
   function add_user($user, $accessLevel)
   {

      if ($accessLevel == SM_CAL_ACCESS_LEVEL_OWNER)
         $this->owners->setValue(array_merge($this->getOwners(), array($user)));

      else if ($accessLevel == SM_CAL_ACCESS_LEVEL_READ)
         $this->readable_users->setValue(array_merge($this->getReadableUsers(), array($user)));

      else if ($accessLevel == SM_CAL_ACCESS_LEVEL_WRITE)
         $this->writeable_users->setValue(array_merge($this->getWriteableUsers(), array($user)));

   }



   /**
     * Determines if this event falls on the given calendar
     *
     * @param string $calID The ID of the calendar to check
     *
     * @return boolean TRUE if this event falls on the given calendar, FALSE otherwise
     *
     * @access public
     *
     */
   function fallsOnCalendar($calID)
   {

      return in_array($calID, $this->getParentCalendars());

   }



   /**
     * Determines if the given user is an owner of this event
     *
     * @param string $user The user to inspect for ownership
     *
     * @return boolean TRUE if the user is an owner of this event, FALSE otherwise
     *
     * @access public
     *
     */
   function isOwner($user)
   {

      // can't just test to see if user is in owner array, since
      // we allow for wildcards in owner names
      //
      foreach ($this->getOwners() as $owner)
         if (preg_match('/^' . str_replace(array('?', '*'), array('\w{1}', '.*?'),
                    strtoupper($owner)) . '$/', strtoupper($user)))
            return TRUE;


      return FALSE;

   }



   /**
     * Determines if the given user has read access to this event
     *
     * NOTE: please be aware that if a username returns FALSE for
     * this function, that user might still have read access if
     * they qualify via isOwner() or canWrite().
     *
     * @param string $user The user to inspect for read permission
     *
     * @return boolean TRUE if the user has read access to this event, FALSE otherwise
     *
     * @access public
     *
     */
   function canRead($user)
   {

      // can't just test to see if user is in readable_users array, since
      // we allow for wildcards in user names
      //
      foreach ($this->getReadableUsers() as $readUser)
         if (preg_match('/^' . str_replace(array('?', '*'), array('\w{1}', '.*?'),
                    strtoupper($readUser)) . '$/', strtoupper($user)))
            return TRUE;


      // also, events on public calendars should all be visible to anyone
      //
      foreach ($this->getParentCalendars() as $parentID)
      {
         $parent = get_calendar($parentID);
         if ($parent->getCalendarType() == SM_CAL_TYPE_PUBLIC)
            return TRUE;
      }


      return FALSE;

   }



   /**
     * Determines if the given user has write access to this event
     *
     * NOTE: please be aware that if a username returns FALSE for
     * this function, that user might still have write access if
     * they qualify via isOwner().
     *
     * @param string $user The user to inspect for write permission
     *
     * @return boolean TRUE if the user has write access to this event, FALSE otherwise
     *
     * @access public
     *
     */
   function canWrite($user)
   {

      // can't just test to see if user is in writeable_users array, since
      // we allow for wildcards in user names
      //
      foreach ($this->getWriteableUsers() as $writeUser)
         if (preg_match('/^' . str_replace(array('?', '*'), array('\w{1}', '.*?'),
                    strtoupper($writeUser)) . '$/', strtoupper($user)))
            return TRUE;


      return FALSE;

   }



   /**              
     * Constructs iCal representation of this event
     *              
     * Note that the returned text ONLY includes event
     * attributes, which, alone, is not, per RFC2445, 
     * a valid iCal object.  The caller should use the
     * output of this function inside of an iCal calendar 
     * data stream.
     *
     * @param boolean $includeExtras When TRUE, all event
     *                               attributes are included
     *                               in the return value, even
     *                               those that are for internal
     *                               use only and should NOT be
     *                               part of output to the
     *                               outside world.  When FALSE,
     *                               only RFC-allowable fields
     *                               are included (optional;
     *                               default = FALSE).
     * @param string $icalLineDelimOverride If given, will be used instead
     *                                      the RFC-standard CRLF to terminate
     *                                      iCal lines (optional)
     *
     * @return string The iCal stream representing this event
     *
     * @access public
     *
     */
   function getICal($includeExtras=FALSE, $icalLineDelimOverride='')
   {

      // figure out line delimiter
      //
      if (empty($icalLineDelimOverride))
         $icalLineDelim = ICAL_LINE_DELIM;
      else
         $icalLineDelim = $icalLineDelimOverride;


      // start building iCal stream
      //
      $iCalText = 'BEGIN:' . $this->getEventType() . $icalLineDelim
                . $this->id->getICal($icalLineDelim)
                . $this->sequence->getICal($icalLineDelim)
                . $this->priority->getICal($icalLineDelim)
                . 'DTSTAMP:' . gmdate('Ymd\THis\Z', $this->thisObjectsTimestamp) . $icalLineDelim
                . $this->createdOn->getICal($icalLineDelim)
                . $this->lastUpdatedOn->getICal($icalLineDelim)
                . $this->status->getICal($icalLineDelim)
                . $this->startDateTime->getICal($icalLineDelim)
//LEFT OFF HERE
//TODO: following three properties should have language and encoding info!
                . $this->summary->getICal($icalLineDelim)
                . $this->description->getICal($icalLineDelim)
                . $this->comments->getICal($icalLineDelim);


      // duration or dtend/due?
      //
      $d = $this->getDuration();
      if (!empty($d))
         $iCalText .= $this->duration->getICal($icalLineDelim);
      else if ($this->isTask())
         $iCalText .= $this->due->getICal($icalLineDelim);
      else
         $iCalText .= $this->endDateTime->getICal($icalLineDelim);


      // only for task/todos
      //
      if ($this->isTask())
         $iCalText .= $this->percentComplete->getICal($icalLineDelim);


      // recurrence info
      //
      $rrule = $this->getRecurrenceRule();
      $rdates = $this->getRecurrenceDates();
      $exrule = $this->getRecurrenceExclusionRule();
      $exdates = $this->getRecurrenceExclusionDates();
      if (!empty($rrule))
         $iCalText .= $this->recurrenceRule->getICal($icalLineDelim);
      if (!empty($exrule))
         $iCalText .= $this->recurrenceExclusionRule->getICal($icalLineDelim);
      if (!empty($rdates))
         $iCalText .= $this->recurrenceDates->getICal($icalLineDelim);
      if (!empty($exdates))
         $iCalText .= $this->recurrenceExclusionDates->getICal($icalLineDelim);


//TODO: possible additions at some point...
// hmm, can we put this together from our internal type of the parent cal?  "CLASS"
//     CLASS:PUBLIC    CLASS:PRIVATE    CLASS:CONFIDENTIAL
// in the future: "ATTENDEE"
// in the future: "CATEGORIES"
// in the future: "CONTACT"
// maybe some day if we have "RELATED-TO" information...
// "TRANSP" if we can figure out the date stuff that requires this prop (anniversaries I think are required to be transparent, but that's regardless of this property anyway), but it's not that important (and/or have a user input for this setting)


      // unknown attributes are included, since they
      // were probably custom attributes defined by an
      // external source
      //
      foreach ($this->unknownAttributes as $attr)
         $iCalText .= $attr . $icalLineDelim;


      // include all our internal attributes?
      //
      if ($includeExtras)
      {
         $iCalText .= $this->dom->getICal($icalLineDelim)
                    . $this->createdBy->getICal($icalLineDelim)
                    . $this->lastUpdatedBy->getICal($icalLineDelim)
                    . $this->owners->getICal($icalLineDelim)
                    . $this->readable_users->getICal($icalLineDelim)
                    . $this->writeable_users->getICal($icalLineDelim)
                    . $this->parentCalendars->getICal($icalLineDelim);
      }


      $iCalText .= 'END:' . $this->getEventType() . $icalLineDelim;


      // fold lines that are too long
      //
      foldICalStreamByRef($iCalText);


      return $iCalText;

   }



   /**
     * Constructs an Event object from the given iCal stream
     *
     * @param array $iCalData The text value to be converted to an
     *                        Event object, one text line in
     *                        each array value.
     *
     * @return object An Event object representing the given iCal stream.
     *
     * @access public
     *
     */
   function getEventFromICal($iCalText)
   {

      // strip out CRLFs from each line
      //
      foreach ($iCalText as $x => $line)
         $iCalText[$x] = str_replace(ICAL_LINE_DELIM, '', $line);


      // unfold text
      //
      unfoldICalStreamByRef($iCalText);


      $id = '';
      $sequence = '';
      $type = '';
      $summary = '';
      $description = '';
      $comments = '';
      $status = '';
      $priority = SM_CAL_EVENT_PRIORITY_NORMAL;
      $createdOn = '';
      $lastUpdatedOn = '';
      $startDateTime = '';
      $endDateTime = '';
      $due = '';
      $duration = '';
      $recurrenceRule = '';
      $recurrenceDates = '';
      $recurrenceExclusionRule = '';
      $recurrenceExclusionDates = '';
      $percentComplete = '';

      $dom = '';
      $parentCalendars = '';
      $owners = '';
      $readable_users = '';
      $writeable_users = '';
      $unknownAttributes = array();
      $createdBy = '';
      $lastUpdatedBy = '';

            
      // pull out properties
      //    
      foreach ($iCalText as $line)
      {        

         $property = Property::extractICalProperty($line);
            
            
         // what do we do with each property?
         //    
         switch ($property->getName())
         {

            // just skip these
            //
            case 'END':
               break;


            // get type -- VTODO or VEVENT?
            //
            case 'BEGIN':
               $type = $property;
               $type->setName('NO-ICAL-TYPE');
               break;


            // event id 
            //
            case 'UID':
               $id = $property;
               break;


            // sequence
            //
            case 'SEQUENCE':
               $sequence = $property;
               break;


            // summary
            //
            case 'SUMMARY':
               $summary = $property;
               break;


            // description
            //
            case 'DESCRIPTION':
               $description = $property;
               break;


            // comments
            //
            case 'COMMENT':
               $comments = $property;
               break;


            // status
            //
            case 'STATUS':
               $status = $property;
               break;


            // priority
            //
            case 'PRIORITY':
               $priority = $property;
               break;


            // date/time stamp
            //
            // at this time, there is no reason we need 
            // to keep this information for ourselves
            //
            case 'DTSTAMP':
               // do nothing
               break;


            // creation date for events created with this application
            // (need to reparse the value as a date)
            //
            case 'CREATED':
               $createdOn = Property::extractICalProperty($line, SM_CAL_ICAL_PROPERTY_TYPE_DATE);
               break;


            // date last modified for events created with this application
            // (need to reparse the value as a date)
            //
            case 'LAST-MODIFIED':
               $lastUpdatedOn = Property::extractICalProperty($line, SM_CAL_ICAL_PROPERTY_TYPE_DATE);
               break;


            // start date
            // (need to reparse the value as a date)
            //
            case 'DTSTART':
               $startDateTime = Property::extractICalProperty($line, SM_CAL_ICAL_PROPERTY_TYPE_DATE);
               break;


            // end date
            // (need to reparse the value as a date)
            //
            case 'DTEND':
               $endDateTime = Property::extractICalProperty($line, SM_CAL_ICAL_PROPERTY_TYPE_DATE);
               break;


            // due
            // (need to reparse the value as a date)
            //
            case 'DUE':
               $due = Property::extractICalProperty($line, SM_CAL_ICAL_PROPERTY_TYPE_DATE);
               break;


            // duration
            //
            case 'DURATION':
               $duration = Property::extractICalProperty($line, SM_CAL_ICAL_PROPERTY_TYPE_DURATION);
               break;


            // recurrence rule
            //
            case 'RRULE':
               $recurrenceRule = Property::extractICalProperty($line, SM_CAL_ICAL_PROPERTY_TYPE_RRULE);
               break;


            // recurrence dates
            // (need to reparse the value as a date (or period))
            //
            case 'RDATE':
               $recurrenceDates = Property::extractICalProperty($line, SM_CAL_ICAL_PROPERTY_TYPE_DATE);
               break;


            // recurrence exclusion rule
            //
            case 'EXRULE':
               $recurrenceExclusionRule = Property::extractICalProperty($line, SM_CAL_ICAL_PROPERTY_TYPE_RRULE);
               break;


            // recurrence exclusion dates
            // (need to reparse the value as a date)
            //
            case 'EXDATE':
               $recurrenceExclusionDates = Property::extractICalProperty($line, SM_CAL_ICAL_PROPERTY_TYPE_DATE);
               break;


            // percent complete
            //
            case 'PERCENT-COMPLETE':
               $percentComplete = $property;
               break;



            // domain for events created with this application
            //
            case 'X-SQ-EVTDOMAIN':
               $dom = $property;
               break;


            // parent calendars for events created with this application
            //
            case 'X-SQ-EVTPARENTCALENDARS':
               $parentCalendars = $property;
               break;


            // event owners for events created with this application
            //
            case 'X-SQ-EVTOWNERS':
               $owners = $property;
               break;


            // event readable users for events created with this application
            //
            case 'X-SQ-EVTREADABLEUSERS':
               $readable_users = $property;
               break;


            // event writeable users for events created with this application
            //
            case 'X-SQ-EVTWRITEABLEUSERS':
               $writeable_users = $property;
               break;


            // user who created this event for events created with this application
            //
            case 'X-SQ-EVTCREATOR':
               $createdBy = $property;
               break;


            // user who last modified this event for events created with this application
            //
            case 'X-SQ-EVTLASTUPDATOR':
               $lastUpdatedBy = $property;
               break;


            // unknown parameters just pile into this array
            //
            default:
               $unknownAttributes[$property->getName()] = $line;
               break;

         }

      }


      return new Event($id, $sequence, $dom, $type, $summary, $description,
                       $comments, $status, $priority, $startDateTime, $endDateTime,
                       $due, $duration, $recurrenceRule, $recurrenceDates,
                       $recurrenceExclusionRule, $recurrenceExclusionDates,
                       $percentComplete, $parentCalendars, $createdBy, $createdOn, 
                       $lastUpdatedBy, $lastUpdatedOn, $owners, $readable_users, 
                       $writeable_users, $unknownAttributes);


   }



}



?>
