/*  NAME:
 * 
 *	filtercmd -- installs/reads filter files (.procmailrc, etc) for squirrelmail
 *
 *  SYNOPSYS:
 *
 * 	filtercmd getrc rctype > temprcfile < credentials
 *	filtercmd putrc rctype < credentials < temprcfile
 *	filtercmd rcexists rctype < credentials
 *  
 *  DESCRIPTION:
 *	
 *	The filtercmd program allows an unpriviledged program
 *	(squirrelmail) to work with users' mail filtering configuration
 *	files. The format of the mail filtering configuration files is not
 *	parsed in any way. This program only handles reading and writing 
 *	the configuration files. These general steps are followed by filtercmd:
 *		- read in IMAP credentials.
 *		- verify credentials (log in to IMAP server).
 *		- determine file location of rc file.
 *		- act upon rc file as directed by command (first argument)
 *
 *	The getrc command will copy the rc file to standard output.
 *
 *	The putrc command will read an rc file from standard input, after 
 *	reading the credentials, and replace the existing rcfile.
 *
 *	The rcexists command will return a zero exit status if the rc file exists,
 *	non-zero otherwise.
 *
 *  CREDENTIALS
 *
 *	Credentials are passed to standard input. The username and
 *	password should be terminated with newline (\n) characters. No other 
 *	characters should be passed.
 *	
 *	The credentials are used to log in to an IMAP server. The server host 
 *	name and port is read from the squirrelmail configuration file.
 *	
 * 	Authentication is performed using the c-client software from UW.
 *	See checkcreds_cclient.c for more details.
 *	
 *  RC FILE LOCATIONS
 *
 *	RC file locations are calculated from settings in the serversidefilter 
 *	configuration file. The settings can contain the following markers which 
 *	are be replaced with values calculated from the username.
 *		[USERNAME] 
 *			text in username before an @ character.
 *		
 *		[DOMAIN]
 *			text in username after an @ character.
 *
 *		[DOMUSER] 
 *			virtualdomains file, with lines in format domain:user, 
 *			is consulted. The value is the "user" matching the domain 
 *			as described above.
 *
 *	If there is no @ character in the username, [USERNAME] will be the entire 
 *	text of the username, [DOMAIN] will be blank, [DOMUSER] will be blank and the 
 *	virtualdomains file will not be consulted. If the domain is not found in the 
 *	virtualdomains file then [DOMUSER] will be blank.
 *	
 *	More than one type of RC file can be handled by this script. The RC file 
 *	types are:
 *		forward	(.qmail file, or .forward file)
 *		filter (.procmailrc, or .mailfilter)
 *
 *  The settings in the configuration file that control the locations are:
 *		$FORWARD_FILE_PATH (for "foward" rc type)
 *		$FILTER_FILE_PATH (for "filter" rc type)
 *
 *  RC FILE OWNERSHIP
 *	
 *	The RC files will be owned by the user named by [DOMUSER], or [USERNAME] if the 
 *	[DOMUSER] is blank.
 *
 *  CONFIGURATION FILES
 *
 *	filtercmd reads three different configuration files:
 *		squirrelmail/config.php
 *		serversidefilter/config.php 
 *		serversidefilter/virtualdomains
 *
 *	Each of the configuration files must be owned by root.
 *
 *  DIAGNOSTICS
 *
 *	Setting an environment variable FILTERCMDDEBUG to 1 will 
 *	cause debugging output to be printed.
 *
 *	filtercmd will exit with a non-zero status if an error occurs.
 *	filtercmd will write text to standard error in the case of
 *	some errors.  See filtercmd.h for details.
 *	
 */
 
#define STR_MAX 1024
#define MAXLEN 1024
#define VERSION "filtercmd 1.41, part of serversidefilter for squirrelmail"

/* define this in filtercmd.opts */
#ifndef SQUIRRELMAILCONFIGFILE
#define SQUIRRELMAILCONFIGFILE "/etc/squirrelmail/config.php"
#endif
#ifndef SERVERSIDEFILTERHOME
#define SERVERSIDEFILTERHOME "../plugins/serversidefilter/"
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <pwd.h>
#include "filtercmd.h"

/* externally defined functions and variables */
extern char *checkcreds_extra_configvar;
int  checkcredentials(char*, int, char*, char*, char*);

/* global variables -- used in path calculation code (findfilterfile) */
char separator[2] = "@";   /* the separator character */
char user[STR_MAX];        /* IMAP username possibly including domain name */
char real_user[STR_MAX];   /* real unix user */
char dom_user[STR_MAX];    /* IMAP username, sans any domain name */
char *domain;		   /* The domain of the IMAP username */

/* function prototypes */
void eperror(register char *s, int exitcode);
void setdebug(int *argc, char ***argv);
void printversion();

int main(int argc, char *argv[]){
	/*
	Developed by
	Pedro L Orso - orso@onda.com.br
	Changed by
	Thiago Melo de Paula - thiago@fafibe.br
	*/
	int err, ok;
	char cmd[STR_MAX];
	char passwd[STR_MAX];
	char filter_file[STR_MAX];
	char temp_file[STR_MAX];
	char imap_server[STR_MAX];
	char checkcreds_extra[STR_MAX] = "";
	int imap_port;
	uid_t UID;
	gid_t GID;
	
	setdebug(&argc, &argv);
	
	/* look for the "version" command early */
	if (argc > 1 && !strcmp("version", argv[1])) {
		printversion();
		return ERR_OK;
	}
	
	err = readcredentials(user, passwd, STR_MAX);
 	if (err) return inerror(err);

 	err = readimapserver(imap_server, &imap_port, STR_MAX);
 	if (err) return inerror(err);

	if (debugon()) {
		fprintf(stderr, "imap user validated: %s\n", user);
	}
	
	if (splitdomain(user, &domain, dom_user, STR_MAX)) {
 		err = readvirtualdomain(domain, real_user, STR_MAX);
 		if (err) return inerror(err);
 	} else {
 		/* without domain mapping file, assume IMAP username is a real unix username */
 		strncpy(real_user, user, STR_MAX);
	}
	
	if (debugon()) {
		fprintf(stderr, "domain: %s\n", domain);
		fprintf(stderr, "dom_user: %s\n", dom_user);
	}
	
 	err = finduidgid(real_user, &UID, &GID);
 	if (err) return inerror(err);

	if((setgid(GID)) < 0) eperror("setgid", ERR_NOT_SUID);
	if((setuid(UID)) < 0) eperror("setuid", ERR_NOT_SUID);

	if (checkcreds_extra_configvar) {
		readconfigvar(SERVERSIDEFILTERHOME "/config.php", checkcreds_extra_configvar, 
			checkcreds_extra, STR_MAX);
	}

	err = checkcredentials(imap_server, imap_port, user, passwd, checkcreds_extra);
	if (err) return inerror(err);

	strncpy(cmd,argv[1],STR_MAX);

	if(!strlen(cmd)){
		return inerror(ERR_NO_CMD);
	} else {
		if (!strcmp("getrc",cmd)) { 
				if (argc < 3) {
					printf("Usage: filtercmd getrc filter_file\n");
					return ERR_USAGE;
				}

				err = findfilterfile(argv[2], filter_file, STR_MAX);
				if (err) return inerror(err);

				return getrc(filter_file);
		} else if (!strcmp("putrc", cmd)) {
                                if (argc < 3) {
                                        printf("Usage: filtercmd putrc filter_file\n");
                                        return ERR_USAGE;
                                }

				err = findfilterfile(argv[2], filter_file, STR_MAX);
				if (err) return inerror(err);

				return putrc(filter_file, UID, GID);
		} else if (!strcmp("rcexists",cmd)) {
                                if (argc < 3) {
                                        printf("Usage: filtercmd rcexists filter_file\n");
                                        return ERR_USAGE;
                                }
				err = findfilterfile(argv[2], filter_file, STR_MAX);
				if (err) return inerror(err);

				return rcexists(filter_file);
		} else if (argc > 1) {
				return inerror(ERR_INVALID_COMMAND);
		} else {
				printf("Usage: filtercmd [getrc|putrc|rcexists]\n");
				return ERR_USAGE;
		}
	}

}

void eperror(s, exitcode)
register char *s;
int exitcode;
{
        /*
        Developed by
        Pedro L Orso - orso@onda.com.br
        Changed by
        Thiago Melo de Paula - thiago@fafibe.br
        */
   char str[STR_MAX];

   snprintf(str,STR_MAX,"filtercmd - %s",s);
   perror(str);
   exit(exitcode);
}

int inerror(errnum)
int errnum;
{
	fprintf(stderr, "%s", err_strings[errnum]);
	return errnum;	
}

static int _debuglevel = 0;
int debugon() {
	return _debuglevel;
}

/*
 * Figure out from the environment/arguments what the current debug level
 * should be. Currently this only reads the environment. Pointers to argc and
 * argv are provided, so that this routine can (in the future) remove any 
 * debug-related arguments.
 *
 * Parameters:
 *   argc - IN/OUT.
 *   argv - IN/OUT. 
 */
void setdebug(int *argc, char ***argv) {
	char *FILTERCMDDEBUG;
	FILTERCMDDEBUG = getenv("FILTERCMDDEBUG");
	if (FILTERCMDDEBUG) {
		sscanf(FILTERCMDDEBUG, "%d", &_debuglevel);
	}
}

void printversion() {
	puts(	VERSION "\n"
		"\n"
		"Compile-time definitions:\n"
		"\tSQUIRRELMAILCONFIGFILE: " SQUIRRELMAILCONFIGFILE "\n"
		"\tSERVERSIDEFILTERHOME: " SERVERSIDEFILTERHOME "\n"
		);
}

/* 
 * Read credentials from stdin; user and passwd should each be on a line
 * user may not include spaces. FIXME: password may not include
 * spaces either, currently.
 *
 * Parameters:
 *   user - OUT. The buffer is filled with username read from stdin.
 *   passwd - OUT. The buffer is filled with password read from stdin.
 *   buflen - IN. The buffer size for both buffers.
 * Returns:
 *   ERR_OK if the credentials were read successfully, non-zero error code 
 *   if not.
 */
int readcredentials(user, passwd, buflen)
char *user;
char *passwd;
int buflen;
{
	int n;
	if (fgets(user, buflen, stdin) == NULL) 
		return ERR_NEED_CREDENTIALS;
	n = strlen(user);
	if (n < 2)
		return ERR_NEED_CREDENTIALS;
	if (user[n-1] == '\n') 
		user[n-1] = '\0';	/* remove \n */

	if (fgets(passwd, buflen, stdin) == NULL) 
		return ERR_NEED_CREDENTIALS;
	n = strlen(passwd);
	if (n < 2)
		return ERR_NEED_CREDENTIALS;
	if (passwd[n-1] == '\n')
		passwd[n-1] = '\0';	/* remove \n */

	return ERR_OK;
}

/*
 * Reads IMAP server information from SQUIRRELMAILCONFIGFILE. Also 
 * reads the port. The routine is not fussy about finding a port 
 * specification in the file -- it will just set a useful default if 
 * there are issues.
 *
 * Parameters:
 *   server - OUT. the servername read from config.php
 *   serverlen - IN. buffer size of the server buffer.
 * Returns:
 *   ERR_OK if it found the necessary parameters, an error code if not.
 */
 
int readimapserver(char *server, int *port, int serverlen) 
{
	int foundserver;
	char port_str[STR_MAX];
	
	*port = 143; /* a useful default */
	if (readconfigvar(SQUIRRELMAILCONFIGFILE, "$imapPort", port_str, STR_MAX)) {
		sscanf(port_str, "%d", port);
	}
	
	foundserver = readconfigvar(SQUIRRELMAILCONFIGFILE, "$imapServerAddress", server, serverlen);
	return foundserver ? ERR_OK : ERR_CANT_READ_IMAP_SERVER;
}

/*
 * Reads a setting from a config file.
 *
 * Parameters:
 *   phpvar - IN. the name of the PHP variable in config.php to parse out.
 *   outbuf - OUT. the setting as read from config.php
 *   outbuflen - IN. buffer size of the outbuf buffer.
 * Returns:
 *   1 if it the parameter, 0 if not.
 */
 
int readconfigvar(configfile, phpvar, outbuf, outbuflen) 
char *phpvar;
char *outbuf;
int outbuflen;
{
	FILE *f;
	char line[STR_MAX];
	int err, found = 0;

	err = trusted_open(&f, configfile, "r");
	if (err) return err;
	while (fgets(line, STR_MAX, f)) {
		found = parsephpstring(phpvar, line, outbuf, outbuflen);
		if (found) break;
	}
	if (debugon()) {
		fprintf(stderr, "config: %s %s -> %d\n", configfile, phpvar, found);
	}
	return found;
}

/*
 * Determines uid and gid given a "real" username. A UID or 
 * GID of 0 is considered invalid. 
 *
 * Parameters:
 *   user - IN. A UNIX username.
 *   uid - OUT. The UID of the UNIX username.
 *   gid - OUT. The GID of the UNIX username.
 * Returns:
 *   ERR_OK if it found a valid uid/gid, an error code if not
 */

int finduidgid(user, uid, gid)
char *user;
uid_t *uid;
gid_t *gid;
{
	struct passwd *p;

	p = getpwnam(user);
	if (!p) return ERR_BAD_UID_GID;
	if (p->pw_uid == 0 || p->pw_gid == 0) {
		/* refuse to allow root's uid/gid to be used */
		return ERR_USER_IS_ROOT;
	}
	*uid = p->pw_uid;
	*gid = p->pw_gid;
	return ERR_OK;
}

/*
 * Looks for a given varname in a line of PHP code; if it is found, the code
 * will try to parse out the number or string between quotes. This function
 * is easily confused and when it returns 1 it means the variable was found,
 * not necessarily that the string contents were successfully parsed.
 *
 * Parameters:
 *   varname - IN. The PHP variable to search for. SHould include $
 *   line - IN. A line of text. May end with \n. 
 *   dest - OUT. Where the string contents will be placed if there is a match.
 *   destlen - IN. Size of the dest buffer.
 *
 * Returns:
 *   1 if the varname was found, 0 if not.
 */

int parsephpstring(varname, line, dest, destlen)
char *varname;
char *line;
char *dest;
int destlen;
{
	char *s;
	char *commentpos;
	int n;

	s = strstr(line, varname);       
	if (s == NULL) return 0;
	
	/* look for // comment marker, making sure the // can't be 
	   treated as a comment by the  compiler */
	commentpos = strstr(line, "/" "/");
	if (commentpos && commentpos < s) return 0;

	/* skip ahead to either ' or " or a number */
	s += strcspn(s, "\"'0123456789\0\n");
	if (strchr("0123456789", *s)) {
	  /* found number - now find semicolon, space or EOL */
	  n = strcspn(s, "; \0\n");
	} else {
	  /* found quote - now find next quote or EOL */
	  /* this does not find matching quotes */
	  s++;
	  n = strcspn(s, "\"'\0\n");
	}
	if (n > destlen) n = destlen;
	strncpy(dest, s, n);
	dest[n] = '\0';

	return 1;
}

int rcexists(char* filter_file)
{
	FILE *filter;

	if(!(filter=fopen(filter_file,"r"))){
		return ERR_NO_RC_FILE;  /* silent */
	} else {
		fclose(filter);
		return 0;
	}
}


int getrc(char* filter_file)
{
	int result;

	result = copy_file(filter_file, NULL, NULL, stdout);
	return result;
}

int putrc(char* filter_file, uid_t uid, gid_t gid)
{
	int result;

	result = copy_file(NULL, stdin, filter_file, NULL);
	if (result) {
	   return result;
	}
        chown(filter_file, uid, gid);
        chmod(filter_file, (S_IRUSR|S_IWUSR));
        return 0;
}

int copy_file(char* old_file, FILE* infile, char* new_file, FILE* outfile)
{
	char buf[MAXLEN];

	if (old_file) {
		if (!(infile=fopen(old_file,"r"))) {
			fprintf(stderr, "Could not open %s\n", old_file);
			return ERR_COPY_CANT_OPEN_SRC;
		}
	}
	if (new_file) {
		if (!(outfile=fopen(new_file,"w"))) {
			fprintf(stderr, "Could not open %s\n", new_file);
			return ERR_COPY_CANT_OPEN_DEST;
		}
	}

	while(fgets(buf,MAXLEN,infile)!=NULL){
		fputs(buf, outfile);
	}
	if (old_file) fclose(outfile);
	if (new_file) fclose(infile);
	return 0;
}

/* Path calculation code. 
 *
 * 
 * Parameters:
 *   filter_type - IN. One of "filter", "forward".
 *   filter_file - OUT. The resulting calculated path.
 *   buflen - IN. Size of the filter_file buffer.
 *
 * Returns:
 *   ERR_OK if the file was calculated, an error message if not.
 */
int findfilterfile(char *filter_type, char *filter_file, int buflen) {
	int err;

	err = findfilterpath(filter_type, filter_file, buflen);
	if (err) return err;

        #ifdef DEBUG
	fprintf(stderr, "filter_file: %s\n", filter_file);
	fprintf(stderr, "dom_user: %s\n", dom_user);
	fprintf(stderr, "domain: %s\n", domain);
	fprintf(stderr, "real_user: %s\n", real_user);
	#endif
	
	replace(filter_file, buflen, "[USERNAME]", dom_user);
	replace(filter_file, buflen, "[DOMAIN]", domain);
	replace(filter_file, buflen, "[DOMUSER]", real_user);
	
        #ifdef DEBUG
	fprintf(stderr, "filter_file: %s\n", filter_file);
	#endif
	
	return ERR_OK;
}

int findfilterpath(char* filter_type, char* filter_file, int buflen) {
	char* phpvar;

	if (0 == strcmp(filter_type, "filter")) {
		phpvar = "$FILTER_FILE_PATH";
	} else if (0 == strcmp(filter_type, "forward")) {
		phpvar = "$FORWARD_FILE_PATH";
	} else {
		return ERR_BAD_RC_FILE_TYPE;
	}

	#ifdef DEBUG
	fprintf(stderr, "phpvar: %s\n", phpvar);
	#endif
	return readconfigvar(SERVERSIDEFILTERHOME "/config.php", phpvar, filter_file, buflen) 
	   ? ERR_OK :  ERR_NO_RC_FILE_PATH;
}

/* Alter buf so that any instances of old are replaced by new 
 *
 * Returns count of instances replaced.
 *
 * This is not a very good replace routine -- should not start from beginning 
 * each time, and should not recalculate so many string lengths.
 */
int replace(char* buf, int buflen, char* old, char* new) {
	int oldlen = strlen(old);
	int newlen = strlen(new);
	int bufstrlen = strlen(buf);
	char* bufend = buf + bufstrlen;

	/* if negative, the string in buf gets shorter with each replacement */
	/* if positive, the string in buf gets longer with each replacement */
	int lendiff = newlen - oldlen;	
	char *oldpos = strstr(buf, old);

	if (oldlen == 0) 
		/* bad parameter - cannot search for empty substring */
		return 0;

	if (oldpos == NULL) 
		/* substring not found */
		return 0;
	if ((bufstrlen + lendiff) > buflen) {
		/* buffer not big enough -- bail out */
		#ifdef DEBUG
		fprintf(stderr, "replace: buffer not big enough -- bailing out\n");
		#endif		
		
		return 0;
	}

	if (lendiff > 0) {
		memmove(oldpos+lendiff, oldpos, bufend-oldpos+1);
	}
	strncpy(oldpos, new, newlen);
	if (lendiff < 0) {
		memmove(oldpos+newlen, oldpos+oldlen, bufend-(oldpos+oldlen)+1);
	}
	return 1 + replace(buf, buflen, old, new);
}


/* 
 * Handles splitting a user into a username and a domain.
 *
 * Globals:
 *   separator - IN. This should be a one-character string, usually "@"
 * Parameters:
 *   user - IN. This can possibly contain the separator character.
 *   domain - OUT. This pointer will be set to part of the dom_user buffer.
 *   dom_user - OUT. This buffer will be set to the first part of the user.
 *   buflen - IN. Length of the dom_user buffer.
 * Returns:
 *   1 if the user contained the separator character, 0 if not.
 */
int splitdomain(char *user, char** domain, char* dom_user, int buflen) {
	char *dom;
	strncpy(dom_user, user, buflen);
	dom = strchr(dom_user, separator[0]);	
	#ifdef DEBUG
	fprintf(stderr, "user:%s dom_user:%s dom:%s buflen:%d\n", user, dom_user, dom, buflen);		
	#endif
	if (dom == NULL) {
		/* no at sign in username implies no virtual domain. */
		/* make domain be the empty string */
		dom = dom_user + strlen(dom_user);
		*domain = dom;
		return 0;
	} else {
		/* terminate the dom_user string at the separator and 
		   make domain point to the rest of the string */
		#ifdef DEBUG
		fprintf(stderr, "user:%s dom_user:%s dom:%s\n", user, dom_user, dom);		
		#endif
		
		*dom = '\0';
		dom++;
		*domain = dom;
		return 1;
	}
}	

/* 
 * Reads the virtualdomain file to find the real unix user name for a domain.
 *
 * Parameters:
 *   domain - IN. The domain name.
 *   real_user - OUT. This buffer will be set to the real unix username.
 *   buflen - IN. Length of the real_user buffer.
 * Returns:
 *   ERR_OK if an entry was found, an error message if not.
 */
int readvirtualdomain(char *domain, char *real_user, int buflen) {
	FILE *f;
	char line[STR_MAX];
	char *colon, *s;
	int err;
	
	/* find the first line with a colon that matches the domain up to the colon.
	 * copy the rest of the line (or up to another colon) into the real_user 
	 * buffer.
	 */
	err = trusted_open(&f, SERVERSIDEFILTERHOME "virtualdomains", "r");
	if (err) return err;
	s = NULL; /* if s is set we know the domain was found */	

	while (fgets(line, STR_MAX, f)) {
		colon = strchr(line, ':');
		if (!colon) continue;			
		*colon = '\0';
		if (strcmp(domain, line)) continue;
		s = colon + 1;
		colon = strpbrk(s, ":\n");
		if (colon) *colon = '\0';
		strncpy(real_user, s, buflen);
		break;
	}
	fclose(f);	
	if (s) return ERR_OK;
	
	return ERR_CANT_FIND_VIRTUAL_DOMAIN;
}

/* Open a file while verifying that it is sufficiently protected so that
 * we can trust its contents. That means the file must be owned by root,
 * and either not group readable or be in root's group. The same test is 
 * applied to all the files, all the way up the tree.
 */
int trusted_open(FILE** f, char *filepath, char *mode) {
	char curpath[STR_MAX];
	int err;

	strncpy(curpath, filepath, STR_MAX);
	err = trusted_check(filepath, curpath);
	if (err) return err;
	*f = fopen(filepath, mode);
	return ERR_OK;
}

int trusted_check(char *filepath, char *curpath) {
	struct stat buf;
	char *ix = NULL;
  	      
	do {
		if (stat(filepath, &buf)) 
			eperror(filepath, ERR_CANT_TRUST_FILE);
	        if (buf.st_mode & S_IWOTH)
        		return trusted_error("writable by others", curpath, filepath);
	        if ((buf.st_mode & S_IWGRP) && (buf.st_gid != 0))
        		return trusted_error("writable by non-root group", curpath, filepath);
		if ((buf.st_mode & S_IWUSR) && (buf.st_uid != 0))
			return trusted_error("writable by non-root user", curpath, filepath);
	
		/* find parent directory */
		ix = rindex(curpath, '/');
		if (!ix) return ERR_OK;
		*ix = '\0';
	} while (1);
}

int trusted_error(char *err, char *path, char *file) {
	//#ifdef DEBUG
	fprintf(stderr, "trust error on %s while checking %s\n  %s\n", path, file, err);
	//#endif
	return ERR_CANT_TRUST_FILE;
}
