<?php

/**
  * SquirrelMail Server Settings Backend Plugin
  * Copyright (c) 2008-2010 Paul Lesniewski <paul@squirrelmail.org>
  * Licensed under the GNU GPL. For full terms see the file COPYING.
  *
  * @package plugins
  * @subpackage server_settings_backend
  *
  */



/**
  * Retrieves a setting from the server using the local file backend.
  *
  * If some fatal error occurs due to misconfiguration or
  * backend failure, an error message is printed and execution
  * stops immediately, unless $quiet is TRUE.
  *
  * If the requested setting consists of multiple values,
  * every effort is made to return them in an array, but
  * specifying the correct value of "MULTIPLE" in the
  * storage configuration settings helps ensure this will
  * work correctly.  If "MULTIPLE" is not found, default
  * behavior is to return multiple values in an array.
  *
  * @param array   $backend_info An array of configuration values
  *                              that point to how the setting value
  *                              should be retrieved.
  *                              This value is assumed to have already
  *                              been error-checked to an extent: it
  *                              is assumed to be an array, and all
  *                              array keys are in all capital letters,
  *                              and if a VALUE or CUSTOM entry is
  *                              present, it has already been dealt
  *                              with.  Same as for multiple storage
  *                              arrays - this array should be non-
  *                              nested.
  * @param boolean $quiet        When TRUE, suppresses any error
  *                              messages and makes this function
  *                              return NULL without doing anything
  *                              else (OPTIONAL; default = FALSE).
  * @param mixed $test_value     When given, this indicates that the
  *                              setting is being retrieved
  *                              specifically for testing per the
  *                              test_server_setting() function (or
  *                              similar), and so if there is a
  *                              different (more efficient) kind of
  *                              lookup for this action, it can be
  *                              enabled.  When given, its value must
  *                              be the value being tested for
  *                              (OPTIONAL; default = NULL).
  * @param boolean $return_file_contents When this parameter is TRUE,
  *                                      the normal return value is
  *                                      placed inside an array to
  *                                      which the file contents are
  *                                      added as the second and last
  *                                      array element, and this two-
  *                                      lement array is then returned.
  *                                      (OPTIONAL; default = FALSE).
  *
  * @return mixed The setting value if it was retrieved normally
  *               or NULL if it was not (and $quiet is TRUE) (or
  *               a two-element array containing the setting value
  *               or NULL when errors ocurred and the file contents
  *               if $return_file_contents is TRUE).
  *
  */
function retrieve_server_setting_local_file($backend_info, $quiet=FALSE,
                                            $test_value=NULL, $return_file_contents=FALSE)
{

   global $username, $domain;
   include_once(SM_PATH . 'plugins/server_settings_backend/local_file_functions.php');
   $error = '';
   $original_text_domain = sq_change_text_domain('server_settings_backend');


   // if this setting consists of multiple values,
   // how will we handle them?
   //
   if (!empty($backend_info['MULTIPLE']))
      $multiple = $backend_info['MULTIPLE'];
   else
      $multiple = 'MULTIPLE';  // note that this seems incorret for scalars, but
                               // MULTIPLE returns the value as is (an array), which
                               // is also exactly what we want to do with scalars


   // get file access type, file, and regular expression
   //
   if (!empty($backend_info['PATTERN_GROUP_NUMBER']) && is_array($backend_info['PATTERN_GROUP_NUMBER']))
      $pattern_group_number = retrieve_server_setting($backend_info['PATTERN_GROUP_NUMBER'], $quiet);
   else $pattern_group_number = 0;
   if (!empty($backend_info['TREAT_AS_EMPTY_WHEN_NOT_FOUND']) && is_array($backend_info['TREAT_AS_EMPTY_WHEN_NOT_FOUND']))
      $treat_as_empty_when_not_found = retrieve_server_setting($backend_info['TREAT_AS_EMPTY_WHEN_NOT_FOUND'], $quiet);
   else $treat_as_empty_when_not_found = 0;
   if (!empty($backend_info['ACCESS_TYPE']) && is_array($backend_info['ACCESS_TYPE']))
      $access_type = retrieve_server_setting($backend_info['ACCESS_TYPE'], $quiet);
   else
      $error = _("Misconfiguration encountered when attempting to retrieve a setting from the server (file access method is misconfigured)");
   if (!empty($backend_info['FILE']) && is_array($backend_info['FILE']))
      $remote_file = retrieve_server_setting($backend_info['FILE'], $quiet);
   else
      $error = _("Misconfiguration encountered when attempting to retrieve a setting from the server (file name is misconfigured)");
   if (!empty($backend_info['PARSE_PATTERN']) && is_array($backend_info['PARSE_PATTERN']))
      $parse_pattern = retrieve_server_setting($backend_info['PARSE_PATTERN'], $quiet);
   else
      $error = _("Misconfiguration encountered when attempting to retrieve a setting from the server (parsing pattern is misconfigured)");
   if (!empty($backend_info['SUID_LOCATION']) && is_array($backend_info['SUID_LOCATION']))
      $suid_binary = retrieve_server_setting($backend_info['SUID_LOCATION'], $quiet);
   else if ($access_type == 'SUID')
      $error = _("Misconfiguration encountered when attempting to retrieve a setting from the server (SUID_LOCATION is misconfigured)");
   else $suid_binary = '';
   if (!empty($backend_info['SUID_DEBUG_OUTPUT']) && is_array($backend_info['SUID_DEBUG_OUTPUT']))
      $suid_debug_file = retrieve_server_setting($backend_info['SUID_DEBUG_OUTPUT'], $quiet);
   else
      $suid_debug_file = '';
   if (!empty($backend_info['MASTER_SECURITY_CHECK']) && is_array($backend_info['MASTER_SECURITY_CHECK']))
      $master_security_check = retrieve_server_setting($backend_info['MASTER_SECURITY_CHECK'], $quiet);
   else
      $master_security_check = FALSE;
   if (!empty($backend_info['MASTER_USERNAME']) && is_array($backend_info['MASTER_USERNAME']))
      $master_username = retrieve_server_setting($backend_info['MASTER_USERNAME'], $quiet);
   else
      $master_username = NULL;


   // if using "master" security check, grab needed values from environment
   //
   if ($master_security_check)
   {
      if (!sqGetGlobalVar('SQUIRRELMAIL_MASTER_PASSWORD', $master_password, SQ_SERVER)
       || $master_password == 'test-password')
         $error = _("Misconfiguration encountered when attempting to retrieve a setting from the server (master password is misconfigured)");
      if ($master_username
       && (!sqGetGlobalVar('SQUIRRELMAIL_MASTER_USERNAME', $master_username, SQ_SERVER)
        || $master_username == 'test-username'))
         $error = _("Misconfiguration encountered when attempting to retrieve a setting from the server (master username is misconfigured)");
   }
   else
      $master_username = $master_password = NULL;


   // manipulate username, domain and password if needed
   //
   list($user, $dom, $pass) = get_credentials($backend_info,
                                              $username,
                                              $domain,
                                              sqauth_read_password(),
                                              $quiet);


   // when all three are null, an error occured and $quiet
   // is turned on, so we can just fake an error
   //
   if ($quiet && is_null($user) && is_null($dom) && is_null($pass))
      $error = 'dummy';


   // validate we have what we need
   //
   if (empty($error)
    && (empty($access_type)
    || empty($remote_file)
    || empty($parse_pattern)
    || ($access_type == 'SUID' && empty($suid_binary))))
      $error = _("Misconfiguration encountered when attempting to retrieve a setting from the server (missing file name, access type, SUID binary, or parsing pattern)");
   else if (empty($error))
   {

      // get our value!
      //
      $remote_file = str_replace(array('%1', '%2'), array(ssb_str_esc($user), ssb_str_esc($dom)), $remote_file);
      $ret = retrieve_server_setting_from_local_file($access_type, $remote_file,
                                                     $user, $pass,
                                                     $parse_pattern,
                                                     $pattern_group_number,
                                                     $treat_as_empty_when_not_found,
                                                     $suid_binary,
                                                     $suid_debug_file,
                                                     $master_username,
                                                     $master_password,
                                                     $quiet, $return_file_contents);


      // pull off file contents if caller asked for them
      //
      if ($return_file_contents)
         list($ret, $file_contents) = $ret;


      if (!is_null($ret))  // don't do anything if no value was found
      {

         // strip out offset values, which are not needed here
         //
         if (!is_array($ret[0]))   // scalar result
            $ret = $ret[0];
         else
         {  // array result
            $new_ret = array();
            foreach ($ret as $value_and_offset)
               $new_ret[] = $value_and_offset[0];
            $ret = $new_ret;
         }


         if ($multiple == 'SERIALIZE')
         {
//TODO: should we use sq_call_function_suppress_errors() here?  NOTE that if the field is too short for having saved the serialized value, this will throw a PHP notice (because it won't be correctly formatted for unserialization) and truncate the value
            $ret = unserialize($ret);
         }
         else if ($multiple != 'MULTIPLE')
            $ret = explode($multiple, $ret);
      }
      sq_change_text_domain($original_text_domain);
      if ($return_file_contents)
         $ret = array($ret, $file_contents);
      return $ret;

   }


   // should not get here without an error message 
   // unless something is very wrong
   //
   if (empty($error))
      $error = _("Unknown error in Server Settings Backend plugin, function retrieve_server_setting()");


   // spit out error and exit
   //
   if ($quiet)
   {
      sq_change_text_domain($original_text_domain);
      return NULL;
   }
   sq_change_text_domain('squirrelmail');  // NOT $original_text_domain
   global $color;
   $ret = plain_error_message($error, $color);
   if (check_sm_version (1, 5, 2))
   {
      echo $ret;
      global $oTemplate;
      $oTemplate->display('footer.tpl');
   }
   exit;

}



/**
  * Stores a setting on the server using the local file backend.
  *
  * If some fatal error occurs due to misconfiguration or
  * backend failure, an error message is printed and execution
  * stops immediately, unless $quiet is TRUE.
  *
  * @param mixed   $new_value    The value to be stored.
  * @param array   $backend_info An array of configuration values
  *                              that point to how the setting value
  *                              should be stored.
  *                              This value is assumed to have already
  *                              been error-checked to an extent: it
  *                              is assumed to be an array, and all
  *                              array keys are in all capital letters,
  *                              and if a VALUE or CUSTOM entry is
  *                              present, it has already been dealt
  *                              with.  Same as for multiple storage
  *                              arrays - this array should be non-
  *                              nested.
  * @param boolean $add          When TRUE, and if the setting being
  *                              updated consists of multiple individual
  *                              values, $new_value is added to them
  *                              and the other values are left as is.
  *                              If the setting's configuration does
  *                              not include a "MULTIPLE" item, then 
  *                              this flag has no effect and the setting
  *                              is updated as if it were turned off
  *                              (OPTIONAL; default = FALSE).
  * @param boolean $remove       When TRUE, and if the setting being
  *                              updated consists of multiple individual
  *                              values, $new_value is removed from
  *                              those values (if found within) and the
  *                              other values are left as is.  If the
  *                              setting's configuration does not include
  *                              a "MULTIPLE" item, then this flag has no
  *                              effect and the setting is updated as if
  *                              it were turned off (OPTIONAL; default
  *                              = FALSE).
  * @param boolean $quiet        When TRUE, suppresses any error
  *                              messages and makes this function
  *                              return NULL without doing anything
  *                              else (OPTIONAL; default = FALSE).
  *
  * @return mixed TRUE if the setting was stored normally or
  *               NULL if it was not (and $quiet is TRUE).
  *
  */
function put_server_setting_local_file($new_value, $backend_info,
                                       $add=FALSE, $remove=FALSE, $quiet=FALSE)
{

   global $username, $domain;
   include_once(SM_PATH . 'plugins/server_settings_backend/local_file_functions.php');
   $error = '';
   $original_text_domain = sq_change_text_domain('server_settings_backend');


   // if this setting consists of multiple values,
   // how will we handle them?
   //
   $original_new_value = $new_value;
   $multiple_save = FALSE;
   if (is_array($new_value))
   {
      if (!empty($backend_info['MULTIPLE']) && $backend_info['MULTIPLE'] == 'MULTIPLE')
         $multiple_save = TRUE;
      else if (!empty($backend_info['MULTIPLE']) && $backend_info['MULTIPLE'] == 'SERIALIZE')
         $new_value = serialize($new_value);
      else if (!empty($backend_info['MULTIPLE']))
         $new_value = implode($backend_info['MULTIPLE'], $new_value);
      else   // just serialize it
         $new_value = serialize($new_value);
   }


   // get file access type, file, directory (if any),
   // new setting template and regular expression
   //
   if (!empty($backend_info['DELETE_INSTEAD_OF_ADD']) && is_array($backend_info['DELETE_INSTEAD_OF_ADD']))
      $delete_instead_of_add = retrieve_server_setting($backend_info['DELETE_INSTEAD_OF_ADD'], $quiet);
   else $delete_instead_of_add = 0;
   if (!empty($backend_info['PATTERN_GROUP_NUMBER']) && is_array($backend_info['PATTERN_GROUP_NUMBER']))
      $pattern_group_number = retrieve_server_setting($backend_info['PATTERN_GROUP_NUMBER'], $quiet);
   else $pattern_group_number = 0;
   if (!empty($backend_info['DELETE_KEEP_PATTERN']) && is_array($backend_info['DELETE_KEEP_PATTERN']))
      $delete_keep_pattern = retrieve_server_setting($backend_info['DELETE_KEEP_PATTERN'], $quiet);
   else $delete_keep_pattern = '';
   if (!empty($backend_info['ADD_TO_TOP']) && is_array($backend_info['ADD_TO_TOP']))
      $add_to_top = retrieve_server_setting($backend_info['ADD_TO_TOP'], $quiet);
   else $add_to_top = 0;
   if (!empty($backend_info['UPDATE_SEARCH_PATTERN']) && is_array($backend_info['UPDATE_SEARCH_PATTERN']))
      $update_search_pattern = retrieve_server_setting($backend_info['UPDATE_SEARCH_PATTERN'], $quiet);
   else $update_search_pattern = '';
   if (!empty($backend_info['UPDATE_REPLACE_PATTERN']) && is_array($backend_info['UPDATE_REPLACE_PATTERN']))
      $update_replace_pattern = retrieve_server_setting($backend_info['UPDATE_REPLACE_PATTERN'], $quiet);
   else $update_replace_pattern = '';
   if (!empty($backend_info['REPLACEMENT_PATTERN']) && is_array($backend_info['REPLACEMENT_PATTERN']))
      $replacement_pattern = retrieve_server_setting($backend_info['REPLACEMENT_PATTERN'], $quiet);
   else $replacement_pattern = '';
   if (!empty($backend_info['DELETE_WHEN_EMPTY']) && is_array($backend_info['DELETE_WHEN_EMPTY']))
      $delete_when_empty = retrieve_server_setting($backend_info['DELETE_WHEN_EMPTY'], $quiet);
   else $delete_when_empty = 0;
   if (!empty($backend_info['ACCESS_TYPE']) && is_array($backend_info['ACCESS_TYPE']))
      $access_type = retrieve_server_setting($backend_info['ACCESS_TYPE'], $quiet);
   else
      $error = _("Misconfiguration encountered when attempting to store a setting on the server (file access method is misconfigured)");
   if (!empty($backend_info['FILE']) && is_array($backend_info['FILE']))
      $remote_file = retrieve_server_setting($backend_info['FILE'], $quiet);
   else
      $error = _("Misconfiguration encountered when attempting to store a setting on the server (file name is misconfigured)");
   if (!empty($backend_info['PARSE_PATTERN']) && is_array($backend_info['PARSE_PATTERN']))
      $parse_pattern = retrieve_server_setting($backend_info['PARSE_PATTERN'], $quiet);
   else
      $error = _("Misconfiguration encountered when attempting to store a setting on the server (parsing pattern is misconfigured)");
   if (!empty($backend_info['NEW_SETTING_TEMPLATE']) && is_array($backend_info['NEW_SETTING_TEMPLATE']))
      $new_setting_template = retrieve_server_setting($backend_info['NEW_SETTING_TEMPLATE'], $quiet);
   else
      $error = _("Misconfiguration encountered when attempting to store a setting on the server (new setting template is misconfigured)");
   if (!empty($backend_info['SUID_LOCATION']) && is_array($backend_info['SUID_LOCATION']))
      $suid_binary = retrieve_server_setting($backend_info['SUID_LOCATION'], $quiet);
   else if ($access_type == 'SUID')
      $error = _("Misconfiguration encountered when attempting to store a setting on the server (SUID_LOCATION is misconfigured)");
   else $suid_binary = '';
   if (!empty($backend_info['SUID_DEBUG_OUTPUT']) && is_array($backend_info['SUID_DEBUG_OUTPUT']))
      $suid_debug_file = retrieve_server_setting($backend_info['SUID_DEBUG_OUTPUT'], $quiet);
   else
      $suid_debug_file = '';
   if (!empty($backend_info['MASTER_SECURITY_CHECK']) && is_array($backend_info['MASTER_SECURITY_CHECK']))
      $master_security_check = retrieve_server_setting($backend_info['MASTER_SECURITY_CHECK'], $quiet);
   else
      $master_security_check = FALSE;
   if (!empty($backend_info['MASTER_USERNAME']) && is_array($backend_info['MASTER_USERNAME']))
      $master_username = retrieve_server_setting($backend_info['MASTER_USERNAME'], $quiet);
   else
      $master_username = NULL;


   // if using "master" security check, grab needed values from environment
   //
   if ($master_security_check)
   {
      if (!sqGetGlobalVar('SQUIRRELMAIL_MASTER_PASSWORD', $master_password, SQ_SERVER)
       || $master_password == 'test-password')
         $error = _("Misconfiguration encountered when attempting to store a setting on the server (master password is misconfigured)");
      if ($master_username
       && (!sqGetGlobalVar('SQUIRRELMAIL_MASTER_USERNAME', $master_username, SQ_SERVER)
        || $master_username == 'test-username'))
         $error = _("Misconfiguration encountered when attempting to store a setting on the server (master username is misconfigured)");
   }
   else
      $master_username = $master_password = NULL;


   // manipulate username, domain and password if needed
   //
   list($user, $dom, $pass) = get_credentials($backend_info,
                                              $username,
                                              $domain,
                                              sqauth_read_password(),
                                              $quiet);


   // when all three are null, an error occured and $quiet
   // is turned on, so we can just fake an error
   //
   if ($quiet && is_null($user) && is_null($dom) && is_null($pass))
      $error = 'dummy';


   // validate we have what we need
   //
   if (empty($error)
    && (empty($access_type)
    || empty($remote_file)
    || empty($parse_pattern)
    || ($access_type == 'SUID' && empty($suid_binary))))
      $error = _("Misconfiguration encountered when attempting to store a setting on the server (missing file name, access type, SUID binary, or parsing pattern)");
   else if (empty($error))
   {

      // first, get current value and file contents
      //
      $remote_file = str_replace(array('%1', '%2'), array(ssb_str_esc($user), ssb_str_esc($dom)), $remote_file);
      list($current_value, $file_contents)
         = retrieve_server_setting_from_local_file($access_type, $remote_file,
                                                   $user, $pass,
                                                   $parse_pattern,
                                                   $pattern_group_number,
                                                   TRUE,
                                                   $suid_binary,
                                                   $suid_debug_file,
                                                   $master_username,
                                                   $master_password,
                                                   $quiet, TRUE);


      // decode serialized or imploded multi-values
      //
      if (!empty($backend_info['MULTIPLE']) && $backend_info['MULTIPLE'] == 'SERIALIZE'
       && is_array($current_value) && !is_array($current_value[0]))
      {
//TODO: should we use sq_call_function_suppress_errors() here?  NOTE that if the field is too short for having saved the serialized value, this will throw a PHP notice (because it won't be correctly formatted for unserialization) and truncate the value
         $current_value = unserialize($current_value[0]);
         $temp = array();
         foreach ($current_value as $value)
            $temp[] = array($value);
         $current_value = $temp;
      }
      else if (!empty($backend_info['MULTIPLE']) && $backend_info['MULTIPLE'] != 'MULTIPLE'
       && is_array($current_value) && !is_array($current_value[0]))
      {
         $current_value = explode($backend_info['MULTIPLE'], $current_value[0]);
         $temp = array();
         foreach ($current_value as $value)
            $temp[] = array($value);
         $current_value = $temp;
      }


      // plain scalar value - convert to multi-format
      //
      else if (is_array($current_value) && !is_array($current_value[0]))
         $current_value = array($current_value);


      // we'll assume NULL return means value isn't yet in the file
      // even though NULL might also mean an error ocurred
      //
      else if (is_null($current_value))
         $current_value = array(array());


      // what happened?
      //
      if (!is_array($current_value) || !is_array($current_value[0]))
         $error = _("Unknown error encountered when attempting to store a setting on the server (unhandled return type getting old value from local file)");


      // update the new value(s) in the file
      //
      else
      {
         // Note that doing $remove = !$remove; creates anomalous
         // problems that are very hard to track down
         //
         if ($delete_instead_of_add) $remove = TRUE;
         $new_file_contents = edit_file_update_setting($file_contents,
                                                       $backend_info,
                                                       $user, $dom,
                                                       $original_new_value,
                                                       $current_value,
                                                       $parse_pattern,
                                                       $delete_keep_pattern,
                                                       $new_setting_template,
                                                       $replacement_pattern,
                                                       $update_search_pattern,
                                                       $update_replace_pattern,
                                                       $add_to_top, $add,
                                                       $remove, $quiet);


         // store our value!
         //
         // Note that $remote_file already had %1
         // and %2 replacements handed above
         //
         sq_change_text_domain($original_text_domain);
         return put_local_file($access_type, $new_file_contents, TRUE,
                               $user, $pass, $remote_file, 
                               $delete_when_empty, $suid_binary,
                               $suid_debug_file, $master_username,
                               $master_password, $quiet);

      }

   }
   
      
   // should not get here without an error message
   // unless something is very wrong
   //    
   if (empty($error))
      $error = _("Unknown error in Server Settings Backend plugin, function put_server_setting()");

         
   // spit out error and exit
   //    
   if ($quiet)
   {
      sq_change_text_domain($original_text_domain);
      return NULL;
   }
   sq_change_text_domain('squirrelmail');  // NOT $original_text_domain
   global $color;
   $ret = plain_error_message($error, $color);
   if (check_sm_version (1, 5, 2))
   {        
      echo $ret;
      global $oTemplate;
      $oTemplate->display('footer.tpl');
   }  
   exit;

}



