<?php

/**
  * SquirrelMail Server Settings Backend Plugin
  * Copyright (c) 2008-2010 Paul Lesniewski <paul@squirrelmail.org>
  * Licensed under the GNU GPL. For full terms see the file COPYING.
  *
  * @package plugins
  * @subpackage server_settings_backend
  *
  */



/**
  * Retrieves a setting from the server using the SQL backend.
  *
  * If some fatal error occurs due to misconfiguration or
  * backend failure, an error message is printed and execution
  * stops immediately, unless $quiet is TRUE.
  *
  * If the requested setting consists of multiple values,
  * every effort is made to return them in an array, but
  * specifying the correct value of "MULTIPLE" in the
  * storage configuration settings helps ensure this will
  * work correctly.  If "MULTIPLE" is not found, default
  * behavior is to return multiple values in an array.
  *
  * @param array   $backend_info An array of configuration values
  *                              that point to how the setting value
  *                              should be retrieved.
  *                              This value is assumed to have already
  *                              been error-checked to an extent: it
  *                              is assumed to be an array, and all
  *                              array keys are in all capital letters,
  *                              and if a VALUE or CUSTOM entry is
  *                              present, it has already been dealt
  *                              with.  Same as for multiple storage
  *                              arrays - this array should be non-
  *                              nested.
  * @param boolean $quiet        When TRUE, suppresses any error
  *                              messages and makes this function
  *                              return NULL without doing anything
  *                              else (OPTIONAL; default = FALSE).
  * @param mixed $test_value     When given, this indicates that the
  *                              setting is being retrieved
  *                              specifically for testing per the
  *                              test_server_setting() function (or
  *                              similar), and so if there is a
  *                              different (more efficient) kind of
  *                              lookup for this action, it can be
  *                              enabled.  When given, its value must
  *                              be the value being tested for
  *                              (OPTIONAL; default = NULL).
  *
  * @return mixed The setting value if it was retrieved normally
  *               or NULL if it was not (and $quiet is TRUE)
  *
  */
function retrieve_server_setting_sql($backend_info, $quiet=FALSE, $test_value=NULL)
{

   global $username, $domain;
   include_once(SM_PATH . 'plugins/server_settings_backend/sql_functions.php');
   $error = '';
   $original_text_domain = sq_change_text_domain('server_settings_backend');


   // if this setting consists of multiple values,
   // how will we handle them?
   //
   if (!empty($backend_info['MULTIPLE']))
      $multiple = $backend_info['MULTIPLE'];
   else
      $multiple = 'MULTIPLE';  // note that this seems incorret for scalars, but
                               // MULTIPLE returns the value as is (an array), which
                               // is also exactly what we want to do with scalars


   // get data source name
   //
   if (!empty($backend_info['DSN']) && is_array($backend_info['DSN']))
      $dsn = retrieve_server_setting($backend_info['DSN'], $quiet);
   else
   {
      $error = _("Misconfiguration encountered when attempting to retrieve a setting from the server (SQL lookup DSN is misconfigured)");
   }


   // look for custom-defined DB abstraction layer
   //
   if (!empty($backend_info['DATABASE_ABSTRACTION_LAYER']) && is_array($backend_info['DATABASE_ABSTRACTION_LAYER']))
      $abstraction_type = retrieve_server_setting($backend_info['DATABASE_ABSTRACTION_LAYER'], $quiet);
   else
      $abstraction_type = 'DB';


   // get test query if necessary and if available
   //
   if (!is_null($test_value)
    && !empty($backend_info['TEST_QUERY']) && is_array($backend_info['TEST_QUERY']))
      $query = retrieve_server_setting($backend_info['TEST_QUERY'], $quiet);


   // get normal query
   //
   else if (!empty($backend_info['QUERY']) && is_array($backend_info['QUERY']))
      $query = retrieve_server_setting($backend_info['QUERY'], $quiet);


   else
   {
      $error = _("Misconfiguration encountered when attempting to retrieve a setting from the server (SQL lookup QUERY is misconfigured)");
   }


   // manipulate username, domain and password if needed
   // (password not used in SQL backend, but let's keep
   // this the same for all backends)
   //
   list($user, $dom, $pass) = get_credentials($backend_info,
                                              $username,
                                              $domain,
                                              sqauth_read_password(),
                                              $quiet);


   // when all three are null, an error occured and $quiet
   // is turned on, so we can just fake an error
   //
   if ($quiet && is_null($user) && is_null($dom) && is_null($pass))
      $error = 'dummy';


   // validate we have what we need
   //
   if (empty($error)
    && (empty($dsn)
    || empty($query)))
      $error = _("Misconfiguration encountered when attempting to retrieve a setting from the server (missing SQL lookup query or DSN)");
   else if (empty($error))
   {

      // get our value!
      //
      $query = str_replace(array('%1', '%2'), array(ssb_str_esc($user), ssb_str_esc($dom)), $query);
      $ret = retrieve_server_setting_from_sql($dsn, $abstraction_type, $query, $quiet);
      if (!is_null($ret))  // don't do anything if no value was found
      {
         if ($multiple == 'SERIALIZE')
         {
            if (!is_string($ret))
               $error = _("Misconfiguration encountered when attempting to retrieve a setting from the server (supposedly serialized setting returned array of values from database)");
//TODO: should we use sq_call_function_suppress_errors() here?  NOTE that if the SQL field is too short for having saved the serialized value, this will throw a PHP notice (because it won't be correctly formatted for unserialization) and truncate the value
            else $ret = unserialize($ret);
         }
         else if ($multiple != 'MULTIPLE')
         {
            if (!is_string($ret))
               $error = _("Misconfiguration encountered when attempting to retrieve a setting from the server (supposedly concatenated setting returned array of values from database)");
            else $ret = explode($multiple, $ret);
         }
      }

      if (empty($error))
      {
         sq_change_text_domain($original_text_domain);
         return $ret;
      }

   }


   // should not get here without an error message 
   // unless something is very wrong
   //
   if (empty($error))
      $error = _("Unknown error in Server Settings Backend plugin, function retrieve_server_setting()");


   // spit out error and exit
   //
   if ($quiet)
   {
      sq_change_text_domain($original_text_domain);
      return NULL;
   }
   sq_change_text_domain('squirrelmail');  // NOT $original_text_domain
   global $color;
   $ret = plain_error_message($error, $color);
   if (check_sm_version (1, 5, 2))
   {
      echo $ret;
      global $oTemplate;
      $oTemplate->display('footer.tpl');
   }
   exit;

}



/**
  * Stores a setting on the server using the SQL backend.
  *
  * If some fatal error occurs due to misconfiguration or
  * backend failure, an error message is printed and execution
  * stops immediately, unless $quiet is TRUE.
  *
  * @param mixed   $new_value    The value to be stored.
  * @param array   $backend_info An array of configuration values
  *                              that point to how the setting value
  *                              should be stored.
  *                              This value is assumed to have already
  *                              been error-checked to an extent: it
  *                              is assumed to be an array, and all
  *                              array keys are in all capital letters,
  *                              and if a VALUE or CUSTOM entry is
  *                              present, it has already been dealt
  *                              with.  Same as for multiple storage
  *                              arrays - this array should be non-
  *                              nested.
  * @param boolean $add          When TRUE, and if the setting being
  *                              updated consists of multiple individual
  *                              values, $new_value is added to them
  *                              and the other values are left as is.
  *                              If the setting's configuration does
  *                              not include a "MULTIPLE" item, then 
  *                              this flag has no effect and the setting
  *                              is updated as if it were turned off
  *                              (OPTIONAL; default = FALSE).
  * @param boolean $remove       When TRUE, and if the setting being
  *                              updated consists of multiple individual
  *                              values, $new_value is removed from
  *                              those values (if found within) and the
  *                              other values are left as is.  If the
  *                              setting's configuration does not include
  *                              a "MULTIPLE" item, then this flag has no
  *                              effect and the setting is updated as if
  *                              it were turned off (OPTIONAL; default
  *                              = FALSE).
  * @param boolean $quiet        When TRUE, suppresses any error
  *                              messages and makes this function
  *                              return NULL without doing anything
  *                              else (OPTIONAL; default = FALSE).
  *
  * @return mixed TRUE if the setting was stored normally or
  *               NULL if it was not (and $quiet is TRUE).
  *
  */
function put_server_setting_sql($new_value, $backend_info,
                                $add=FALSE, $remove=FALSE, $quiet=FALSE)
{

   global $username, $domain;
   include_once(SM_PATH . 'plugins/server_settings_backend/sql_functions.php');
   $error = '';
   $original_text_domain = sq_change_text_domain('server_settings_backend');


   // if this setting consists of multiple values,
   // how will we handle them?
   //
   $original_new_value = $new_value;
   $multiple_save = FALSE;
   if (is_array($new_value))
   {
      if (!empty($backend_info['MULTIPLE']) && $backend_info['MULTIPLE'] == 'MULTIPLE')
         $multiple_save = TRUE;
      else if (!empty($backend_info['MULTIPLE']) && $backend_info['MULTIPLE'] == 'SERIALIZE')
         $new_value = serialize($new_value);
      else if (!empty($backend_info['MULTIPLE']))
         $new_value = implode($backend_info['MULTIPLE'], $new_value);
      else   // just serialize it
         $new_value = serialize($new_value);
   }


   $skip_pre_save_query = FALSE;
   $query = '';


   // get data source name
   //
   if (!empty($backend_info['DSN']) && is_array($backend_info['DSN']))
      $dsn = retrieve_server_setting($backend_info['DSN'], $quiet);
   else
   {
      $error = _("Misconfiguration encountered when attempting to store a setting on the server (SQL lookup DSN is misconfigured)");
   }


   // look for custom-defined DB abstraction layer
   //
   if (!empty($backend_info['DATABASE_ABSTRACTION_LAYER']) && is_array($backend_info['DATABASE_ABSTRACTION_LAYER']))
      $abstraction_type = retrieve_server_setting($backend_info['DATABASE_ABSTRACTION_LAYER'], $quiet);
   else
      $abstraction_type = 'DB';


   // if adding a value to a multi-value setting...
   //
   if (empty($error) && $add && !empty($backend_info['MULTIPLE']))
   {

      // if a special query is available, use it, skip 
      // other special processing here
      //
      if (!empty($backend_info['ADD_QUERY']) && is_array($backend_info['ADD_QUERY']))
      {
         $query = retrieve_server_setting($backend_info['ADD_QUERY'], $quiet);
         $skip_pre_save_query = TRUE;
      }

      // otherwise, we have to grab the setting value,
      // adjust it, and then save it (per normal)
      //
      else
      {

         // first, get current value
         //
         $current_value = retrieve_server_setting($backend_info, $quiet);
         if (empty($current_value))
            $current_value = array();
         if (!is_array($current_value))
            $current_value = array($current_value);


         // start over with original new value argument
         //
         if (!is_array($original_new_value))
            $new_value = array($original_new_value);
         else
            $new_value = $original_new_value;


         // now concatenate the new value to it
         //
         $new_value = array_merge($current_value, $new_value);


         // re-serialize or implode if needed
         //
         if ($backend_info['MULTIPLE'] == 'MULTIPLE')
            $multiple_save = TRUE;
         else if ($backend_info['MULTIPLE'] == 'SERIALIZE')
            $new_value = serialize($new_value);
         else
            $new_value = implode($backend_info['MULTIPLE'], $new_value);

      }

   }


   // if removing a value from a multi-value setting...
   //
   else if (empty($error) && $remove && !empty($backend_info['MULTIPLE']))
   {

      // if a special query is available, use it, skip
      // other special processing here
      //
      if (!empty($backend_info['REMOVE_QUERY']) && is_array($backend_info['REMOVE_QUERY']))
      {
         $query = retrieve_server_setting($backend_info['REMOVE_QUERY'], $quiet);
         $skip_pre_save_query = TRUE;
      }

      // otherwise, we have to grab the setting value,
      // adjust it, and then save it (per normal)
      //
      else
      {

         // first, get current value
         //
         $current_value = retrieve_server_setting($backend_info, $quiet);
         if (empty($current_value))
            $current_value = array();
         if (!is_array($current_value))
            $current_value = array($current_value);


         // start over with original new value argument
         // 
         if (!is_array($original_new_value))
            $new_value = array($original_new_value);
         else
            $new_value = $original_new_value;


         // now remove the new value from it
         //
         $new_value = array_diff($current_value, $new_value);


         // re-serialize or implode if needed
         //
         if ($backend_info['MULTIPLE'] == 'MULTIPLE')
            $multiple_save = TRUE;
         else if ($backend_info['MULTIPLE'] == 'SERIALIZE')
            $new_value = serialize($new_value);
         else
            $new_value = implode($backend_info['MULTIPLE'], $new_value);

      }

   }


   // get pre-save query
   //
   if (!$skip_pre_save_query
    && !empty($backend_info['PRE_SAVE_QUERY'])
    && is_array($backend_info['PRE_SAVE_QUERY']))
      $pre_save_query = retrieve_server_setting($backend_info['PRE_SAVE_QUERY'], $quiet);
   else if (!$skip_pre_save_query && !empty($backend_info['PRE_SAVE_QUERY']))
   {
      $error = _("Misconfiguration encountered when attempting to store a setting on the server (SQL lookup PRE_SAVE_QUERY is misconfigured)");
   }
   else
      $pre_save_query = '';


   // get query (unless a special one was already found above)
   //
   if (empty($query) && !empty($backend_info['SAVE_QUERY']) && is_array($backend_info['SAVE_QUERY']))
      $query = retrieve_server_setting($backend_info['SAVE_QUERY'], $quiet);
   else if (empty($query))
   {
      $error = _("Misconfiguration encountered when attempting to store a setting on the server (SQL lookup SAVE_QUERY is misconfigured)");
   }


   // manipulate username, domain and password if needed
   // (password not used in SQL backend, but let's keep
   // this the same for all backends)
   //
   list($user, $dom, $pass) = get_credentials($backend_info,
                                              $username,
                                              $domain,
                                              sqauth_read_password(),
                                              $quiet);


   // when all three are null, an error occured and $quiet
   // is turned on, so we can just fake an error
   //
   if ($quiet && is_null($user) && is_null($dom) && is_null($pass))
      $error = 'dummy';


   // validate we have what we need
   //
   if (empty($error)
    && (empty($dsn)
    || empty($query)))
      $error = _("Misconfiguration encountered when attempting to store a setting on the server (missing SQL lookup query or DSN)");
   else if (empty($error))
   {
      if (!empty($pre_save_query))
      {
         $pre_save_query = str_replace(array('%1', '%2'), array(ssb_str_esc($user), ssb_str_esc($dom)), $pre_save_query);
         execute_sql_query($dsn, $abstraction_type, $pre_save_query, $quiet);
      }
      if (is_array($new_value) && $multiple_save)
      {
         $ret = NULL;

         if (empty($new_value))
         {
            sq_change_text_domain($original_text_domain);
            return TRUE;
         }

         foreach ($new_value as $value)
         {
            $save_query = str_replace(array('%1', '%2', '%3'), array(ssb_str_esc($user), ssb_str_esc($dom), ssb_str_esc($value)), $query);
            $ret = execute_sql_query($dsn, $abstraction_type, $save_query, $quiet);
            if (is_null($ret)) break;
         }

         sq_change_text_domain($original_text_domain);
         return $ret;
      }
      else
      {

         // if a REMOVE query is provided, we can delete the entire
         // setting when it is empty if desired
         //
         if (empty($new_value)
          && !empty($backend_info['DELETE_WHEN_EMPTY']) && is_array($backend_info['DELETE_WHEN_EMPTY'])
          && !empty($backend_info['REMOVE_QUERY']) && is_array($backend_info['REMOVE_QUERY']))
            $delete_when_empty = retrieve_server_setting($backend_info['DELETE_WHEN_EMPTY'], $quiet);
         else $delete_when_empty = 0;
         if ($delete_when_empty)
            $query = retrieve_server_setting($backend_info['REMOVE_QUERY'], $quiet);


         $query = str_replace(array('%1', '%2', '%3'), array(ssb_str_esc($user), ssb_str_esc($dom), ssb_str_esc($new_value)), $query);
         sq_change_text_domain($original_text_domain);
         return execute_sql_query($dsn, $abstraction_type, $query, $quiet);
      }
   }
   
      
   // should not get here without an error message
   // unless something is very wrong
   //    
   if (empty($error))
      $error = _("Unknown error in Server Settings Backend plugin, function put_server_setting()");

         
   // spit out error and exit
   //    
   if ($quiet)
   {
      sq_change_text_domain($original_text_domain);
      return NULL;
   }
   sq_change_text_domain('squirrelmail');  // NOT $original_text_domain
   global $color;
   $ret = plain_error_message($error, $color);
   if (check_sm_version (1, 5, 2))
   {        
      echo $ret;
      global $oTemplate;
      $oTemplate->display('footer.tpl');
   }  
   exit;

}



