<?php

/**
  * SquirrelMail Server Settings Backend Plugin
  * Copyright (c) 2008-2010 Paul Lesniewski <paul@squirrelmail.org>
  * Licensed under the GNU GPL. For full terms see the file COPYING.
  *
  * @package plugins
  * @subpackage server_settings_backend
  *
  */



/**
  * Constants that correspond to the same constants used in the SUID backend
  *
  */
define('SSB_ERR_OK', 0);
define('SSB_ERR_NOTFOUND', 1);



/**
  * Retrieves a setting from a local file.
  *
  * If the target file has more than one occurance of
  * the setting, the results will be returned as a
  * set of nested arrays.  When no results were found,
  * NULL is returned.
  *
  * If some fatal error occurs due to misconfiguration or
  * backend failure, an error message is printed and execution
  * stops immediately, unless $quiet is TRUE.
  *
  * @param string  $access_type     The method of file access, which
  *                                 should be either 'PHP' or 'SUID'.
  * @param string  $file            The full path to the file containing
  *                                 the needed setting (relative to the
  *                                 user's home directory in the case
  *                                 of the set-uid backend).
  * @param string  $user            The username for whom to get the setting.
  * @param string  $pass            The user's password.
  * @param string  $parse_pattern   The regular expression pattern
  *                                 needed to parse the setting value
  *                                 out of the file.
  * @param string  $pattern_group_number The group number of the regular
  *                                      expression parsing pattern that
  *                                      contains ONLY the actual value
  *                                      (and not, for example, also the
  *                                      setting name).
  * @param boolean $treat_as_empty_when_not_found When TRUE, if the target
  *                                               file is non-existent, the
  *                                               return value is as if the
  *                                               setting was not found in
  *                                               the file (OPTIONAL;
  *                                               default = FALSE).
  * @param string  $suid_binary     The location of the SUID binary executable
  *                                 (OPTIONAL (but required if $access_type
  *                                 is "SUID"); default none/empty string)
  * @param string  $suid_debug_file The location to send any SUID debug/error
  *                                 output (OPTIONAL; default none/empty string).
  * @param string  $master_username The username to send for an additional
  *                                 security check to the SUID backend; when
  *                                 NULL, no master username will be sent
  *                                 (OPTIONAL; default NULL).
  * @param string  $master_password The password to send for an additional
  *                                 security check to the SUID backend; when
  *                                 NULL, no master password will be sent
  *                                 (OPTIONAL; default NULL).
  * @param boolean $quiet           When TRUE, suppresses any error
  *                                 messages and makes this function
  *                                 return NULL without doing anything
  *                                 else (OPTIONAL; default = FALSE).
  * @param boolean $return_file_contents When TRUE, the normal return
  *                                      value is placed inside an array
  *                                      to which the file contents are
  *                                      added as the second and last
  *                                      array element, and this two-
  *                                      element array is then returned
  *                                      (OPTIONAL; default = FALSE).
  *
  * @return mixed An array containing the setting value and its offset in
  *               the target file (in that order) if it was retrieved
  *               normally and was only found once in the file (scalar
  *               value), or an array of arrays, where the sub-arrays
  *               contain each setting value and its offset in the target
  *               file (in that order) if the setting is found more than
  *               once in the file (the caller should check if the return
  *               value's first array element is an array itself to
  *               determine if the result is non-scalar), or NULL if the
  *               setting is not found in the file or if an error occurred
  *               (and $quiet is TRUE).  Note that if $return_file_contents
  *               is TRUE, the normal return value described above will be
  *               placed inside a two-element array, and the file contents
  *               placed as the second array element, and that is returned.
  *
  */
function retrieve_server_setting_from_local_file($access_type, $file, $user, $pass, 
                                                 $parse_pattern, $pattern_group_number,
                                                 $treat_as_empty_when_not_found=FALSE,
                                                 $suid_binary='', $suid_debug_file='',
                                                 $master_username=NULL, $master_password=NULL,
                                                 $quiet=FALSE, $return_file_contents=FALSE)
{

   $error = '';

   $temp_file = get_local_file($access_type, $user, $pass, $file,
                               $treat_as_empty_when_not_found,
                               $suid_binary, $suid_debug_file,
                               $master_username, $master_password,
                               $quiet);


   // first deal with file-not-found
   //
   if (empty($temp_file))
   {

      if ($treat_as_empty_when_not_found)
      {  
         // parse_value() found in functions.php
         // 
         $ret = parse_value('', $parse_pattern, $pattern_group_number, $quiet);
         if ($return_file_contents)
            $ret = array($ret, '');
         return $ret;
      }  

      if ($quiet) return NULL;

      sq_change_text_domain('server_settings_backend');
      global $sm_debug_mode, $server_settings_backend_debug;
      server_settings_backend_init();
      if ($server_settings_backend_debug) $sm_debug_mode |= SM_DEBUG_MODE_SIMPLE;
      if ($sm_debug_mode & SM_DEBUG_MODE_SIMPLE)
         $error = sprintf(_("Unable to read file \"%s\" when attempting to retrieve a setting from the server"), $file);
      else
         $error = _("Unable to read file when attempting to retrieve a setting from the server");
      sq_change_text_domain('squirrelmail');

   }


   // parse out our target value from the file
   //
   else
   {

      // parse_value_from_file() found in functions.php
      //
      $ret = parse_value_from_file($temp_file, $parse_pattern,
                                   $pattern_group_number, $quiet,
                                   $return_file_contents);

      // don't remove original file in the case of PHP file access
      //
      if ($access_type != 'PHP')
         unlink($temp_file);

      return $ret;

   }


   // spew error
   //
   global $color;
   $ret = plain_error_message($error, $color);
   if (check_sm_version (1, 5, 2))
   {
      echo $ret;
      global $oTemplate;
      $oTemplate->display('footer.tpl');
   }
   exit;

}



/**
  * Stores a file on the local server
  *
  * If some fatal error occurs due to misconfiguration or
  * backend failure, an error message is printed and execution
  * stops immediately, unless $quiet is TRUE.
  *
  * @param string  $access_type       The method of file access, which
  *                                   should be either 'PHP' or 'SUID'.
  * @param string  $file_contents     The contents of the file to be
  *                                   placed on the server.
  * @param boolean $allow_overwrite   When TRUE, any existing file at
  *                                   $path will be overwritten with
  *                                   $contents, otherwise, an existing
  *                                   file will create an error.
  * @param string  $user              The username for whom the file is
  *                                   being saved.
  * @param string  $pass              The user's password.
  * @param string  $remote_file       The full path to the file where
  *                                   the given contents should be saved
  *                                   (relative to the user's home
  *                                   directory in the case of the
  *                                   set-uid backend).
  * @param boolean $delete_when_empty When TRUE, if the file contents
  *                                   are empty, the file will be
  *                                   deleted instead of left as an
  *                                   empty file (OPTIONAL; default = FALSE).
  * @param string  $suid_binary       The location of the SUID binary executable
  *                                   (OPTIONAL (but required if $access_type
  *                                   is "SUID"); default none/empty string)
  * @param string  $suid_debug_file   The location to send any SUID debug/error
  *                                   output (OPTIONAL; default none/empty string).
  * @param string  $master_username   The username to send for an additional
  *                                   security check to the SUID backend; when
  *                                   NULL, no master username will be sent
  *                                   (OPTIONAL; default NULL).
  * @param string  $master_password   The password to send for an additional
  *                                   security check to the SUID backend; when
  *                                   NULL, no master password will be sent
  *                                   (OPTIONAL; default NULL).
  * @param boolean $quiet             When TRUE, suppresses any error
  *                                   messages and makes this function
  *                                   return NULL without doing anything
  *                                   else (OPTIONAL; default = FALSE).
  *
  * @return mixed TRUE when completed normally, or NULL if an error
  *               occurred (and $quiet is TRUE).
  *
  */
function put_local_file($access_type, $file_contents, $allow_overwrite,
                        $user, $pass, $remote_file,
                        $delete_when_empty=FALSE, $suid_binary='',
                        $suid_debug_file='', $master_username=NULL,
                        $master_password=NULL, $quiet=FALSE)
{

   $error = '';
   while (strlen($remote_file) > 1
    && ($remote_file[strlen($remote_file) - 1] == '/'
     || $remote_file[strlen($remote_file) - 1] == '\\'))
      $remote_file = substr($remote_file, 0, -1);


   // if overwriting not allowed, bail if path already exists
   //
   if (!$allow_overwrite
    && local_file_path_exists($access_type, $user, $pass, $remote_file, FALSE,
                              $suid_binary, $suid_debug_file, $master_username,
                              $master_password, $quiet))
   {
      if ($quiet) return NULL;

      sq_change_text_domain('server_settings_backend');
      global $sm_debug_mode, $server_settings_backend_debug;
      server_settings_backend_init();
      if ($server_settings_backend_debug) $sm_debug_mode |= SM_DEBUG_MODE_SIMPLE;
      if ($sm_debug_mode & SM_DEBUG_MODE_SIMPLE)
         $error = sprintf(_("File or directory \"%s\" already exists; cannot overwrite"), $remote_file);
      else
         $error = _("File or directory already exists; cannot overwrite");
      sq_change_text_domain('squirrelmail');
   }


   // if file is empty, should we just delete it?
   //
   if (empty($error) && $delete_when_empty)
   {

      $trimmed_file_contents = trim($file_contents);

      if (empty($trimmed_file_contents))
      {

         // only way this can return as non-TRUE is if $quiet
         // and an error occured, so if it does, just return NULL
         //
         if (!local_file_delete($access_type, $user, $pass, $remote_file, FALSE,
                                FALSE, $suid_binary, $suid_debug_file,
                                $master_username, $master_password, $quiet))
            return NULL;

         return TRUE;
      }

   }


   // proceed to upload
   //
   if (empty($error))
   {

      // PHP file access
      //
      if ($access_type == 'PHP')
      {

         // write the file
         //
         if (check_php_version(5, 0, 0))
         {
            if (sq_call_function_suppress_errors('file_put_contents', array($remote_file, $file_contents)) === FALSE)
            {
               sq_change_text_domain('server_settings_backend');
               global $sm_debug_mode, $server_settings_backend_debug;
               server_settings_backend_init();
               if ($server_settings_backend_debug) $sm_debug_mode |= SM_DEBUG_MODE_SIMPLE;
               if ($sm_debug_mode & SM_DEBUG_MODE_SIMPLE)
                  $error = sprintf(_("Could not write to file \"%s\""), $remote_file);
               else
                  $error = _("Could not write to file");
               sq_change_text_domain('squirrelmail');
            }
            else return TRUE;
         }
         else
         {
            $FILE = @fopen($remote_file, 'w');
            if (!is_resource($FILE))
            {
               sq_change_text_domain('server_settings_backend');
               global $sm_debug_mode, $server_settings_backend_debug;
               server_settings_backend_init();
               if ($server_settings_backend_debug) $sm_debug_mode |= SM_DEBUG_MODE_SIMPLE;
               if ($sm_debug_mode & SM_DEBUG_MODE_SIMPLE)
                  $error = sprintf(_("Could not open file \"%s\""), $remote_file);
               else
                  $error = _("Could not open file");
               sq_change_text_domain('squirrelmail');
            }
            else
            {
               //if (is_array($file_contents)) $file_contents = implode($file_contents);
               if (!fwrite($FILE, $file_contents))
               {
                  @fclose($FILE);
                  sq_change_text_domain('server_settings_backend');
                  global $sm_debug_mode, $server_settings_backend_debug;
                  server_settings_backend_init();
                  if ($server_settings_backend_debug) $sm_debug_mode |= SM_DEBUG_MODE_SIMPLE;
                  if ($sm_debug_mode & SM_DEBUG_MODE_SIMPLE)
                     $error = sprintf(_("Could not write to file \"%s\""), $remote_file);
                  else
                     $error = _("Could not write to file");
                  sq_change_text_domain('squirrelmail');
               }
               else
               {
                  @fclose($FILE);
                  return TRUE;
               }
            }
         }

      }


      // set-uid wrapper file access
      //
      else if ($access_type == 'SUID')
      {

         // get a temp file for uploading the target file
         //
         $temp = ssb_get_temp_file($quiet);
         if (is_null($temp)) // implied $quiet is TRUE
            return NULL;
         list($local_file, $FILE) = $temp;


         // now, write the file contents out to the local file
         //
         if (sq_call_function_suppress_errors('fwrite', array($FILE, $file_contents)) === FALSE)
         {
            if ($quiet)
               return NULL;
            sq_change_text_domain('server_settings_backend');
            global $sm_debug_mode, $server_settings_backend_debug;
            server_settings_backend_init();
            if ($server_settings_backend_debug) $sm_debug_mode |= SM_DEBUG_MODE_SIMPLE;
            if ($sm_debug_mode & SM_DEBUG_MODE_SIMPLE)
               $error = sprintf(_("File error when attempting to write to temporary file \"%s\""), $local_file);
            else
               $error = _("File error when attempting to write to a temporary file");

            sq_change_text_domain('squirrelmail');
            fclose($FILE);
            unlink($local_file);
         }


         // and put the file on the server
         //
         else
         {

            fclose($FILE);


            // set up debugging output if needed
            //
            if (!empty($suid_debug_file))
               $suid_debug = ' 2>>' . $suid_debug_file . ' ';
            else
               $suid_debug = '';


            // now, actually send file
            //
            $cmd = $suid_binary
                 . " 'localhost' " // "server" argument isn't currently used - just use "localhost"
                 . escapeshellarg($user)
                 . " 'put_file' "
                 . escapeshellarg($local_file)
                 . ' ' . escapeshellarg($remote_file)
                 . $suid_debug;

            if (!($SUID = sq_call_function_suppress_errors('popen', array($cmd, 'w'))))
            {

               // show command details when in debug mode
               //
               sq_change_text_domain('server_settings_backend');
               global $sm_debug_mode, $server_settings_backend_debug;
               server_settings_backend_init();
               if ($server_settings_backend_debug) $sm_debug_mode |= SM_DEBUG_MODE_SIMPLE;
               if ($sm_debug_mode & SM_DEBUG_MODE_SIMPLE)
                  $error = sprintf(_("Could not establish pipe to command \"%s\""), $cmd);
               else
                  $error = _("Error establishing connection when attempting to store a file on the server");

               sq_change_text_domain('squirrelmail');

               @unlink($local_file);

            }
            else if (!is_null($master_username) && sq_call_function_suppress_errors('fwrite', array($SUID, $master_username . "\n")) === FALSE)
            {

               // show command details and password (CAREFUL!) when in debug mode
               //
               sq_change_text_domain('server_settings_backend');
               global $sm_debug_mode, $server_settings_backend_debug;
               server_settings_backend_init();
               if ($server_settings_backend_debug) $sm_debug_mode |= SM_DEBUG_MODE_SIMPLE;
               if ($sm_debug_mode & SM_DEBUG_MODE_SIMPLE)
                  $error = sprintf(_("Could not send master username \"%s\" to pipe command \"%s\""), $master_username, $cmd);
               else
                  $error = _("Unable to send master username when attempting to store a file on the server");

               sq_change_text_domain('squirrelmail');

               @unlink($local_file);

            }
            else if (!is_null($master_password) && sq_call_function_suppress_errors('fwrite', array($SUID, $master_password . "\n")) === FALSE)
            {

               // show command details and password (CAREFUL!) when in debug mode
               //
               sq_change_text_domain('server_settings_backend');
               global $sm_debug_mode, $server_settings_backend_debug;
               server_settings_backend_init();
               if ($server_settings_backend_debug) $sm_debug_mode |= SM_DEBUG_MODE_SIMPLE;
               if ($sm_debug_mode & SM_DEBUG_MODE_SIMPLE)
                  $error = sprintf(_("Could not send master password \"%s\" to pipe command \"%s\""), $master_password, $cmd);
               else
                  $error = _("Unable to send master password when attempting to store a file on the server");

               sq_change_text_domain('squirrelmail');

               @unlink($local_file);

            }
            else if (sq_call_function_suppress_errors('fwrite', array($SUID, $pass . "\n")) === FALSE)
            {

               // show command details and password (CAREFUL!) when in debug mode
               //
               sq_change_text_domain('server_settings_backend');
               global $sm_debug_mode, $server_settings_backend_debug;
               server_settings_backend_init();
               if ($server_settings_backend_debug) $sm_debug_mode |= SM_DEBUG_MODE_SIMPLE;
               if ($sm_debug_mode & SM_DEBUG_MODE_SIMPLE)
                  $error = sprintf(_("Could not send password \"%s\" to pipe command \"%s\""), $pass, $cmd);
               else
                  $error = _("Unable to send password when attempting to store a file on the server");

               sq_change_text_domain('squirrelmail');

               @unlink($local_file);

            }
            else
            {

               // get command return code
               //
               $status = ssb_pclose($SUID);


               @unlink($local_file);


               // no error?  ready to return
               //
               if ($status == SSB_ERR_OK)
                  return TRUE;


               // show command details when in debug mode
               //
               sq_change_text_domain('server_settings_backend');
               global $sm_debug_mode, $server_settings_backend_debug;
               server_settings_backend_init();
               if ($server_settings_backend_debug) $sm_debug_mode |= SM_DEBUG_MODE_SIMPLE;
               if ($sm_debug_mode & SM_DEBUG_MODE_SIMPLE)
                  $error = sprintf(_("Unable to put file using command \"%s\""), $cmd);
               else
                  $error = sprintf(_("Unable to store file on server (%s)"), $status);

               sq_change_text_domain('squirrelmail');

            }

         }

      }


      // unknown file access type
      //
      else
      {
         sq_change_text_domain('server_settings_backend');
         $error = sprintf(_("Unknown file access type \"%s\""), $access_type);
         sq_change_text_domain('squirrelmail');
      }

   }


   // spew error
   //
   if ($quiet) return NULL;
   global $color;
   $ret = plain_error_message($error, $color);
   if (check_sm_version (1, 5, 2))
   {
      echo $ret;
      global $oTemplate;
      $oTemplate->display('footer.tpl');
   }
   exit;

}



/**
  * Determines if a certain file or directory exists on the local server.
  *
  * If some fatal error occurs due to misconfiguration or
  * backend failure, an error message is printed and execution
  * stops immediately, unless $quiet is TRUE.
  *
  * @param string  $access_type     The method of file access, which
  *                                 should be either 'PHP' or 'SUID'.
  * @param string  $user            The username for whom to check
  *                                 file/directory existence.
  * @param string  $pass            The user's password.
  * @param string  $path            The (full) path to inspect for existence
  *                                 (relative to the user's home directory
  *                                 in the case of the set-uid backend).
  * @param boolean $return_info     When TRUE, information about the path
  *                                 is returned instead of just a boolean
  *                                 return value.
  * @param string  $suid_binary     The location of the SUID binary executable
  *                                 (OPTIONAL (but required if $access_type
  *                                 is "SUID"); default none/empty string)
  * @param string  $suid_debug_file The location to send any SUID debug/error
  *                                 output (OPTIONAL; default none/empty string).
  * @param string  $master_username The username to send for an additional
  *                                 security check to the SUID backend; when
  *                                 NULL, no master username will be sent
  *                                 (OPTIONAL; default NULL).
  * @param string  $master_password The password to send for an additional
  *                                 security check to the SUID backend; when
  *                                 NULL, no master password will be sent
  *                                 (OPTIONAL; default NULL).
  * @param boolean $quiet           When TRUE, suppresses any error
  *                                 messages and makes this function
  *                                 return NULL without doing anything
  *                                 else (OPTIONAL; default = FALSE).
  * @param boolean $use_lstat       When TRUE, lstat() is used, wherein links
  *                                 themselves are stat'ed instead of their
  *                                 targets; when FALSE, use stat() instead
  *                                 (OPTIONAL; default = TRUE)
  *
  * @return mixed TRUE if the path exists and $return_info is FALSE,
  *               FALSE if the path does not exist or NULL if an error
  *               occurred and $quiet is TRUE, or when the path exists and
  *               $return_info is TRUE, an array containing the path's
  *               file/directory attributes is returned.  This array
  *               will contain possibly several different key-value pairs,
  *               each for a different file attribute, but should always
  *               at least contain a "name" key whose value is the file
  *               or directory name.  Other possible attributes are: "type"
  *               ("file", "directory", "link", "pipe", "socket"), "owner",
  *               "group", "size", "permissions", "date_modified" (date last
  *               modified), "date_accessed" (date last accessed),
  *               "date_changed" (date last changed), "link_saki" (link
  *               destination), "numeric_permissions".
  *
  */
function local_file_path_exists($access_type, $user, $pass, $path,
                                $return_info, $suid_binary='',
                                $suid_debug_file='', $master_username=NULL,
                                $master_password=NULL, $quiet=FALSE,
                                $use_lstat=TRUE)
{

   $error = '';
   while (strlen($path) > 1
    && ($path[strlen($path) - 1] == '/'
     || $path[strlen($path) - 1] == '\\'))
      $path = substr($path, 0, -1);


   // PHP file access
   //
   if ($access_type == 'PHP')
   {

      // if it doesn't exist, bail
      //
      if (!file_exists($path)) return FALSE;


      // if we don't need any details to return, we're done
      //
      if (!$return_info) return TRUE;


      // otherwise, need more info about the file/path
      //
      // NULL return means $quiet is TRUE, so we can return NULL ourselves
      //
      if (is_null($attributes = get_php_file_stat($path, $quiet, $use_lstat)))
         return NULL;

      return $attributes;

   }



   // set-uid wrapper file access
   //
   else if ($access_type == 'SUID')
   {

      // set up debugging output if needed
      //    
      if (!empty($suid_debug_file))
         $suid_debug = ' 2>>' . $suid_debug_file . ' ';
      else
         $suid_debug = '';


      // get a temp file for downloading the target file stats
      //
      $temp = ssb_get_temp_file($quiet, '', '', TRUE);
      if (is_null($temp)) // implied $quiet is TRUE
         return NULL;
      list($local_file, $FILE) = $temp;


      // now, actually stat the path
      // 
      $cmd = $suid_binary
           . " 'localhost' " // "server" argument isn't currently used - just use "localhost"
           . escapeshellarg($user)
           . ($use_lstat ? " 'lstat' " : " 'stat' ")
           . escapeshellarg($path)
           . ' ' . escapeshellarg($local_file)
           . $suid_debug;
         
      if (!($SUID = sq_call_function_suppress_errors('popen', array($cmd, 'w'))))
      {  
            
         // show command details when in debug mode
         // 
         sq_change_text_domain('server_settings_backend');
         global $sm_debug_mode, $server_settings_backend_debug;
         server_settings_backend_init();
         if ($server_settings_backend_debug) $sm_debug_mode |= SM_DEBUG_MODE_SIMPLE;
         if ($sm_debug_mode & SM_DEBUG_MODE_SIMPLE)
            $error = sprintf(_("Could not establish pipe to command \"%s\""), $cmd);
         else
            $error = _("Error establishing connection when attempting to inspect file");

         sq_change_text_domain('squirrelmail');

         @unlink($local_file);

      }
      else if (!is_null($master_username) && sq_call_function_suppress_errors('fwrite', array($SUID, $master_username . "\n")) === FALSE)
      {

         // show command details and password (CAREFUL!) when in debug mode
         //
         sq_change_text_domain('server_settings_backend');
         global $sm_debug_mode, $server_settings_backend_debug;
         server_settings_backend_init();
         if ($server_settings_backend_debug) $sm_debug_mode |= SM_DEBUG_MODE_SIMPLE;
         if ($sm_debug_mode & SM_DEBUG_MODE_SIMPLE)
            $error = sprintf(_("Could not send master username \"%s\" to pipe command \"%s\""), $master_username, $cmd);
         else
            $error = _("Unable to send master username when attempting to inspect file");

         sq_change_text_domain('squirrelmail');

         @unlink($local_file);

      }
      else if (!is_null($master_password) && sq_call_function_suppress_errors('fwrite', array($SUID, $master_password . "\n")) === FALSE)
      {

         // show command details and password (CAREFUL!) when in debug mode
         //
         sq_change_text_domain('server_settings_backend');
         global $sm_debug_mode, $server_settings_backend_debug;
         server_settings_backend_init();
         if ($server_settings_backend_debug) $sm_debug_mode |= SM_DEBUG_MODE_SIMPLE;
         if ($sm_debug_mode & SM_DEBUG_MODE_SIMPLE)
            $error = sprintf(_("Could not send master password \"%s\" to pipe command \"%s\""), $master_password, $cmd);
         else
            $error = _("Unable to send master password when attempting to inspect file");

         sq_change_text_domain('squirrelmail');

         @unlink($local_file);

      }
      else if (sq_call_function_suppress_errors('fwrite', array($SUID, $pass . "\n")) === FALSE) 
      {

         // show command details and password (CAREFUL!) when in debug mode
         //
         sq_change_text_domain('server_settings_backend');
         global $sm_debug_mode, $server_settings_backend_debug;
         server_settings_backend_init();
         if ($server_settings_backend_debug) $sm_debug_mode |= SM_DEBUG_MODE_SIMPLE;
         if ($sm_debug_mode & SM_DEBUG_MODE_SIMPLE)
            $error = sprintf(_("Could not send password \"%s\" to pipe command \"%s\""), $pass, $cmd);
         else
            $error = _("Unable to send password when attempting to inspect file");

         sq_change_text_domain('squirrelmail');

         @unlink($local_file);

      }
      else
      {

         // get command return code
         //
         $status = ssb_pclose($SUID);


         // if file-not-found, return FALSE
         //
         if ($status == SSB_ERR_NOTFOUND)
         {
            @unlink($local_file);
            return FALSE;
         }


         // any other error condition is more critical
         //
         if ($status != SSB_ERR_OK)
         {

            // show command details when in debug mode
            //
            sq_change_text_domain('server_settings_backend');
            global $sm_debug_mode, $server_settings_backend_debug;
            server_settings_backend_init();
            if ($server_settings_backend_debug) $sm_debug_mode |= SM_DEBUG_MODE_SIMPLE;
            if ($sm_debug_mode & SM_DEBUG_MODE_SIMPLE)
               $error = sprintf(_("Unable to retrieve path information from command \"%s\""), $cmd);
            else
               $error = sprintf(_("Unable to retrieve path information (%s)"), $status);

            sq_change_text_domain('squirrelmail');

            @unlink($local_file);

         }

         // at this point, we know something was found
         //
         else
         {

            if (!$return_info)
            {
               @unlink($local_file);
               return TRUE;
            }

            // otherwise, need to parse the stat information sent
            // from the SUID wrapper
            //
            $stat_info = file($local_file);

            // NULL return means $quiet is TRUE, so we can return NULL ourselves
            //
            if (is_null($attributes = get_suid_file_stat($stat_info, $quiet)))
            {
               @unlink($local_file);
               return NULL;
            }

            // done
            //
            @unlink($local_file);
            return $attributes;

         }

      }

   }



   // unknown file access type
   //
   else
   {
      sq_change_text_domain('server_settings_backend');
      $error = sprintf(_("Unknown file access type \"%s\""), $access_type);
      sq_change_text_domain('squirrelmail');
   }



   // spew error
   //
   if ($quiet) return NULL;
   global $color;
   $ret = plain_error_message($error, $color);
   if (check_sm_version (1, 5, 2))
   {
      echo $ret;
      global $oTemplate;
      $oTemplate->display('footer.tpl');
   }
   exit;

}



/**
  * Parses the stat() output from the SUID wrapper
  * into something we can use
  *
  * If some fatal error occurs due to misconfiguration or
  * backend failure, an error message is printed and execution
  * stops immediately, unless $quiet is TRUE.
  *
  * @param array $stat_info An array containing the raw stat()
  *                         output as returned by the SUID wrapper,
  *                         specifically in this order:
  *                         base name, path, mode, UID, GID, size,
  *                         atime, mtime, ctime
  *
  * @param boolean $quiet   When TRUE, suppresses any error
  *                         messages and makes this function
  *                         return NULL without doing anything
  *                         else (OPTIONAL; default = FALSE).
  *
  * @return mixed NULL when an error occurred and $quiet is TRUE, otherwise
  *               an array is returned containing several different key-value
  *               pairs, each for a different file attribute.  At a minimum,
  *               for any actual file or directory, it should always contain
  *               a "name" key whose value is the file or directory name.
  *               Other possible attributes are: "type" ("file", "directory",
  *               "link", "pipe", "socket"), "owner", "group", "size",
  *               "permissions", "date_modified" (date last modified),
  *               "date_accessed" (date last accessed), "date_changed"
  *               (date last changed), "link_saki" (link destination),
  *               "numeric_permissions".
  *
  */
function get_suid_file_stat($stat_info, $quiet=FALSE)
{

   $ret = array(
      'name'                => trim($stat_info[0]),
      'directory'           => trim($stat_info[1]),
      'numeric_permissions' => decoct($stat_info[2]),
      'raw_mode'            => $stat_info[2],
      'permissions'         => parse_permissions((int)$stat_info[2]),
      'size'                => $stat_info[5],
      'date_accessed'       => $stat_info[6],
      'date_modified'       => $stat_info[7],
      'date_changed'        => $stat_info[8],
      'uid'                 => $stat_info[3],
      'gid'                 => $stat_info[4],
   );


   switch (decoct($stat_info[2] & 0170000)) // mask off file encoding bit
   {
      case 10000: // 0010000
         $ret['type'] = 'pipe';
         break;
      case 20000: // 0020000
         $ret['type'] = 'char';
         break;
      case 40000: // 0040000
         $ret['type'] = 'directory';
         break;
      case 60000: // 0060000
         $ret['type'] = 'block';
         break;
      case 100000: // 0100000
         $ret['type'] = 'file';
         break;
      case 120000:  // won't ever be a link unless we use lstat()
         $ret['type'] = 'link';
         break;
      case 140000: // 0140000
         $ret['type'] = 'socket';
         break;
   }


   if (function_exists('posix_getpwuid')
    && ($owner = @posix_getpwuid($ret['uid']))
    && !empty($owner['name']))
      $ret['owner'] = $owner['name'];
   if (function_exists('posix_getgrgid')
    && ($group = @posix_getgrgid($ret['gid']))
    && !empty($group['name']))
      $ret['group'] = $group['name'];


   return $ret;

}



/**   
  * Parses PHP's stat() output into something we can use
  *
  * If some fatal error occurs due to misconfiguration or
  * backend failure, an error message is printed and execution
  * stops immediately, unless $quiet is TRUE.
  *   
  * @param string $path       The full path to the file/directory in question
  * @param boolean $quiet     When TRUE, suppresses any error
  *                           messages and makes this function
  *                           return NULL without doing anything
  *                           else (OPTIONAL; default = FALSE).
  * @param boolean $use_lstat When TRUE, use lstat(); when FALSE, use stat()
  *                           (OPTIONAL; default = TRUE)
  *
  * @return mixed NULL when an error occurred and $quiet is TRUE, otherwise
  *               an array is returned containing several different key-value
  *               pairs, each for a different file attribute.  At a minimum,
  *               for any actual file or directory, it should always contain
  *               a "name" key whose value is the file or directory name.
  *               Other possible attributes are: "type" ("file", "directory",
  *               "link", "pipe", "socket"), "owner", "group", "size",
  *               "permissions", "date_modified" (date last modified),
  *               "date_accessed" (date last accessed), "date_changed"
  *               (date last changed), "link_saki" (link destination),
  *               "numeric_permissions".
  *            
  */           
function get_php_file_stat($path, $quiet=FALSE, $use_lstat=TRUE)
{           

   if ($use_lstat)
      $attributes = @lstat($path);
   else
      $attributes = @stat($path);


   if (!$attributes)
      if ($quiet) return NULL;
      else
      {
         // spew error
         //
         sq_change_text_domain('server_settings_backend');
         global $sm_debug_mode, $server_settings_backend_debug;
         server_settings_backend_init();
         if ($server_settings_backend_debug) $sm_debug_mode |= SM_DEBUG_MODE_SIMPLE;
         if ($sm_debug_mode & SM_DEBUG_MODE_SIMPLE)
            $error = sprintf(_("Could not stat \"%s\""), $path);
         else
            $error = _("Could not stat file/directory");
         sq_change_text_domain('squirrelmail');
         global $color;
         $ret = plain_error_message($error, $color);
         if (check_sm_version (1, 5, 2))
         {
            echo $ret;
            global $oTemplate;
            $oTemplate->display('footer.tpl');
         }
         exit;
      }


   $ret = array(
      'name'                => basename($path),
      'directory'           => dirname($path),
      'numeric_permissions' => decoct($attributes[2]),
      'raw_mode'            => $attributes[2],
      'permissions'         => parse_permissions($attributes[2]),
      'size'                => $attributes[7],
      'date_accessed'       => $attributes[8],
      'date_modified'       => $attributes[9],
      'date_changed'        => $attributes[10],
      'uid'                 => $attributes[4],
      'gid'                 => $attributes[5],
   );


   switch (decoct($attributes[2] & 0170000)) // mask off file encoding bit
   {
      case 10000: // 0010000
         $ret['type'] = 'pipe';
         break;
      case 20000: // 0020000
         $ret['type'] = 'char';
         break;
      case 40000: // 0040000
         $ret['type'] = 'directory';
         break;
      case 60000: // 0060000
         $ret['type'] = 'block';
         break;
      case 100000: // 0100000
         $ret['type'] = 'file';
         break;
      case 120000:  // won't ever be a link unless we use lstat() above
         $ret['type'] = 'link';
         break;
      case 140000: // 0140000
         $ret['type'] = 'socket';
         break;
   }


   if (function_exists('posix_getpwuid')
    && ($owner = @posix_getpwuid($ret['uid']))
    && !empty($owner['name']))
      $ret['owner'] = $owner['name'];
   if (function_exists('posix_getgrgid')
    && ($group = @posix_getgrgid($ret['gid']))
    && !empty($group['name']))
      $ret['group'] = $group['name'];


   return $ret;

}



/**
  * Deletes a file or directory from the local server.  If the
  * file or directory does not exist, behavior will be as if it
  * was correctly deleted.
  *
  * If some fatal error occurs due to misconfiguration or
  * backend failure, an error message is printed and execution
  * stops immediately, unless $quiet is TRUE.
  *
  * @param string  $access_type     The method of file access, which
  *                                 should be either 'PHP' or 'SUID'.
  * @param string  $user            The username for whom to delete
  *                                 the file/directory.
  * @param string  $pass            The user's password.
  * @param string  $target          The full path to the file or directory
  *                                 to be removed.
  * @param boolean $is_directory    When TRUE, the target being deleted is
  *                                 assumed to be a directory.  When FALSE,
  *                                 it is assumed to be a file.  This
  *                                 information is determined by actually
  *                                 inspecting the target if at all possible,
  *                                 but if that is for some reason not possible,
  *                                 the value passed in here will be used (which
  *                                 will normally not be the case).
  * @param boolean $autodetect      When FALSE, $is_directory will be used
  *                                 unquestioningly.  When TRUE, the $path
  *                                 will be inspected to try to determine
  *                                 if it is a file, directory, etc.
  *                                 (OPTIONAL; default = TRUE)
  * @param string  $suid_binary     The location of the SUID binary executable
  *                                 (OPTIONAL (but required if $access_type
  *                                 is "SUID"); default none/empty string)
  * @param string  $suid_debug_file The location to send any SUID debug/error
  *                                 output (OPTIONAL; default none/empty string).
  * @param string  $master_username The username to send for an additional
  *                                 security check to the SUID backend; when
  *                                 NULL, no master username will be sent
  *                                 (OPTIONAL; default NULL).
  * @param string  $master_password The password to send for an additional
  *                                 security check to the SUID backend; when
  *                                 NULL, no master password will be sent
  *                                 (OPTIONAL; default NULL).
  * @param boolean $quiet           When TRUE, suppresses any error
  *                                 messages and makes this function
  *                                 return NULL without doing anything
  *                                 else (OPTIONAL; default = FALSE).
  *
  * @return mixed TRUE when completed normally, or NULL if an error
  *               occurred (and $quiet is TRUE).
  *
  */
//TODO -- removing directories only works if the directory is empty.... do we want to add a flag for recursive delete that will delete all directory contents (and subdirectories)?  if so, do it for the FTP backend, too
function local_file_delete($access_type, $user, $pass, $target, $is_directory,
                           $autodetect=TRUE, $suid_binary='', $suid_debug_file='',
                           $master_username=NULL, $master_password=NULL,
                           $quiet=FALSE)
{

   $error = '';
   while (strlen($target) > 1
    && ($target[strlen($target) - 1] == '/'
     || $target[strlen($target) - 1] == '\\'))
      $target = substr($target, 0, -1);


   // PHP file access
   //
   if ($access_type == 'PHP')
   {

      // grab information about target
      //
      $file_info = local_file_path_exists($access_type, $user, $pass,
                                          $target, TRUE, $suid_binary,
                                          $suid_debug_file, $master_username,
                                          $master_password, $quiet);
      if (is_null($file_info)) return NULL;  // pass along error ($quiet must be TRUE)


      // if file doesn't even exist in the first place, just bail -
      // our job is already done for us
      //
      if (!$file_info)
         return TRUE;


      // if possible, try to determine $is_directory on our own
      //
      if ($autodetect)
      {
         if (!empty($file_info['type']) && $file_info['type'] == 'directory')
            $is_directory = TRUE;
         else if (!empty($file_info['type']) && $file_info['type'] == 'file')
            $is_directory = FALSE;
         else if (!empty($file_info['type']))
            $is_directory = FALSE;  // for now, we'll assume we can delete other types too but can throw an error here if need be
      }


      // well, we'll cheat with autodetect even when told not to --
      // specifically, when the path is a link to a directory, we'll
      // allow the removal of the link
      //
      if (!$autodetect && $is_directory && !empty($file_info['type'])
       && $file_info['type'] == 'link')
         $is_directory = FALSE;


      if ($is_directory) $delete_function = 'rmdir';
      else $delete_function = 'unlink';


      // now, actually delete file/directory
      //
      if (empty($error)
       && !sq_call_function_suppress_errors($delete_function, array($target)))
      {
         if ($quiet) return NULL;

         sq_change_text_domain('server_settings_backend');
         global $sm_debug_mode, $server_settings_backend_debug;
         server_settings_backend_init();
         if ($server_settings_backend_debug) $sm_debug_mode |= SM_DEBUG_MODE_SIMPLE;
         if ($sm_debug_mode & SM_DEBUG_MODE_SIMPLE)
            $error = sprintf(_("Error deleting file or directory \"%s\""), $target);
         else
            $error = _("Error deleting file or directory");
         sq_change_text_domain('squirrelmail');
      }

   }


   // set-uid wrapper file access
   //
   else if ($access_type == 'SUID')
   {

      // set up debugging output if needed
      //
      if (!empty($suid_debug_file))
         $suid_debug = ' 2>>' . $suid_debug_file . ' ';
      else
         $suid_debug = '';


      // now, actually delete...
      //
      $cmd = $suid_binary
           . " 'localhost' " // "server" argument isn't currently used - just use "localhost"
           . escapeshellarg($user)
           . ($autodetect ? " 'delete_file_or_directory' "
                          : ($is_directory ? " 'delete_directory' " : " 'delete_file' "))
           . escapeshellarg($target)
           . $suid_debug;

      if (!($SUID = sq_call_function_suppress_errors('popen', array($cmd, 'w'))))
      {

         // show command details when in debug mode
         //
         sq_change_text_domain('server_settings_backend');
         global $sm_debug_mode, $server_settings_backend_debug;
         server_settings_backend_init();
         if ($server_settings_backend_debug) $sm_debug_mode |= SM_DEBUG_MODE_SIMPLE;
         if ($sm_debug_mode & SM_DEBUG_MODE_SIMPLE)
            $error = sprintf(_("Could not establish pipe to command \"%s\""), $cmd);
         else
            $error = _("Error establishing connection when attempting to delete file or directory");

         sq_change_text_domain('squirrelmail');

      }
      else if (!is_null($master_username) && sq_call_function_suppress_errors('fwrite', array($SUID, $master_username . "\n")) === FALSE)
      {

         // show command details and password (CAREFUL!) when in debug mode
         //
         sq_change_text_domain('server_settings_backend');
         global $sm_debug_mode, $server_settings_backend_debug;
         server_settings_backend_init();
         if ($server_settings_backend_debug) $sm_debug_mode |= SM_DEBUG_MODE_SIMPLE;
         if ($sm_debug_mode & SM_DEBUG_MODE_SIMPLE)
            $error = sprintf(_("Could not send master username \"%s\" to pipe command \"%s\""), $master_username, $cmd);
         else
            $error = _("Unable to send master username when attempting to delete file or directory");

         sq_change_text_domain('squirrelmail');

         @unlink($local_file);

      }
      else if (!is_null($master_password) && sq_call_function_suppress_errors('fwrite', array($SUID, $master_password . "\n")) === FALSE)
      {

         // show command details and password (CAREFUL!) when in debug mode
         //
         sq_change_text_domain('server_settings_backend');
         global $sm_debug_mode, $server_settings_backend_debug;
         server_settings_backend_init();
         if ($server_settings_backend_debug) $sm_debug_mode |= SM_DEBUG_MODE_SIMPLE;
         if ($sm_debug_mode & SM_DEBUG_MODE_SIMPLE)
            $error = sprintf(_("Could not send master password \"%s\" to pipe command \"%s\""), $master_password, $cmd);
         else
            $error = _("Unable to send master password when attempting to delete file or directory");

         sq_change_text_domain('squirrelmail');

         @unlink($local_file);

      }
      else if (sq_call_function_suppress_errors('fwrite', array($SUID, $pass . "\n")) === FALSE)
      {

         // show command details and password (CAREFUL!) when in debug mode
         //
         sq_change_text_domain('server_settings_backend');
         global $sm_debug_mode, $server_settings_backend_debug;
         server_settings_backend_init();
         if ($server_settings_backend_debug) $sm_debug_mode |= SM_DEBUG_MODE_SIMPLE;
         if ($sm_debug_mode & SM_DEBUG_MODE_SIMPLE)
            $error = sprintf(_("Could not send password \"%s\" to pipe command \"%s\""), $pass, $cmd);
         else
            $error = _("Unable to send password when attempting to delete file or directory");

         sq_change_text_domain('squirrelmail');

      }
      else
      {

         // get command return code
         //
         $status = ssb_pclose($SUID);

         // if file-not-found, our work is done for us
         //
         if ($status == SSB_ERR_NOTFOUND)
            return TRUE;

         // any other error condition is more critical
         //
         if ($status != SSB_ERR_OK)
         {

            // show command details when in debug mode
            //
            sq_change_text_domain('server_settings_backend');
            global $sm_debug_mode, $server_settings_backend_debug;
            server_settings_backend_init();
            if ($server_settings_backend_debug) $sm_debug_mode |= SM_DEBUG_MODE_SIMPLE;
            if ($sm_debug_mode & SM_DEBUG_MODE_SIMPLE)
               $error = sprintf(_("Unable to delete file or directory using command \"%s\""), $cmd);
            else
               $error = sprintf(_("Unable to delete file or directory (%s)"), $status);

            sq_change_text_domain('squirrelmail');

         }

      }

   }



   // unknown file access type
   //
   else
   {
      sq_change_text_domain('server_settings_backend');
      $error = sprintf(_("Unknown file access type \"%s\""), $access_type);
      sq_change_text_domain('squirrelmail');
   }



   // success
   //
   if (empty($error))
      return TRUE;


   // spew error
   //
   if ($quiet) return NULL;
   global $color;
   $ret = plain_error_message($error, $color);
   if (check_sm_version (1, 5, 2))
   {
      echo $ret;
      global $oTemplate;
      $oTemplate->display('footer.tpl');
   }
   exit;

}



/**
  * Retrieves a file from the local server.
  *
  * If some fatal error occurs due to misconfiguration or
  * backend failure, an error message is printed and execution
  * stops immediately, unless $quiet is TRUE.
  *
  * @param string  $access_type     The method of file access, which
  *                                 should be either 'PHP' or 'SUID'.
  * @param string  $user            The username for whom to retrieve
  *                                 the file.
  * @param string  $pass            The user's password.
  * @param string  $remote_file     The full path to the desired file
  *                                 (relative to the user's home
  *                                 directory in the case of the
  *                                 set-uid backend).
  * @param boolean $ignore_non_existent When TRUE, if the file does not exist,
  *                                     NULL is returned, when FALSE and the
  *                                     file does not exist, an error is
  *                                     triggered (OPTIONAL; default = FALSE).
  * @param string  $suid_binary     The location of the SUID binary executable
  *                                 (OPTIONAL (but required if $access_type
  *                                 is "SUID"); default none/empty string)
  * @param string  $suid_debug_file The location to send any SUID debug/error
  *                                 output (OPTIONAL; default none/empty string).
  * @param string  $master_username The username to send for an additional
  *                                 security check to the SUID backend; when
  *                                 NULL, no master username will be sent
  *                                 (OPTIONAL; default NULL).
  * @param string  $master_password The password to send for an additional
  *                                 security check to the SUID backend; when
  *                                 NULL, no master password will be sent
  *                                 (OPTIONAL; default NULL).
  * @param boolean $quiet           When TRUE, suppresses any error
  *                                 messages and makes this function
  *                                 return NULL without doing anything
  *                                 else (OPTIONAL; default = FALSE).
  *
  * @return mixed A string containing the full file path to the local
  *               copy of the file (typically in the SquirrelMail
  *               attachments directory) if it was retrieved normally,
  *               or NULL if an error occurred and $quiet is TRUE or
  *               an empty string if the file does not exist and
  *               $ignore_non_existent is TRUE.
  *
  */
function get_local_file($access_type, $user, $pass, $remote_file,
                        $ignore_non_existent=FALSE, $suid_binary='',
                        $suid_debug_file='', $master_username=NULL,
                        $master_password=NULL, $quiet=FALSE)
{

   $error = '';
   while (strlen($remote_file) > 1
    && ($remote_file[strlen($remote_file) - 1] == '/'
     || $remote_file[strlen($remote_file) - 1] == '\\'))
      $remote_file = substr($remote_file, 0, -1);


   // PHP file access
   // 
   if ($access_type == 'PHP')
   {     

      // deal with file-not-found
      //
      if (!is_file($remote_file))
      {

         // it might be OK that it wasn't found
         //
         if ($ignore_non_existent)
            return '';

         // otherwise, it's an error condition
         //
         if ($quiet)
            return NULL;

         sq_change_text_domain('server_settings_backend');
         global $sm_debug_mode, $server_settings_backend_debug;
         server_settings_backend_init();
         if ($server_settings_backend_debug) $sm_debug_mode |= SM_DEBUG_MODE_SIMPLE;
         if ($sm_debug_mode & SM_DEBUG_MODE_SIMPLE)
            $error = sprintf(_("Unable to find file \"%s\""), $remote_file);
         else
            $error = _("Unable to find file");
         sq_change_text_domain('squirrelmail');

      }


      // now, we merely need to ensure that the file is readable
      // 
      if (empty($error) && !is_readable($remote_file))
      {  
         if ($quiet) return NULL;

         sq_change_text_domain('server_settings_backend');
         global $sm_debug_mode, $server_settings_backend_debug;
         server_settings_backend_init();
         if ($server_settings_backend_debug) $sm_debug_mode |= SM_DEBUG_MODE_SIMPLE;
         if ($sm_debug_mode & SM_DEBUG_MODE_SIMPLE)
            $error = sprintf(_("Unable to read file \"%s\""), $remote_file);
         else
            $error = _("Unable to read file");
         sq_change_text_domain('squirrelmail');
      }
            
               
      // the original file path that was given is all we have to return
      //
      if (empty($error))
         return $remote_file;

   }


   
   // set-uid wrapper file access
   //
   else if ($access_type == 'SUID')
   {

      // get a temp file for downloading the target file
      //
      $temp = ssb_get_temp_file($quiet, '', '', TRUE);
      if (is_null($temp)) // implied $quiet is TRUE
         return NULL;
      list($local_file, $FILE) = $temp;


      // set up debugging output if needed
      //
      if (!empty($suid_debug_file))
         $suid_debug = ' 2>>' . $suid_debug_file . ' ';
      else
         $suid_debug = '';


      // now, actually pull file
      //
      $cmd = $suid_binary
           . " 'localhost' " // "server" argument isn't currently used - just use "localhost"
           . escapeshellarg($user)
           . " 'get_file' "
           . escapeshellarg($remote_file)
           . ' ' . escapeshellarg($local_file)
           . $suid_debug;

      if (!($SUID = sq_call_function_suppress_errors('popen', array($cmd, 'w'))))
      {

         // show command details when in debug mode
         //
         sq_change_text_domain('server_settings_backend');
         global $sm_debug_mode, $server_settings_backend_debug;
         server_settings_backend_init();
         if ($server_settings_backend_debug) $sm_debug_mode |= SM_DEBUG_MODE_SIMPLE;
         if ($sm_debug_mode & SM_DEBUG_MODE_SIMPLE)
            $error = sprintf(_("Could not establish pipe to command \"%s\""), $cmd);
         else
            $error = _("Error establishing connection when attempting to retrieve file");

         sq_change_text_domain('squirrelmail');

         @unlink($local_file);

      }
      else if (!is_null($master_username) && sq_call_function_suppress_errors('fwrite', array($SUID, $master_username . "\n")) === FALSE)
      {

         // show command details and password (CAREFUL!) when in debug mode
         //
         sq_change_text_domain('server_settings_backend');
         global $sm_debug_mode, $server_settings_backend_debug;
         server_settings_backend_init();
         if ($server_settings_backend_debug) $sm_debug_mode |= SM_DEBUG_MODE_SIMPLE;
         if ($sm_debug_mode & SM_DEBUG_MODE_SIMPLE)
            $error = sprintf(_("Could not send master username \"%s\" to pipe command \"%s\""), $master_username, $cmd);
         else
            $error = _("Unable to send master username when attempting to retrieve file");

         sq_change_text_domain('squirrelmail');

         @unlink($local_file);

      }
      else if (!is_null($master_password) && sq_call_function_suppress_errors('fwrite', array($SUID, $master_password . "\n")) === FALSE)
      {

         // show command details and password (CAREFUL!) when in debug mode
         //
         sq_change_text_domain('server_settings_backend');
         global $sm_debug_mode, $server_settings_backend_debug;
         server_settings_backend_init();
         if ($server_settings_backend_debug) $sm_debug_mode |= SM_DEBUG_MODE_SIMPLE;
         if ($sm_debug_mode & SM_DEBUG_MODE_SIMPLE)
            $error = sprintf(_("Could not send master password \"%s\" to pipe command \"%s\""), $master_password, $cmd);
         else
            $error = _("Unable to send master password when attempting to retrieve file");

         sq_change_text_domain('squirrelmail');

         @unlink($local_file);

      }
      else if (sq_call_function_suppress_errors('fwrite', array($SUID, $pass . "\n")) === FALSE)
      {

         // show command details and password (CAREFUL!) when in debug mode
         //
         sq_change_text_domain('server_settings_backend');
         global $sm_debug_mode, $server_settings_backend_debug;
         server_settings_backend_init();
         if ($server_settings_backend_debug) $sm_debug_mode |= SM_DEBUG_MODE_SIMPLE;
         if ($sm_debug_mode & SM_DEBUG_MODE_SIMPLE)
            $error = sprintf(_("Could not send password \"%s\" to pipe command \"%s\""), $pass, $cmd);
         else
            $error = _("Unable to send password when attempting to retrieve file");

         sq_change_text_domain('squirrelmail');

         @unlink($local_file);

      }
      else
      {

         // get command return code
         //
         $status = ssb_pclose($SUID);

         // if file-not-found, it might be OK
         //
         if ($status == SSB_ERR_NOTFOUND && $ignore_non_existent)
         {
            @unlink($local_file);
            return '';
         }

         // any other error condition is more critical
         //
         if ($status != SSB_ERR_OK)
         {

            // show command details when in debug mode
            //
            sq_change_text_domain('server_settings_backend');
            global $sm_debug_mode, $server_settings_backend_debug;
            server_settings_backend_init();
            if ($server_settings_backend_debug) $sm_debug_mode |= SM_DEBUG_MODE_SIMPLE;
            if ($sm_debug_mode & SM_DEBUG_MODE_SIMPLE)
               $error = sprintf(_("Unable to retrieve file from command \"%s\""), $cmd);
            else
               $error = sprintf(_("Unable to retrieve file (%s)"), $status);

            sq_change_text_domain('squirrelmail');

            @unlink($local_file);

         }

      }


      // the original file path that was given is all we have to return
      //
      if (empty($error))
         return $local_file;

   }



   // unknown file access type
   //
   else
   {
      sq_change_text_domain('server_settings_backend');
      $error = sprintf(_("Unknown file access type \"%s\""), $access_type);
      sq_change_text_domain('squirrelmail');
   }



   // spew error
   //
   if ($quiet) return NULL;
   global $color;
   $ret = plain_error_message($error, $color);
   if (check_sm_version (1, 5, 2))
   {
      echo $ret;
      global $oTemplate;
      $oTemplate->display('footer.tpl');
   }
   exit;

}



/**
  * Retrieves file listing of a local directory on the server.
  *
  * If some fatal error occurs due to misconfiguration or
  * backend failure, an error message is printed and execution
  * stops immediately, unless $quiet is TRUE.
  *
  * @param string  $access_type     The method of file access, which
  *                                 should be either 'PHP' or 'SUID'.
  * @param string  $user            The username to log in with.
  * @param string  $pass            The password to log in with.
  * @param string  $directory       The directory to get the listing of.
  * @param string  $suid_binary     The location of the SUID binary executable
  *                                 (OPTIONAL (but required if $access_type
  *                                 is "SUID"); default none/empty string)
  * @param string  $suid_debug_file The location to send any SUID debug/error
  *                                 output (OPTIONAL; default none/empty string).
  * @param string  $master_username The username to send for an additional
  *                                 security check to the SUID backend; when
  *                                 NULL, no master username will be sent
  *                                 (OPTIONAL; default NULL).
  * @param string  $master_password The password to send for an additional
  *                                 security check to the SUID backend; when
  *                                 NULL, no master password will be sent
  *                                 (OPTIONAL; default NULL).
  * @param boolean $quiet           When TRUE, suppresses any error
  *                                 messages and makes this function
  *                                 return NULL without doing anything
  *                                 else (OPTIONAL; default = FALSE).
  * @param boolean $use_lstat       When TRUE, lstat() is used, wherein links
  *                                 themselves are stat'ed instead of their
  *                                 targets; when FALSE, use stat() instead
  *                                 (OPTIONAL; default = TRUE)
  *
  * @return mixed NULL when an error occurred and $quiet is TRUE, otherwise
  *               when the listing was retrieved normally, an array is
  *               returned containing sub-arrays which each represent a
  *               file or directory in the listing.  These sub-arrays
  *               will contain possibly several different key-value
  *               pairs, each for a different file attribute, but should
  *               always at least contain a "name" key whose value is
  *               the file or directory name.  Other possible attributes
  *               are: "type" ("file", "directory", "link", "pipe",
  *               "socket"), "owner", "group", "size", "permissions",
  *               "date_modified" (date last modified), "date_accessed"
  *               (date last accessed), "date_changed" (date last changed),
  *               "link_saki" (link destination), "numeric_permissions".
  *
  */
function get_local_directory_listing($access_type, $user, $pass, $directory,
                                     $suid_binary='', $suid_debug_file='',
                                     $master_username=NULL, $master_password=NULL,
                                     $quiet=FALSE, $use_lstat=TRUE)
{

   $directory_listing = array();
   $error = '';
   while (strlen($directory) > 1
    && ($directory[strlen($directory) - 1] == '/'
     || $directory[strlen($directory) - 1] == '\\'))
      $directory = substr($directory, 0, -1);


   // PHP file access
   // 
   if ($access_type == 'PHP')
   {     

      // open directory (NB: no need to test is_dir() because opendir() will fail anyway
      //
      if (!($DIR = sq_call_function_suppress_errors('opendir', array($directory))))
      {
         if ($quiet) return NULL;

         sq_change_text_domain('server_settings_backend');
         global $sm_debug_mode, $server_settings_backend_debug;
         server_settings_backend_init();
         if ($server_settings_backend_debug) $sm_debug_mode |= SM_DEBUG_MODE_SIMPLE;
         if ($sm_debug_mode & SM_DEBUG_MODE_SIMPLE)
            $error = sprintf(_("Failed to retrieve file list of \"%s\""), $directory);
         else
            $error = _("Error retrieving file list");
         sq_change_text_domain('squirrelmail');
      }


      // read in directory
      //
      while (empty($error) && ($file = readdir($DIR)) !== FALSE)
         $directory_listing[] = $file;
      sq_call_function_suppress_errors('closedir', array($DIR));


      // iterate through the listing and get detailed stats
      //
      if (empty($error))
      {

         // iterate through directory
         //
         $directory_listing_full_details = array();
         foreach ($directory_listing as $entry)
         {

            $entry = trim($entry);


            $file_info = local_file_path_exists($access_type, $user, $pass,
                                                $directory . '/' . $entry,
                                                TRUE, '', '', NULL, NULL,
                                                $quiet, $use_lstat);

            // pass along error ($quiet must be TRUE)
            //
            if (is_null($file_info)) return NULL;


            // permissions issues getting more details?
            //
            if (!$file_info)
            {
               if ($quiet) return NULL;

               sq_change_text_domain('server_settings_backend');
               global $sm_debug_mode, $server_settings_backend_debug;
               server_settings_backend_init();
               if ($server_settings_backend_debug) $sm_debug_mode |= SM_DEBUG_MODE_SIMPLE;
               if ($sm_debug_mode & SM_DEBUG_MODE_SIMPLE)
                  $error = sprintf(_("Error retrieving information about file or directory \"%s\" in \"%s\""), $entry, $directory);
               else
                  $error = _("Error retrieving information about file or directory");
               sq_change_text_domain('squirrelmail');
               break;
            }

            else $directory_listing_full_details[] = $file_info;

         }


         // success - return the listing
         //
         if (empty($error))
            return $directory_listing_full_details;

      }

   }


   
   // set-uid wrapper file access
   //
   else if ($access_type == 'SUID')
   {

      // get a temp file for downloading the directory listing
      //
      $temp = ssb_get_temp_file($quiet, '', '', TRUE);
      if (is_null($temp)) // implied $quiet is TRUE
         return NULL;
      list($local_file, $FILE) = $temp;


      // set up debugging output if needed
      //
      if (!empty($suid_debug_file))
         $suid_debug = ' 2>>' . $suid_debug_file . ' ';
      else
         $suid_debug = '';


      // now, actually get the listing
      //
      $cmd = $suid_binary
           . " 'localhost' " // "server" argument isn't currently used - just use "localhost"
           . escapeshellarg($user)
           . ($use_lstat ? " 'list_directory_detailed_lstat' "
                         : " 'list_directory_detailed_stat' ")
           . escapeshellarg($directory)
           . ' ' . escapeshellarg($local_file)
           . $suid_debug;


      if (!($SUID = sq_call_function_suppress_errors('popen', array($cmd, 'w'))))
      {

         // show command details when in debug mode
         //
         sq_change_text_domain('server_settings_backend');
         global $sm_debug_mode, $server_settings_backend_debug;
         server_settings_backend_init();
         if ($server_settings_backend_debug) $sm_debug_mode |= SM_DEBUG_MODE_SIMPLE;
         if ($sm_debug_mode & SM_DEBUG_MODE_SIMPLE)
            $error = sprintf(_("Could not establish pipe to command \"%s\""), $cmd);
         else
            $error = _("Error establishing connection when attempting to retrieve directory listing");

         sq_change_text_domain('squirrelmail');

         @unlink($local_file);

      }
      else if (!is_null($master_username) && sq_call_function_suppress_errors('fwrite', array($SUID, $master_username . "\n")) === FALSE)
      {

         // show command details and password (CAREFUL!) when in debug mode
         //
         sq_change_text_domain('server_settings_backend');
         global $sm_debug_mode, $server_settings_backend_debug;
         server_settings_backend_init();
         if ($server_settings_backend_debug) $sm_debug_mode |= SM_DEBUG_MODE_SIMPLE;
         if ($sm_debug_mode & SM_DEBUG_MODE_SIMPLE)
            $error = sprintf(_("Could not send master username \"%s\" to pipe command \"%s\""), $master_username, $cmd);
         else
            $error = _("Unable to send master username when attempting to retrieve directory listing");

         sq_change_text_domain('squirrelmail');

         @unlink($local_file);

      }
      else if (!is_null($master_password) && sq_call_function_suppress_errors('fwrite', array($SUID, $master_password . "\n")) === FALSE)
      {

         // show command details and password (CAREFUL!) when in debug mode
         //
         sq_change_text_domain('server_settings_backend');
         global $sm_debug_mode, $server_settings_backend_debug;
         server_settings_backend_init();
         if ($server_settings_backend_debug) $sm_debug_mode |= SM_DEBUG_MODE_SIMPLE;
         if ($sm_debug_mode & SM_DEBUG_MODE_SIMPLE)
            $error = sprintf(_("Could not send master password \"%s\" to pipe command \"%s\""), $master_password, $cmd);
         else
            $error = _("Unable to send master password when attempting to retrieve directory listing");

         sq_change_text_domain('squirrelmail');

         @unlink($local_file);

      }
      else if (sq_call_function_suppress_errors('fwrite', array($SUID, $pass . "\n")) === FALSE)
      {

         // show command details and password (CAREFUL!) when in debug mode
         //
         sq_change_text_domain('server_settings_backend');
         global $sm_debug_mode, $server_settings_backend_debug;
         server_settings_backend_init();
         if ($server_settings_backend_debug) $sm_debug_mode |= SM_DEBUG_MODE_SIMPLE;
         if ($sm_debug_mode & SM_DEBUG_MODE_SIMPLE)
            $error = sprintf(_("Could not send password \"%s\" to pipe command \"%s\""), $pass, $cmd);
         else
            $error = _("Unable to send password when attempting to retrieve directory listing");

         sq_change_text_domain('squirrelmail');

         @unlink($local_file);

      }
      else
      {

         // get command return code
         //
         $status = ssb_pclose($SUID);


//TODO: return FALSE instead?
         // if file-not-found, return NULL
         //
         if ($status == SSB_ERR_NOTFOUND)
         {
            @unlink($local_file);
            return NULL;
         }


         // any other error condition is more critical
         //
         if ($status != SSB_ERR_OK)
         {

            // show command details when in debug mode
            //
            sq_change_text_domain('server_settings_backend');
            global $sm_debug_mode, $server_settings_backend_debug;
            server_settings_backend_init();
            if ($server_settings_backend_debug) $sm_debug_mode |= SM_DEBUG_MODE_SIMPLE;
            if ($sm_debug_mode & SM_DEBUG_MODE_SIMPLE)
               $error = sprintf(_("Unable to retrieve directory listing from command \"%s\""), $cmd);
            else
               $error = sprintf(_("Unable to retrieve directory listing (%s)"), $status);

            sq_change_text_domain('squirrelmail');

            @unlink($local_file);

         }

      }


      // need to grab listing from the local file
      //
      if (empty($error))
      {

         $directory_listing = file($local_file);
         @unlink($local_file);

         // now, parse the results
         //
         $directory_listing_full_details = array();
         foreach ($directory_listing as $entry)
         {
            $entry = explode('||', $entry);
            $attributes = get_suid_file_stat(array($entry[8], $entry[7],
                                                   $entry[0], $entry[1],
                                                   $entry[2], $entry[3],
                                                   $entry[4], $entry[5],
                                                   $entry[6]),
                                             $quiet);

            if (is_null($attributes)) return NULL;  // $quiet must be NULL

            $directory_listing_full_details[] = $attributes;
         }

         // success
         //
         return $directory_listing_full_details;

      }

   }



   // unknown file access type
   //
   else
   {
      sq_change_text_domain('server_settings_backend');
      $error = sprintf(_("Unknown file access type \"%s\""), $access_type);
      sq_change_text_domain('squirrelmail');
   }



   // spew error
   //
   if ($quiet) return NULL;
   global $color;
   $ret = plain_error_message($error, $color);
   if (check_sm_version (1, 5, 2))
   {
      echo $ret;
      global $oTemplate;
      $oTemplate->display('footer.tpl');
   }
   exit;

}



/**
  * Changes permissions on a file or directory on the server.
  *
  * If some fatal error occurs due to misconfiguration or
  * backend failure, an error message is printed and execution
  * stops immediately, unless $quiet is TRUE.
  *
  * @param string  $access_type     The method of file access, which
  *                                 should be either 'PHP' or 'SUID'.
  * @param string  $user            The username to log in with.
  * @param string  $pass            The password to log in with.
  * @param string  $path            The path to the file or directory to
  *                                 change permissions on.
  * @param int     $mode            The octal permissions number representing
  *                                 the permissions to apply.
  * @param string  $suid_binary     The location of the SUID binary executable
  *                                 (OPTIONAL (but required if $access_type
  *                                 is "SUID"); default none/empty string)
  * @param string  $suid_debug_file The location to send any SUID debug/error
  *                                 output (OPTIONAL; default none/empty string).
  * @param string  $master_username The username to send for an additional
  *                                 security check to the SUID backend; when
  *                                 NULL, no master username will be sent
  *                                 (OPTIONAL; default NULL).
  * @param string  $master_password The password to send for an additional
  *                                 security check to the SUID backend; when
  *                                 NULL, no master password will be sent
  *                                 (OPTIONAL; default NULL).
  * @param boolean $quiet           When TRUE, suppresses any error
  *                                 messages and makes this function
  *                                 return NULL without doing anything
  *                                 else (OPTIONAL; default = FALSE).
  *
  * @return mixed TRUE when completed normally, or NULL if an error
  *               occurred (and $quiet is TRUE).
  *
  */
function chmod_local_file_or_directory($access_type, $user, $pass, $path, $mode,
                                       $suid_binary='', $suid_debug_file='',
                                       $master_username=NULL, $master_password=NULL,
                                       $quiet=FALSE)
{

   $error = '';
   while (strlen($path) > 1
    && ($path[strlen($path) - 1] == '/'
     || $path[strlen($path) - 1] == '\\'))
      $path = substr($path, 0, -1);


   // PHP file access
   // 
   if ($access_type == 'PHP')
   {     
   
      // attempt to change permissions
      // 
      if (!sq_call_function_suppress_errors('chmod', array($path, $mode)))
      {
         if ($quiet) return NULL;
         
         sq_change_text_domain('server_settings_backend');
         global $sm_debug_mode, $server_settings_backend_debug;
         server_settings_backend_init();
         if ($server_settings_backend_debug) $sm_debug_mode |= SM_DEBUG_MODE_SIMPLE;
         if ($sm_debug_mode & SM_DEBUG_MODE_SIMPLE)
            $error = sprintf(_("Failed to change permissions on \"%s\""), $path);
         else
            $error = _("Error changing permissions");
         sq_change_text_domain('squirrelmail');
      }

   }



   // set-uid wrapper file access
   //
   else if ($access_type == 'SUID')
   {

      // set up debugging output if needed
      //
      if (!empty($suid_debug_file))
         $suid_debug = ' 2>>' . $suid_debug_file . ' ';
      else
         $suid_debug = '';


      // now, actually change permissions...
      //
      $cmd = $suid_binary
           . " 'localhost' " // "server" argument isn't currently used - just use "localhost"
           . escapeshellarg($user)
           . " 'change_permissions' "
           . escapeshellarg($path) . ' '
           . escapeshellarg($mode)
           . $suid_debug;

      if (!($SUID = sq_call_function_suppress_errors('popen', array($cmd, 'w'))))
      {

         // show command details when in debug mode
         //
         sq_change_text_domain('server_settings_backend');
         global $sm_debug_mode, $server_settings_backend_debug;
         server_settings_backend_init();
         if ($server_settings_backend_debug) $sm_debug_mode |= SM_DEBUG_MODE_SIMPLE;
         if ($sm_debug_mode & SM_DEBUG_MODE_SIMPLE)
            $error = sprintf(_("Could not establish pipe to command \"%s\""), $cmd);
         else
            $error = _("Error establishing connection when attempting to change permissions on file or directory");

         sq_change_text_domain('squirrelmail');

      }
      else if (!is_null($master_username) && sq_call_function_suppress_errors('fwrite', array($SUID, $master_username . "\n")) === FALSE)
      {

         // show command details and password (CAREFUL!) when in debug mode
         //
         sq_change_text_domain('server_settings_backend');
         global $sm_debug_mode, $server_settings_backend_debug;
         server_settings_backend_init();
         if ($server_settings_backend_debug) $sm_debug_mode |= SM_DEBUG_MODE_SIMPLE;
         if ($sm_debug_mode & SM_DEBUG_MODE_SIMPLE)
            $error = sprintf(_("Could not send master username \"%s\" to pipe command \"%s\""), $master_username, $cmd);
         else
            $error = _("Unable to send master username when attempting to change permissions on file or directory");

         sq_change_text_domain('squirrelmail');

         @unlink($local_file);

      }
      else if (!is_null($master_password) && sq_call_function_suppress_errors('fwrite', array($SUID, $master_password . "\n")) === FALSE)
      {

         // show command details and password (CAREFUL!) when in debug mode
         //
         sq_change_text_domain('server_settings_backend');
         global $sm_debug_mode, $server_settings_backend_debug;
         server_settings_backend_init();
         if ($server_settings_backend_debug) $sm_debug_mode |= SM_DEBUG_MODE_SIMPLE;
         if ($sm_debug_mode & SM_DEBUG_MODE_SIMPLE)
            $error = sprintf(_("Could not send master password \"%s\" to pipe command \"%s\""), $master_password, $cmd);
         else
            $error = _("Unable to send master password when attempting to change permissions on file or directory");

         sq_change_text_domain('squirrelmail');

         @unlink($local_file);

      }
      else if (sq_call_function_suppress_errors('fwrite', array($SUID, $pass . "\n")) === FALSE)
      {

         // show command details and password (CAREFUL!) when in debug mode
         //
         sq_change_text_domain('server_settings_backend');
         global $sm_debug_mode, $server_settings_backend_debug;
         server_settings_backend_init();
         if ($server_settings_backend_debug) $sm_debug_mode |= SM_DEBUG_MODE_SIMPLE;
         if ($sm_debug_mode & SM_DEBUG_MODE_SIMPLE)
            $error = sprintf(_("Could not send password \"%s\" to pipe command \"%s\""), $pass, $cmd);
         else
            $error = _("Unable to send password when attempting to change permissions on file or directory");

         sq_change_text_domain('squirrelmail');

      }
      else
      {

         // get command return code
         //
         $status = ssb_pclose($SUID);


         // any error condition is critical
         //
         if ($status != SSB_ERR_OK)
         {

            // show command details when in debug mode
            //
            sq_change_text_domain('server_settings_backend');
            global $sm_debug_mode, $server_settings_backend_debug;
            server_settings_backend_init();
            if ($server_settings_backend_debug) $sm_debug_mode |= SM_DEBUG_MODE_SIMPLE;
            if ($sm_debug_mode & SM_DEBUG_MODE_SIMPLE)
               $error = sprintf(_("Unable to change permissions on file or directory using command \"%s\""), $cmd);
            else
               $error = sprintf(_("Unable to change permissions on file or directory (%s)"), $status);

            sq_change_text_domain('squirrelmail');

         }

      }

   }



   // unknown file access type
   //
   else
   {
      sq_change_text_domain('server_settings_backend');
      $error = sprintf(_("Unknown file access type \"%s\""), $access_type);
      sq_change_text_domain('squirrelmail');
   }



   // success
   //
   if (empty($error))
      return TRUE;



   // spew error
   //
   if ($quiet) return NULL;
   global $color;
   $ret = plain_error_message($error, $color);
   if (check_sm_version (1, 5, 2))
   {
      echo $ret;
      global $oTemplate;
      $oTemplate->display('footer.tpl');
   }
   exit;

}


 
/**
  * Creates a directory on the server.
  *
  * If some fatal error occurs due to misconfiguration or
  * backend failure, an error message is printed and execution
  * stops immediately, unless $quiet is TRUE.
  *
  * @param string  $access_type     The method of file access, which
  *                                 should be either 'PHP' or 'SUID'.
  * @param string  $user            The username to log in with.
  * @param string  $pass            The password to log in with.
  * @param string  $directory_path  The directory path to create.
  * @param int     $mode            The octal permissions number to
  *                                 apply to the newly created directory.
  * @param string  $suid_binary     The location of the SUID binary executable
  *                                 (OPTIONAL (but required if $access_type
  *                                 is "SUID"); default none/empty string)
  * @param string  $suid_debug_file The location to send any SUID debug/error
  *                                 output (OPTIONAL; default none/empty string).
  * @param string  $master_username The username to send for an additional
  *                                 security check to the SUID backend; when
  *                                 NULL, no master username will be sent
  *                                 (OPTIONAL; default NULL).
  * @param string  $master_password The password to send for an additional
  *                                 security check to the SUID backend; when
  *                                 NULL, no master password will be sent
  *                                 (OPTIONAL; default NULL).
  * @param boolean $quiet           When TRUE, suppresses any error
  *                                 messages and makes this function
  *                                 return NULL without doing anything
  *                                 else (OPTIONAL; default = FALSE).
  *
  * @return mixed TRUE when completed normally, or NULL if an error
  *               occurred (and $quiet is TRUE).
  *
  */
function local_file_mkdir($access_type, $user, $pass, $directory_path, $mode,
                          $suid_binary='', $suid_debug_file='',
                          $master_username=NULL, $master_password=NULL, $quiet=FALSE)
{

   $error = '';
   while (strlen($directory_path) > 1
    && ($directory_path[strlen($directory_path) - 1] == '/'
     || $directory_path[strlen($directory_path) - 1] == '\\'))
      $directory_path = substr($directory_path, 0, -1);


   // PHP file access
   // 
   if ($access_type == 'PHP')
   {     
   
      // attempt to create directory
      // 
      if (!sq_call_function_suppress_errors('mkdir', array($directory_path, $mode)))
      {
         if ($quiet) return NULL;
         
         sq_change_text_domain('server_settings_backend');
         global $sm_debug_mode, $server_settings_backend_debug;
         server_settings_backend_init();
         if ($server_settings_backend_debug) $sm_debug_mode |= SM_DEBUG_MODE_SIMPLE;
         if ($sm_debug_mode & SM_DEBUG_MODE_SIMPLE)
            $error = sprintf(_("Failed to create directory \"%s\""), $directory_path);
         else
            $error = _("Error creating directory");
         sq_change_text_domain('squirrelmail');
      }

   }



   // set-uid wrapper file access
   //
   else if ($access_type == 'SUID')
   {

      // set up debugging output if needed
      //
      if (!empty($suid_debug_file))
         $suid_debug = ' 2>>' . $suid_debug_file . ' ';
      else
         $suid_debug = '';


      // now, actually create the directory...
      //
      $cmd = $suid_binary
           . " 'localhost' " // "server" argument isn't currently used - just use "localhost"
           . escapeshellarg($user)
           . " 'mkdir' "
           // TODO: this is how to do it as mkdir -p (recursively)
           //. " 'mkdir_recursive' "
           . escapeshellarg($directory_path) . ' '
           . escapeshellarg($mode)
           . $suid_debug;

      if (!($SUID = sq_call_function_suppress_errors('popen', array($cmd, 'w'))))
      {

         // show command details when in debug mode
         //
         sq_change_text_domain('server_settings_backend');
         global $sm_debug_mode, $server_settings_backend_debug;
         server_settings_backend_init();
         if ($server_settings_backend_debug) $sm_debug_mode |= SM_DEBUG_MODE_SIMPLE;
         if ($sm_debug_mode & SM_DEBUG_MODE_SIMPLE)
            $error = sprintf(_("Could not establish pipe to command \"%s\""), $cmd);
         else
            $error = _("Error establishing connection when attempting to create directory");

         sq_change_text_domain('squirrelmail');

      }
      else if (!is_null($master_username) && sq_call_function_suppress_errors('fwrite', array($SUID, $master_username . "\n")) === FALSE)
      {

         // show command details and password (CAREFUL!) when in debug mode
         //
         sq_change_text_domain('server_settings_backend');
         global $sm_debug_mode, $server_settings_backend_debug;
         server_settings_backend_init();
         if ($server_settings_backend_debug) $sm_debug_mode |= SM_DEBUG_MODE_SIMPLE;
         if ($sm_debug_mode & SM_DEBUG_MODE_SIMPLE)
            $error = sprintf(_("Could not send master username \"%s\" to pipe command \"%s\""), $master_username, $cmd);
         else
            $error = _("Unable to send master username when attempting to create directory");

         sq_change_text_domain('squirrelmail');

         @unlink($local_file);

      }
      else if (!is_null($master_password) && sq_call_function_suppress_errors('fwrite', array($SUID, $master_password . "\n")) === FALSE)
      {

         // show command details and password (CAREFUL!) when in debug mode
         //
         sq_change_text_domain('server_settings_backend');
         global $sm_debug_mode, $server_settings_backend_debug;
         server_settings_backend_init();
         if ($server_settings_backend_debug) $sm_debug_mode |= SM_DEBUG_MODE_SIMPLE;
         if ($sm_debug_mode & SM_DEBUG_MODE_SIMPLE)
            $error = sprintf(_("Could not send master password \"%s\" to pipe command \"%s\""), $master_password, $cmd);
         else
            $error = _("Unable to send master password when attempting to create directory");

         sq_change_text_domain('squirrelmail');

         @unlink($local_file);

      }
      else if (sq_call_function_suppress_errors('fwrite', array($SUID, $pass . "\n")) === FALSE)
      {

         // show command details and password (CAREFUL!) when in debug mode
         //
         sq_change_text_domain('server_settings_backend');
         global $sm_debug_mode, $server_settings_backend_debug;
         server_settings_backend_init();
         if ($server_settings_backend_debug) $sm_debug_mode |= SM_DEBUG_MODE_SIMPLE;
         if ($sm_debug_mode & SM_DEBUG_MODE_SIMPLE)
            $error = sprintf(_("Could not send password \"%s\" to pipe command \"%s\""), $pass, $cmd);
         else
            $error = _("Unable to send password when attempting to create directory");

         sq_change_text_domain('squirrelmail');

      }
      else
      {

         // get command return code
         //
         $status = ssb_pclose($SUID);


         // any error condition is critical
         //
         if ($status != SSB_ERR_OK)
         {

            // show command details when in debug mode
            //
            sq_change_text_domain('server_settings_backend');
            global $sm_debug_mode, $server_settings_backend_debug;
            server_settings_backend_init();
            if ($server_settings_backend_debug) $sm_debug_mode |= SM_DEBUG_MODE_SIMPLE;
            if ($sm_debug_mode & SM_DEBUG_MODE_SIMPLE)
               $error = sprintf(_("Unable to create directory using command \"%s\""), $cmd);
            else
               $error = sprintf(_("Unable to create directory (%s)"), $status);

            sq_change_text_domain('squirrelmail');

         }

      }

   }


   // unknown file access type
   //
   else
   {
      sq_change_text_domain('server_settings_backend');
      $error = sprintf(_("Unknown file access type \"%s\""), $access_type);
      sq_change_text_domain('squirrelmail');
   }


   // success
   //
   if (empty($error))
      return TRUE;


   // spew error
   //
   if ($quiet) return NULL;
   global $color;
   $ret = plain_error_message($error, $color);
   if (check_sm_version (1, 5, 2))
   {
      echo $ret;
      global $oTemplate;
      $oTemplate->display('footer.tpl');
   }
   exit;

}



/**
  * Rename/move a file or directory on the server.
  *
  * If some fatal error occurs due to misconfiguration or
  * backend failure, an error message is printed and execution
  * stops immediately, unless $quiet is TRUE.
  *
  * @param string  $access_type     The method of file access, which
  *                                 should be either 'PHP' or 'SUID'.
  * @param string  $user            The username to log in with.
  * @param string  $pass            The password to log in with.
  * @param string  $source          The source (remote) file/directory
  *                                 name.
  * @param string  $destination     The destination (remote)
  *                                 file/directory path.
  * @param boolean $allow_overwrite When TRUE, any existing file at
  *                                 $destination will be overwritten,
  *                                 otherwise, an existing file will
  *                                 create an error.
  * @param string  $suid_binary     The location of the SUID binary executable
  *                                 (OPTIONAL (but required if $access_type
  *                                 is "SUID"); default none/empty string)
  * @param string  $suid_debug_file The location to send any SUID debug/error
  *                                 output (OPTIONAL; default none/empty string).
  * @param string  $master_username The username to send for an additional
  *                                 security check to the SUID backend; when
  *                                 NULL, no master username will be sent
  *                                 (OPTIONAL; default NULL).
  * @param string  $master_password The password to send for an additional
  *                                 security check to the SUID backend; when
  *                                 NULL, no master password will be sent
  *                                 (OPTIONAL; default NULL).
  * @param boolean $quiet           When TRUE, suppresses any error
  *                                 messages and makes this function
  *                                 return NULL without doing anything
  *                                 else (OPTIONAL; default = FALSE).
  *
  * @return mixed TRUE when completed normally, or NULL if an error
  *               occurred (and $quiet is TRUE).
  *
  */
function local_file_rename($access_type, $user, $pass, $source, $destination,
                           $allow_overwrite, $suid_binary='',
                           $suid_debug_file='', $master_username=NULL,
                           $master_password=NULL, $quiet=FALSE)
{

   $error = '';
   $cwd = '.';
   while (strlen($source) > 1
    && ($source[strlen($source) - 1] == '/'
     || $source[strlen($source) - 1] == '\\'))
      $source = substr($source, 0, -1);
   while (strlen($destination) > 1
    && ($destination[strlen($destination) - 1] == '/'
     || $destination[strlen($destination) - 1] == '\\'))
      $destination = substr($destination, 0, -1);


   // check if the source file/directory exists in the first place
   //
   if (!local_file_path_exists($access_type, $user, $pass, $source, FALSE,
                               $suid_binary, $suid_debug_file, $master_username,
                               $master_password, $quiet))
   {
      if ($quiet) return NULL;

      sq_change_text_domain('server_settings_backend');
      global $sm_debug_mode, $server_settings_backend_debug;
      server_settings_backend_init();
      if ($server_settings_backend_debug) $sm_debug_mode |= SM_DEBUG_MODE_SIMPLE;
      if ($sm_debug_mode & SM_DEBUG_MODE_SIMPLE)
         $error = sprintf(_("Cannot move: \"%s\" was not found"), $source);
      else
         $error = _("Cannot move: File or directory not found");
      sq_change_text_domain('squirrelmail');
   }


   // if overwriting not allowed, bail if destination already exists
   //
   if (empty($error)
    && !$allow_overwrite
    && local_file_path_exists($access_type, $user, $pass, $destination,
                              FALSE, $suid_binary, $suid_debug_file,
                              $master_username, $master_password, $quiet))
   {
      if ($quiet) return NULL;

      sq_change_text_domain('server_settings_backend');
      global $sm_debug_mode, $server_settings_backend_debug;
      server_settings_backend_init();
      if ($server_settings_backend_debug) $sm_debug_mode |= SM_DEBUG_MODE_SIMPLE;
      if ($sm_debug_mode & SM_DEBUG_MODE_SIMPLE)
         $error = sprintf(_("File or directory \"%s\" already exists; cannot overwrite"), $destination);
      else
         $error = _("File or directory already exists; cannot overwrite");
      sq_change_text_domain('squirrelmail');
   }



   // PHP file access
   // (yes, the additional if() here is intentional and important!)
   // 
   if (empty($error))
   if ($access_type == 'PHP')
   {     
   
      // now, actually rename the file/directory
      //
      if (!sq_call_function_suppress_errors('rename', array($source, $destination)))
      {
         if ($quiet) return NULL;

         sq_change_text_domain('server_settings_backend');
         global $sm_debug_mode, $server_settings_backend_debug;
         server_settings_backend_init();
         if ($server_settings_backend_debug) $sm_debug_mode |= SM_DEBUG_MODE_SIMPLE;
         if ($sm_debug_mode & SM_DEBUG_MODE_SIMPLE)
            $error = sprintf(_("Could not rename \"%s\" to \"%s\""), $source, $destination);
         else
            $error = _("Error when attempting to rename file or directory");
         sq_change_text_domain('squirrelmail');
      }
   
   }



   // set-uid wrapper file access
   //
   else if ($access_type == 'SUID')
   {

      // set up debugging output if needed
      //
      if (!empty($suid_debug_file))
         $suid_debug = ' 2>>' . $suid_debug_file . ' ';
      else
         $suid_debug = '';


      // now, actually rename the file/directory
      //
      $cmd = $suid_binary
           . " 'localhost' " // "server" argument isn't currently used - just use "localhost"
           . escapeshellarg($user)
           . " 'rename' "
           . escapeshellarg($source) . ' '
           . escapeshellarg($destination)
           . $suid_debug;

      if (!($SUID = sq_call_function_suppress_errors('popen', array($cmd, 'w'))))
      {

         // show command details when in debug mode
         //
         sq_change_text_domain('server_settings_backend');
         global $sm_debug_mode, $server_settings_backend_debug;
         server_settings_backend_init();
         if ($server_settings_backend_debug) $sm_debug_mode |= SM_DEBUG_MODE_SIMPLE;
         if ($sm_debug_mode & SM_DEBUG_MODE_SIMPLE)
            $error = sprintf(_("Could not establish pipe to command \"%s\""), $cmd);
         else
            $error = _("Error establishing connection when attempting to rename file or directory");

         sq_change_text_domain('squirrelmail');

      }
      else if (!is_null($master_username) && sq_call_function_suppress_errors('fwrite', array($SUID, $master_username . "\n")) === FALSE)
      {

         // show command details and password (CAREFUL!) when in debug mode
         //
         sq_change_text_domain('server_settings_backend');
         global $sm_debug_mode, $server_settings_backend_debug;
         server_settings_backend_init();
         if ($server_settings_backend_debug) $sm_debug_mode |= SM_DEBUG_MODE_SIMPLE;
         if ($sm_debug_mode & SM_DEBUG_MODE_SIMPLE)
            $error = sprintf(_("Could not send master username \"%s\" to pipe command \"%s\""), $master_username, $cmd);
         else
            $error = _("Unable to send master username when attempting to rename file or directory");

         sq_change_text_domain('squirrelmail');

         @unlink($local_file);

      }
      else if (!is_null($master_password) && sq_call_function_suppress_errors('fwrite', array($SUID, $master_password . "\n")) === FALSE)
      {

         // show command details and password (CAREFUL!) when in debug mode
         //
         sq_change_text_domain('server_settings_backend');
         global $sm_debug_mode, $server_settings_backend_debug;
         server_settings_backend_init();
         if ($server_settings_backend_debug) $sm_debug_mode |= SM_DEBUG_MODE_SIMPLE;
         if ($sm_debug_mode & SM_DEBUG_MODE_SIMPLE)
            $error = sprintf(_("Could not send master password \"%s\" to pipe command \"%s\""), $master_password, $cmd);
         else
            $error = _("Unable to send master password when attempting to rename file or directory");

         sq_change_text_domain('squirrelmail');

         @unlink($local_file);

      }
      else if (sq_call_function_suppress_errors('fwrite', array($SUID, $pass . "\n")) === FALSE)
      {

         // show command details and password (CAREFUL!) when in debug mode
         //
         sq_change_text_domain('server_settings_backend');
         global $sm_debug_mode, $server_settings_backend_debug;
         server_settings_backend_init();
         if ($server_settings_backend_debug) $sm_debug_mode |= SM_DEBUG_MODE_SIMPLE;
         if ($sm_debug_mode & SM_DEBUG_MODE_SIMPLE)
            $error = sprintf(_("Could not send password \"%s\" to pipe command \"%s\""), $pass, $cmd);
         else
            $error = _("Unable to send password when attempting to rename file or directory");

         sq_change_text_domain('squirrelmail');

      }
      else
      {

         // get command return code
         //
         $status = ssb_pclose($SUID);


         // any error condition is critical
         //
         if ($status != SSB_ERR_OK)
         {

            // show command details when in debug mode
            //
            sq_change_text_domain('server_settings_backend');
            global $sm_debug_mode, $server_settings_backend_debug;
            server_settings_backend_init();
            if ($server_settings_backend_debug) $sm_debug_mode |= SM_DEBUG_MODE_SIMPLE;
            if ($sm_debug_mode & SM_DEBUG_MODE_SIMPLE)
               $error = sprintf(_("Unable to rename file or directory using command \"%s\""), $cmd);
            else
               $error = sprintf(_("Unable to rename file or directory (%s)"), $status);

            sq_change_text_domain('squirrelmail');

         }

      }

   }



   // unknown file access type
   //
   else
   {
      sq_change_text_domain('server_settings_backend');
      $error = sprintf(_("Unknown file access type \"%s\""), $access_type);
      sq_change_text_domain('squirrelmail');
   }



   // success
   //
   if (empty($error))
      return TRUE;


   // spew error
   //
   if ($quiet) return NULL;
   global $color;
   $ret = plain_error_message($error, $color);
   if (check_sm_version (1, 5, 2))
   {
      echo $ret;
      global $oTemplate;
      $oTemplate->display('footer.tpl');
   }
   exit;

}



