<?php

/**
  * SquirrelMail Server Settings Plugin
  * Copyright (c) 2008-2010 Paul Lesniewski <paul@squirrelmail.org>
  * Licensed under the GNU GPL. For full terms see the file COPYING.
  *
  * @package plugins
  * @subpackage server_settings
  *
  */



/**
  * Updates the given setting value.
  *
  * This can be used by other plugins or related code as
  * needed, such as the Spam Buttons plugin, when a Spam
  * button or custom button has been clicked.
  *
  * This function itself is a wrapper for the Server Settings
  * Backend plugin API for updating a server setting value.
  *
  * @param string $setting_name The name of the setting to be
  *                             updated.  This name MUST match
  *                             the name of the setting as
  *                             configured in one of the
  *                             Server Settings option pages'
  *                             "widgets" list.  If it is
  *                             not, an error will be raised.
  * @param mixed $new_value     The value to use for updating.
  *                             If the requested setting
  *                             consists of multiple values,
  *                             this will replace all of them
  *                             unless $add is turned on and
  *                             the widget is correctly
  *                             configured for multiple values.
  * @param string  $title       If the target widget name is
  *                             not unique amongst Server Settings
  *                             options pages, this can be
  *                             specified in order to identify
  *                             which page it should be taken
  *                             from (OPTIONAL; default empty
  *                             string - all pages will be
  *                             searched, and the first matching
  *                             widget used).
  * @param string  $section     If the target widget name is
  *                             not unique amongst Server Settings
  *                             sections in an option page, this
  *                             can be specified in order to
  *                             identify which section it should
  *                             be taken from (OPTIONAL; default
  *                             empty string - all page sections
  *                             will be searched, and the first
  *                             matching widget used).
  * @param boolean $add         When TRUE, and if the setting being
  *                             updated consists of multiple individual
  *                             values, $new_value is added to them
  *                             and the other values are left as is.
  *                             If the setting's configuration does
  *                             not include a "MULTIPLE" item, then
  *                             this flag has no effect and the setting
  *                             is updated as if it were turned off
  *                             (OPTIONAL; default = FALSE).
  * @param boolean $remove      When TRUE, and if the setting being
  *                             updated consists of multiple individual
  *                             values, $new_value is removed from
  *                             those values (if found within) and the
  *                             other values are left as is.  If the
  *                             setting's configuration does not include
  *                             a "MULTIPLE" item, then this flag has no
  *                             effect and the setting is updated as if
  *                             it were turned off (OPTIONAL; default
  *                             = FALSE).
  * @param boolean $quiet       When TRUE, suppresses any error
  *                             messages and makes this function
  *                             return NULL without doing anything
  *                             else (OPTIONAL; default = FALSE).
  *
  * @return mixed TRUE if the setting was stored normally or
  *               NULL if it was not (and $quiet is TRUE).
  *
  */
function server_settings_update_value($setting_name, $new_value, $title='',
                                      $section='', $add=FALSE, $remove=FALSE,
                                      $quiet=FALSE)
{

   $widget = ss_find_setting($setting_name, $title, $section);


   // couldn't find widget?  spit out error
   //
   if (empty($widget))
   {

      if ($quiet) return NULL;
      sq_change_text_domain('server_settings');
      $error = sprintf(_("Cannot find requested option widget (\"%s\") in Server Settings plugin update value function"), $setting_name);
      sq_change_text_domain('squirrelmail');
      global $color;
      $ret = plain_error_message($error, $color);
      if (check_sm_version (1, 5, 2))
      {
         echo $ret;
         global $oTemplate;
         $oTemplate->display('footer.tpl');
      }
      exit;

   }


   if (!empty($widget['STORAGE'])
    && is_array($widget['STORAGE']))
      $backend_info = $widget['STORAGE'];


   // no storage information is available, so we'll
   // set the value in the SquirrelMail preferences
   // ourselves
   //
   else
   {

      global $username, $data_dir;


      // deal with mullti-value settings
      //
      if (!empty($widget['TYPE'])
       && ($widget['TYPE'] == SMOPT_TYPE_FLDRLIST_MULTI
        || $widget['TYPE'] == SMOPT_TYPE_EDIT_LIST
        || $widget['TYPE'] == SMOPT_TYPE_STRLIST_MULTI))
      {

         // when adding to a mullti-value setting, first 
         // retrieve the value and put it into array format
         //
         if ($add)
         {

            $value = getPref($data_dir, $username, $widget['NAME']);

            // unserialize
            //
            if (is_string($value) && !empty($value))
//TODO: should we suppress errors on this?  it (an error) should only happen per bad configuration I think, but....
               $value = unserialize($value);
            else if (empty($value))
               $value = array();
            else
               $value = array($value);


            // add new value
            //
            if (!is_array($new_value))
               $new_value = array($new_value);
            $new_value = array_merge($value, $new_value);

         }


         // when removing from a mullti-value setting, first 
         // retrieve the value and put it into array format
         //
         else if ($remove)
         {

            $value = getPref($data_dir, $username, $widget['NAME']);

            // unserialize
            //
            if (is_string($value) && !empty($value))
//TODO: should we suppress errors on this?  it (an error) should only happen per bad configuration I think, but....
               $value = unserialize($value);
            else if (empty($value))
               $value = array();
            else
               $value = array($value);


            // remove new value
            //
            if (!is_array($new_value))
               $new_value = array($new_value);
            $new_value = array_diff($value, $new_value);

         }


         // multiple types get serialized before saving
         //
         $new_value = serialize($new_value);

      }


      // save it...
      // (setPref doesn't have a return value, so just assume success)
      //
      setPref($data_dir, $username, $widget['NAME'], $new_value);
      return TRUE;

   }


   // the put_server_setting() function is found 
   // in the Server Settings Backend plugin
   //
   include_once(SM_PATH . 'plugins/server_settings_backend/functions.php');
   return put_server_setting($new_value, $backend_info, $add, $remove, $quiet);

}



/**
  * Retrieves a setting's value.
  *
  * This can be used by other plugins or related code as
  * needed.
  *
  * This function itself is a wrapper for the Server Settings
  * Backend plugin API for retrieving a server setting value.
  *
  * @param string $setting_name The name of the setting to be
  *                             retrieved.  This name MUST match
  *                             the name of the setting as
  *                             configured in one of the
  *                             Server Settings option pages'
  *                             "widgets" list.  If it is
  *                             not, an error will be raised.
  * @param string  $title       If the target widget name is
  *                             not unique amongst Server Settings
  *                             options pages, this can be
  *                             specified in order to identify
  *                             which page it should be taken
  *                             from (OPTIONAL; default empty
  *                             string - all pages will be
  *                             searched, and the first matching
  *                             widget used).
  * @param string  $section     If the target widget name is
  *                             not unique amongst Server Settings
  *                             sections in an option page, this
  *                             can be specified in order to
  *                             identify which section it should
  *                             be taken from (OPTIONAL; default
  *                             empty string - all page sections
  *                             will be searched, and the first
  *                             matching widget used).
  * @param boolean $quiet       When TRUE, suppresses any error
  *                             messages and makes this function
  *                             return NULL without doing anything
  *                             else (OPTIONAL; default = FALSE).
  *
  * @return mixed The setting's value, or NULL if some problem/error
  *               occurred (and $quiet is TRUE).
  *
  */
function server_settings_retrieve_value($setting_name, $title='',
                                        $section='', $quiet=FALSE)
{
   return server_settings_test_value($setting_name, '', $title, $section,
                                     $quiet, TRUE);
}



/**
  * Checks if a given setting value matches a test value.
  *
  * This can be used by other plugins or related code as
  * needed, such as the Spam Buttons plugin, when deciding
  * if a Spam button or custom button should be shown on
  * a message view screen.
  *
  * This function itself is a wrapper for the Server Settings
  * Backend plugin API for testing a server setting value.
  *
  * @param string $setting_name The name of the setting to be
  *                             tested.  This name MUST match
  *                             the name of the setting as
  *                             configured in one of the
  *                             Server Settings option pages'
  *                             "widgets" list.  If it is
  *                             not, an error will be raised.
  * @param mixed $test_value    The value to test against.
  *                             If the requested setting
  *                             consists of multiple values,
  *                             the test value is deemed to
  *                             be a match if it matches
  *                             any one of the values
  *                             inside the setting.
  * @param string  $title       If the target widget name is
  *                             not unique amongst Server Settings
  *                             options pages, this can be
  *                             specified in order to identify
  *                             which page it should be taken
  *                             from (OPTIONAL; default empty
  *                             string - all pages will be
  *                             searched, and the first matching
  *                             widget used).
  * @param string  $section     If the target widget name is
  *                             not unique amongst Server Settings
  *                             sections in an option page, this
  *                             can be specified in order to
  *                             identify which section it should
  *                             be taken from (OPTIONAL; default
  *                             empty string - all page sections
  *                             will be searched, and the first
  *                             matching widget used).
  * @param boolean $quiet       When TRUE, suppresses any error
  *                             messages and makes this function
  *                             return NULL without doing anything
  *                             else (OPTIONAL; default = FALSE).
  * @param boolean $dont_test   When TRUE, no value testing happens;
  *                             the setting's value is merely
  *                             returned instead (mainly for use
  *                             with server_settings_retrieve_value())
  *                             (OPTIONAL; default = FALSE).
  *
  * @return mixed TRUE if the setting value matches the test
  *               value, FALSE if it does not, or if $dont_test
  *               is TRUE, the setting's value, or NULL if some
  *               problem/error occurred (and $quiet is TRUE).
  *
  */
function server_settings_test_value($setting_name, $test_value,
                                    $title='', $section='',
                                    $quiet=FALSE, $dont_test=FALSE)
{

   $widget = ss_find_setting($setting_name, $title, $section);


   // couldn't find widget?  spit out error
   //
   if (empty($widget))
   {

      if ($quiet) return NULL;
      sq_change_text_domain('server_settings');
      $error = sprintf(_("Cannot find requested option widget (\"%s\") in Server Settings plugin"), $setting_name);
      sq_change_text_domain('squirrelmail');
      global $color;
      $ret = plain_error_message($error, $color);
      if (check_sm_version (1, 5, 2))
      {
         echo $ret;
         global $oTemplate;
         $oTemplate->display('footer.tpl');
      }
      exit;

   }


   if (!empty($widget['STORAGE'])
    && is_array($widget['STORAGE']))
      $backend_info = $widget['STORAGE'];


   // no storage information is available, so we'll
   // get the value to test from the SquirrelMail
   // preferences and test it ourselves
   //
//TODO: should we check that $widget['NAME'] is not empty?
   else
   {

      global $username, $data_dir;
      $value = getPref($data_dir, $username, $widget['NAME']);

      // unserialize if necessary
      //
      if (!empty($widget['TYPE'])
       && ($widget['TYPE'] == SMOPT_TYPE_FLDRLIST_MULTI
        || $widget['TYPE'] == SMOPT_TYPE_EDIT_LIST
        || $widget['TYPE'] == SMOPT_TYPE_STRLIST_MULTI)
       && is_string($value) && !empty($value))
         $value = unserialize($value);


      // don't test?  just return value
      //
      if ($dont_test)
         return $value;


      // test it...
      //
      if (is_array($value)) return in_array($test_value, $value);
      else return $test_value == $value;

   }


   // we have backend storage configuration information
   // so we can proceed by referring to the Server Settings
   // Backend plugin...
   //
   include_once(SM_PATH . 'plugins/server_settings_backend/functions.php');


   // don't test?  just return value
   //
   if ($dont_test)
      return retrieve_server_setting($backend_info, $quiet);


   // now actually test the value
   //
   return test_server_setting($test_value, $backend_info, $quiet);

}



/**
  * Finds a setting in the Server Settings configuration by
  * searching with the given name (and/or section and/or
  * page title).
  *
  * @param string $setting_name The name of the setting to be
  *                             searched for.  This name MUST
  *                             match the name of the setting
  *                             as configured in one of the
  *                             Server Settings option pages'
  *                             "widgets" list.  If it is
  *                             not, FALSE will be returned.
  * @param string  $title       If the target widget name is
  *                             not unique amongst Server Settings
  *                             options pages, this can be
  *                             specified in order to identify
  *                             which page it should be taken
  *                             from (OPTIONAL; default empty
  *                             string - all pages will be
  *                             searched, and the first matching
  *                             widget used).
  * @param string  $section     If the target widget name is
  *                             not unique amongst Server Settings
  *                             sections in an option page, this
  *                             can be specified in order to
  *                             identify which section it should
  *                             be taken from (OPTIONAL; default
  *                             empty string - all page sections
  *                             will be searched, and the first
  *                             matching widget used).
  * @param boolean $quiet       When TRUE, suppresses any error
  *                             messages and makes this function
  *                             return NULL without doing anything
  *                             else (OPTIONAL; default = FALSE).
  *
  * @return mixed FALSE if the setting was not found, or an array
  *               of the setting's configuration values (with all
  *               keys having been converted to upper case) when
  *               it is found as expected.
  *
  */
function ss_find_setting($setting_name, $title='', $section='', $quiet=FALSE)
{

   global $server_settings_option_pages;
   include_once(SM_PATH . 'plugins/server_settings/init.php');
   server_settings_init();


   // for sanity, let's do everything in uppercase
   //
   $uppercase_pages = array();
   if (!empty($server_settings_option_pages) && is_array($server_settings_option_pages))
   {
      foreach ($server_settings_option_pages as $key => $value)
         $uppercase_pages[strtoupper($key)] = $value;
   }


   // spit out error
   //
   else
   {
      if ($quiet) return NULL;
      sq_change_text_domain('server_settings');
      $error = _("Server Settings plugin is misconfigured; \$server_settings_option_pages is not an array");
      sq_change_text_domain('squirrelmail');
      global $color;
      $ret = plain_error_message($error, $color);
      if (check_sm_version (1, 5, 2))
      {
         echo $ret;
         global $oTemplate;
         $oTemplate->display('footer.tpl');
      }
      exit;
   }


   // iterate the pages
   //
   foreach ($uppercase_pages as $page)
   {

      // for sanity, let's do everything in uppercase
      //
      $uppercase_page = array();
      if (is_array($page))
         foreach ($page as $key => $value)
            $uppercase_page[strtoupper($key)] = $value;


      // iterate the pages
      //
      if (is_array($uppercase_page['WIDGETS']))
      foreach ($uppercase_page['WIDGETS'] as $widget)
      {

         // for sanity, let's do everything in uppercase
         //
         $uppercase_widget = array();
         if (is_array($widget))
            foreach ($widget as $key => $value)
               $uppercase_widget[strtoupper($key)] = $value;


         // match first widget with the same name and,
         // only if given, under the requested section
         // and/or page title
         //
         if ((empty($title)
           || (!empty($title)
            && !empty($uppercase_page['TITLE'])
            && strtoupper($uppercase_page['TITLE']) == strtoupper($title)))
          && (empty($section)
           || (!empty($section)
            && !empty($uppercase_widget['SECTION'])
            && strtoupper($uppercase_widget['SECTION']) == strtoupper($section)))
          && !empty($uppercase_widget['NAME'])
          && strtoupper($uppercase_widget['NAME']) == strtoupper($setting_name))
         {
            return $uppercase_widget;
         }

      }

   }


   // widget was not found
   //
   return FALSE;

}



