<?php

/**
  * SquirrelMail Server Settings Plugin
  * Copyright (c) 2008-2009 Paul Lesniewski <paul@squirrelmail.org>
  * Licensed under the GNU GPL. For full terms see the file COPYING.
  *
  * @package plugins
  * @subpackage server_settings
  *
  */



/**
  * Retrieve the requested options page out of the Server
  * Settings options configuration settings (mainly used
  * as an internal utility function)
  *
  * If not found, an error is thrown (does not return)
  * unless $quiet is TRUE
  *
  * @param int $index     The index to the desired page
  * @param boolean $quiet When TRUE, suppresses any error
  *                       messages and makes this function
  *                       return NULL without doing anything
  *                       else (OPTIONAL; default = FALSE).
  *
  * @return mixed The desired options page configuration array
  *               with all array keys in upper case, or NULL
  *               if an error occurred and $quiet is TRUE
  *
  */
function ss_get_options_page($index, $quiet=FALSE)
{

   global $server_settings_option_pages;
   include_once(SM_PATH . 'plugins/server_settings/init.php');
   server_settings_init();


   // check if the page is available
   //
   if (empty($server_settings_option_pages[$index]))
   {
      if ($quiet) return NULL;
      sq_change_text_domain('server_settings');
      $error = sprintf(_("The Server Settings plugin is misconfigured (page index \"%s\" not found)"), $index);
      sq_change_text_domain('squirrelmail');
      global $color;
      $ret = plain_error_message($error, $color);
      if (check_sm_version (1, 5, 2))
      {
         echo $ret;
         global $oTemplate;
         $oTemplate->display('footer.tpl');
      }
      exit;
   }


   // for sanity, let's convert all array keys to upper case before returning it
   //
   $original_page_info = $server_settings_option_pages[$index];
   $page_info = array();
   foreach ($original_page_info as $key => $value)
      $page_info[strtoupper($key)] = $value;
   return $page_info;

} 


/**
  * Add our custom options page if it is being called
  *
  */
function server_settings_setup_options_page_do()
{

   global $optpage, $optpage_name, $optpage_file,
          $optpage_loader, $optpage_loadhook;

   if (preg_match('/^ss(\d+)$/', $optpage, $matches)
    && ($matches[1] == '0' || !empty($matches[1])))
   {

      $previous_text_domain = sq_change_text_domain('server_settings');
      $my_optpage = $matches[1];
      $page_info = ss_get_options_page($my_optpage);


      if (empty($page_info['TITLE']))
      {
         $error = _("The Server Settings plugin is misconfigured (cannot find page title)");
         sq_change_text_domain('squirrelmail');
         global $color;
         $ret = plain_error_message($error, $color);
         if (check_sm_version (1, 5, 2))
         {
            echo $ret;
            global $oTemplate;
            $oTemplate->display('footer.tpl');
         }
         exit;
      }


      $optpage_name = _($page_info['TITLE']);
      $optpage_file = SM_PATH . 'plugins/server_settings/options.php';
      $optpage_loader = 'load_optpage_data_server_settings';
      $optpage_loadhook = 'optpage_loadhook_server_settings';
      sq_change_text_domain($previous_text_domain);

   }

}



/**
  * This function builds an array with all the information about
  * the options to be displayed on this option page.  The options
  * are grouped by the groups in which they are displayed.
  *
  * @return array The options in SquirrelMail option API format
  *
  */
function load_optpage_data_server_settings()
{

   // make sure plugin is activated!
   //
   global $plugins, $optpage, $ss_widgets, $ss_return_to_self, $color;
   if (!in_array('server_settings', $plugins))
      return;
   $ss_return_to_self = FALSE;


   // what page are we serving?  we just want the number
   //
   if (!preg_match('/^ss(\d+)$/', $optpage, $matches)
    || ($matches[1] != '0' && empty($matches[1])))
      return;
   $my_optpage = $matches[1];


   $page_info = ss_get_options_page($my_optpage);
   $optgrps = array();
   $optvals = array();


   // now segregate option widgets into each section
   // (all widget array keys are converted to lower
   // case to comply with the SquirrelMail options API)
   // and figure out which page we are serving
   //
   if (!empty($page_info['WIDGETS']))
      $ss_widgets = $page_info['WIDGETS'];
   if (empty($ss_widgets) || empty($page_info['TITLE']))
   {
      sq_change_text_domain('server_settings');
      $error = _("The Server Settings plugin is misconfigured (cannot find page title or widgets list is empty)");
      sq_change_text_domain('squirrelmail');
      $ret = plain_error_message($error, $color);
      if (check_sm_version (1, 5, 2))
      {
         echo $ret;
         global $oTemplate;
         $oTemplate->display('footer.tpl');
      }
      exit;
   }
   foreach ($ss_widgets as $original_widget)
   {

      $widget = array();
      foreach ($original_widget as $key => $value)
         $widget[strtolower($key)] = $value;


      // do any setup needed to populate initial values of widget
      //
      $widget = ss_widget_setup($widget);


      if (!empty($widget['section']))
      {
         $previous_text_domain = sq_change_text_domain('server_settings');
         $optgrps[$widget['section']] = _($widget['section']);
         sq_change_text_domain($previous_text_domain);
         $optvals[$widget['section']][] = $widget;
      }
      else
      {
         // 1.5.2 omits the page title, so let's use it as the
         // section title when there is no section title
         //
         if (check_sm_version(1, 5, 2))
            $title = $page_info['TITLE'];
         else
            $title = '';

         $optgrps['###NO TITLE###'] = $title;
         $optvals['###NO TITLE###'][] = $widget;
      }

   }


   // detect if page submit should return to itself
   //
   if (!empty($page_info['RETURN_TO_SELF']))
      $ss_return_to_self = TRUE;


   // return final options groups
   //
   $result = array(
      'grps' => $optgrps,
      'vals' => $optvals,
      'xtra' => ''
   );
   return ($result);
 
}



/**
  * Set up an option widget by getting initial value(s)
  * from the backend specified in the "STORAGE" section of
  * the widget informtaion.
  *
  * @param array $widget The full widget information array.
  *
  * @return array The (possibly updated/modified) widget information array.
  *
  */
function ss_widget_setup($widget)
{

   // where do we put the setting when we're finished?
   //
   $setup_location = 'initial_value';
   if (!empty($widget['setup_location']))
      $setup_location = $widget['setup_location'];


   // grab backend storage information
   //
   $storage_info = array();
   if (!empty($widget['storage']) && is_array($widget['storage']))
      $storage_info = $widget['storage'];


   // if storage information is missing, get value from SquirrelMail prefs
   //
   if (empty($storage_info) && !empty($widget['name']))
   {
      global $username, $data_dir;
      $widget[$setup_location] = getPref($data_dir, $username, $widget['name']);

      // unserialize if necessary
      //
      if (!empty($widget['type']) 
       && ($widget['type'] == SMOPT_TYPE_FLDRLIST_MULTI
        || $widget['type'] == SMOPT_TYPE_EDIT_LIST
        || $widget['type'] == SMOPT_TYPE_STRLIST_MULTI)
       && is_string($widget[$setup_location])
       && !empty($widget[$setup_location]))
         $widget[$setup_location] = unserialize($widget[$setup_location]);
   }


   // this function is found in the Server Settings Backend 
   // plugin note that it will unserialize for us when needed
   //
   else
   {
      include_once(SM_PATH . 'plugins/server_settings_backend/functions.php');
      $setting = retrieve_server_setting($storage_info);
      $widget[$setup_location] = $setting;
   }


   // translate the caption, trailing_text, posvals,
   // yes_text, no_text, and comments in our own text domain
//TODO: initial_value?  maybe only for text widgets?
//TODO: should posvals only be translated for certain types (strlist, for example)?
//TODO: what other elements still need to be added here?
   //
   $previous_text_domain = sq_change_text_domain('server_settings');
   if (!empty($widget['caption']))
      $widget['caption'] = _($widget['caption']);
   if (!empty($widget['yes_text']))
      $widget['yes_text'] = _($widget['yes_text']);
   if (!empty($widget['no_text']))
      $widget['no_text'] = _($widget['no_text']);
   if (!empty($widget['trailing_text']))
      $widget['trailing_text'] = _($widget['trailing_text']);
   if (!empty($widget['comment']))
      $widget['comment'] = _($widget['comment']);
   if (!empty($widget['posvals']))
      if (is_array($widget['posvals']))
         foreach ($widget['posvals'] as $key => $value)
         {
            // should only be folder lists where $value isn't scalar here
            //
            if (is_string($value)) $widget['posvals'][$key] = _($value);
         }
      else
         $widget['posvals'] = _($widget['posvals']);
   sq_change_text_domain($previous_text_domain);


   // override the save function
   //
   if (!empty($widget['save']))
      $widget['original_save'] = $widget['save'];
   else
      $widget['original_save'] = NULL;
   $widget['save'] = 'ss_widget_save';


   //sm_print_r($widget);

   return $widget;

}



/**
  * Saves an option widget setting on the server by getting
  * using the backend specified in the "STORAGE" section of
  * the widget informtaion.
  *
  * If the widget has a custom save function of its own specified,
  * it will be called from here.
  *
  * @param object $option The widget in SquirrelMail Option class format.
  *
  * @return void
  *
  */
function ss_widget_save($option)
{

   // $ss_widgets is already set up in load_optpage_data_server_settings(),
   // so we just make it global so we can use it here too
   //
   global $optmode, $ss_widgets, $ss_return_to_self, $color;
   if (is_array($ss_widgets)) foreach ($ss_widgets as $original_widget)
   {

      $widget = array();
      foreach ($original_widget as $key => $value)
         $widget[strtolower($key)] = $value;

      if ($widget['name'] == $option->name)
      {

         // return to options page after saving?
         //
         if ($ss_return_to_self)
            $optmode = SMOPT_MODE_DISPLAY;


         // grab backend storage information
         //
         $storage_info = array();
         if (!empty($widget['storage']) && is_array($widget['storage']))
            $storage_info = $widget['storage'];


         // call out to widget's original save routine if it has one
         //
         if (!empty($widget['save']))
         {

            if (!function_exists($widget['save']))
            {
               sq_change_text_domain('server_settings');
               $error = sprintf(_("The Server Settings plugin is misconfigured (\"%s\" widget save function \"%s\" not found)"), $widget['name'], $widget['save']);
               sq_change_text_domain('squirrelmail');
               $ret = plain_error_message($error, $color);
               if (check_sm_version (1, 5, 2))
               {
                  echo $ret;
                  global $oTemplate;
                  $oTemplate->display('footer.tpl');
               }
               exit;
            }

            // no error - call the save function
            //
            $widget['save']($option, $storage_info);

         }

         // no save function, so we need to save the
         // value on the server ourselves
         //
         else
         {

            // if storage information is missing, store value in SquirrelMail prefs
            //
            if (empty($storage_info))
               save_option($option);


            // otherwise, save it to the server backend
            //
            else
            {
               // grab whether or not updates to multiple
               // values are saved all in one
               //
               $save_multiple_values_all_at_once = TRUE;
               if (!empty($widget['multiple_value_updates_seperately']))
                  $save_multiple_values_all_at_once = FALSE;

               ss_save_option($option, $storage_info, $save_multiple_values_all_at_once);
            }

         }

         break;  // out of the foreach $ss_widgets loop

      }

   }

}



/**
  * Saves an option widget setting on the server using
  * the backend specified in the "STORAGE" section of
  * the widget informtaion.
  *
  * @param object  $option       The widget in SquirrelMail Option class format.
  * @param array   $storage_info The backend storage information needed to
  *                              save the setting on the server.
  * @param boolean $save_multiple_values_all_at_once When FALSE, multiple value
  *                                                  widgets will have new items
  *                                                  added to them and old ones
  *                                                  deleted in two different
  *                                                  actions; when FALSE, the
  *                                                  full final list is compiled
  *                                                  and the save action is done
  *                                                  at once (OPTIONAL; default=TRUE).
  *
  * @return void
  *
  */
function ss_save_option($option, $storage_info, $save_multiple_values_all_at_once=TRUE)
{

   // the put_server_setting() function is found
   // in the Server Settings Backend plugin
   //
   include_once(SM_PATH . 'plugins/server_settings_backend/functions.php');


   // edit lists need more intricate processing
   //
   if (!empty($option->type) && $option->type == SMOPT_TYPE_EDIT_LIST)
   {

      // the following is the edit list save part of
      // save_option() in functions/options.php
      //
      if (empty($option->possible_values)) $option->possible_values = array();
      if (!is_array($option->possible_values)) $option->possible_values = array($option->possible_values);

      // add element if given
      //
      if ((isset($option->use_add_widget) && $option->use_add_widget)
       && sqGetGlobalVar('add_' . $option->name, $new_element, SQ_POST))
      {
         $new_element = trim($new_element);
         if (!empty($new_element)
          && !in_array($new_element, $option->possible_values))
         {
            // note that the use of the third argument below assumes
            // that this setting has MULTIPLE set in its storage config
            //
            if (!$save_multiple_values_all_at_once)
               put_server_setting($new_element, $storage_info, TRUE);

            // build final full list
            //
            $option->possible_values[] = $new_element;
         }
      }

      // delete selected elements if needed
      //
      if ((isset($option->use_delete_widget) && $option->use_delete_widget)
       && is_array($option->new_value)
       && sqGetGlobalVar('delete_' . $option->name, $ignore, SQ_POST))
      {
         // note that the use of the fourth argument below assumes
         // that this setting has MULTIPLE set in its storage config
         //
         if (!$save_multiple_values_all_at_once)
            put_server_setting($option->new_value, $storage_info, FALSE, TRUE);

         // build final full list
         //
         $option->possible_values = array_diff($option->possible_values, $option->new_value);
      }

      // save full list (stored in "possible_values")
      //
      if ($save_multiple_values_all_at_once)
         put_server_setting($option->possible_values, $storage_info);

   }


   // have to pick apart multiple values if we are saving adds and
   // deletes separately
   //
   else if (!$save_multiple_values_all_at_once
    && ((!empty($option->type) && $option->type == SMOPT_TYPE_STRLIST_MULTI)
     || (!empty($option->type) && $option->type == SMOPT_TYPE_STRLIST_MULTI)))
   {
      $new_elements = array_diff($option->new_value, $option->value);
      $delete_elements = array_diff($option->value, $option->new_value);
      put_server_setting($new_elements, $storage_info, TRUE);
      put_server_setting($delete_elements, $storage_info, FALSE, TRUE);
   }


   // checkboxes, when unchecked, don't submit anything in
   // the POST, so set to SMPREF_OFF if not found
   //
   else if (!empty($option->type)
    && ($option->type == SMOPT_TYPE_BOOLEAN
     || $option->type == SMOPT_TYPE_BOOLEAN_CHECKBOX)
    && empty($option->new_value))
      put_server_setting(SMPREF_OFF, $storage_info);


   else put_server_setting($option->new_value, $storage_info);


   // if a checkbox or multi select is zeroed/cleared out, it
   // needs to have an empty value pushed into its "new_value" slot
   // (IIRC, this is only necessary when RETURN_TO_SELF is used)
   //
   if (($option->type == SMOPT_TYPE_STRLIST_MULTI
     || $option->type == SMOPT_TYPE_BOOLEAN
     || $option->type == SMOPT_TYPE_BOOLEAN_CHECKBOX)
    && is_null($option->new_value))
      $option->new_value = '';

}



