<?php

/**
  * SquirrelMail One-Time Password Plugin
  * Copyright (c) 2008-2009 Paul Lesniewski <paul@squirrelmail.org>
  * Licensed under the GNU GPL. For full terms see the file COPYING.
  *
  * @package plugins
  * @subpackage otp
  *
  */



/**
  * Get crypto hash phrase
  *
  * @return mixed NULL if the crypto hash phrase was not found;
  *               otherwise, the string containing the hash phrase
  *
  */
function get_otp_crypto_hash_phrase()
{
   $OTP_CRYPTO_HASH_PHRASE = NULL;
   if (!sqGetGlobalVar('OTP_CRYPTO_HASH_PHRASE', $OTP_CRYPTO_HASH_PHRASE, SQ_SERVER))
   {
      global $otp_crypto_hash_phrase;
      otp_init();
      $OTP_CRYPTO_HASH_PHRASE = $otp_crypto_hash_phrase;
   }
   return $OTP_CRYPTO_HASH_PHRASE;
}



/**
  * Get crypto algorithm
  *
  * @return mixed NULL if the crypto algorithm was not found;
  *               otherwise, the string containing the algorithm
  *
  */
function get_otp_crypto_algorithm()
{
   $OTP_CRYPTO_ALGORITHM = NULL;
   if (!sqGetGlobalVar('OTP_CRYPTO_ALGORITHM', $OTP_CRYPTO_ALGORITHM, SQ_SERVER))
   {
      global $otp_crypto_algorithm;
      otp_init();
      $OTP_CRYPTO_ALGORITHM = $otp_crypto_algorithm;
   }
   return $OTP_CRYPTO_ALGORITHM;
}



/**
  * Get OTP prefix
  *
  * @return mixed NULL if the password prefix was not found;
  *               otherwise, the string containing the prefix
  *
  */
function get_otp_prefix()
{
   $OTP_PREFIX = NULL;
   if (!sqGetGlobalVar('OTP_PREFIX', $OTP_PREFIX, SQ_SERVER))
   {
      global $otp_prefix;
      otp_init();
      $OTP_PREFIX = $otp_prefix;
   }
   return $OTP_PREFIX;
}



/**
  * Initialize this plugin (load config values)
  *
  * @return boolean FALSE if no configuration file could be loaded, TRUE otherwise
  *
  */
function otp_init()
{

   if (!@include_once (SM_PATH . 'config/config_otp.php'))
      if (!@include_once (SM_PATH . 'plugins/otp/config.php'))
         return FALSE;

   return TRUE;

}



/**
  * Validate that this plugin is configured correctly
  *
  * @return boolean Whether or not there was a
  *                 configuration error for this plugin.
  *
  */
function otp_check_configuration_do()
{

   // make sure base config is available
   //
   if (!otp_init())
   {
      do_err('One-Time Passowrd plugin is missing its main configuration file', FALSE);
      return TRUE;
   }


   // make sure we have a crypto hash phrase and algorithm
   //
   if (($hash_phrase = get_otp_crypto_hash_phrase()) == NULL)
   {
      do_err('One-Time Password plugin is misconfigured: no cryptographic hash phrase was found', FALSE);
      return TRUE;
   }
   if (($algo = get_otp_crypto_algorithm()) == NULL)
   {
      do_err('One-Time Password plugin is misconfigured: no cryptographic algorithm was found', FALSE);
      return TRUE;
   }


   // The default hash phrase is not acceptable
   //
   if ($hash_phrase == 'Llama')
   {
      do_err('One-Time Password plugin is misconfigured: the cryptographic hash phrase has not been changed from its default value', FALSE);
      return TRUE;
   }


   // if not using our built-in encryption routine, test that
   // mycrypt is there and the chosen algorithm is available
   //
   if ($algo != 'rc4_builtin' && !function_exists('mcrypt_generic'))
   {
      do_err('One-Time Password plugin is misconfigured: its cryptographic algorithm specifies an mcrypt algorithm, but mcrypt support is not present in PHP', FALSE);
      return TRUE;
   }


   // encourage more secure configuration
   //
   if (!sqGetGlobalVar('OTP_CRYPTO_HASH_PHRASE', $OTP_CRYPTO_HASH_PHRASE, SQ_SERVER))
   {
      do_err('You are encouraged to configure the One-Time Password plugin more securely by specifying the cryptographic algorithm in the web server configuration instead of in the PHP plugin configuration file.', FALSE);
   }
   if (!sqGetGlobalVar('OTP_CRYPTO_ALGORITHM', $OTP_CRYPTO_ALGORITHM, SQ_SERVER))
   {
      do_err('You are encouraged to configure the One-Time Password plugin more securely by specifying the cryptographic hash phrase in the web server configuration instead of in the PHP plugin configuration file.', FALSE);
   }


   // only need to do this pre-1.5.2, as 1.5.2 will make this
   // check for us automatically
   //
   if (!check_sm_version(1, 5, 2))
   {

      // try to find Compatibility, and then that it is v2.0.9+
      //
      if (function_exists('check_plugin_version')
       && check_plugin_version('compatibility', 2, 0, 9, TRUE))
         return FALSE;


      // something went wrong
      //
      do_err('One-Time Passowrd plugin requires the Compatibility plugin version 2.0.9+', FALSE);
      return TRUE;

   }

}



/**
  * Make sure Password Forget fires first;
  * also override any configuration values
  * to restrict users logged in with a temporary
  * password
  *
  */
function otp_config_override_do()
{

   // override any configuration settings needed when
   // user is logged in with a temporary password
   //
   if (sqGetGlobalVar('logged_in_with_otp', $logged_in_with_otp, SQ_SESSION)
    && $logged_in_with_otp)
   {

      global $otp_locked_out_config, $otp_locked_out_plugins;
      otp_init();
      foreach ($otp_locked_out_config as $config => $locked_out_value)
      {
         global $$config;
         $$config = $locked_out_value;
      }

      foreach ($otp_locked_out_plugins as $plugin_name)
         remove_plugin($plugin_name);

   }


   // make sure Password Forget goes first on login_before hook
   //
   global $PHP_SELF;
   if (strpos($PHP_SELF, '/redirect.php') !== FALSE)
      reposition_plugin_on_hook('otp', 'login_before', FALSE, 'password_forget');

}



/**
  * Check if a OTP is being used to log in
  *
  */
function otp_validate_login_do() 
{

   global $otp_strict_order, $data_dir, $plugins,
          $login_username, $secretkey, $otp_usernames;
   otp_init();
   $use_otp = FALSE;


   $user = $login_username;
   $pass = $secretkey;
   $password_variable = 'secretkey';
   $username_variable = 'login_username';


   // check for prefix if needed; strip the prefix if
   // found, otherwise bail (if the user didn't provide it)
   //
   $prefix = get_otp_prefix();
   if ($prefix !== '' && strpos($pass, $prefix) !== 0)
      return;
   else
      $pass = substr($pass, strlen($prefix));


   // need to fish for the right user OTPs if the
   // username is part of the OTP
   //
   if ($otp_usernames)
   {
      if (($pos = strpos($user, '@')) !== FALSE)
         $user = substr($user, 0, $pos);
      $result = find_otp_username_password($user, $pass, $otp_strict_order);
      if (is_array($result) && !empty($result[0]) && isset($result[1]))
      {
         $user = $result[0];
         $otps = $result[1];
         $use_otp = TRUE;
      }
   }


   // are OTPs allowable for the user logging in?
   //
   $allow_otp = getPref($data_dir, $user, 'allow_otp', 0);
   if (!$allow_otp) return;


   // if we haven't already, validate...
   //
   if (!$otp_usernames)
   {

      // pull user's OTP list
      //
      $otps = get_user_otps($user, $otp_usernames);
      if (empty($otps)) return;


      // compare login pwd to the list
      //
      if ($otp_strict_order)
      {

         // only need to check the first OTP
         //
         $otp = array_shift($otps);
         if (!empty($otp) && $otp === $pass)
            $use_otp = TRUE;

      }
      else
      {
         // iterate through all the OTPs; any will do
         //
         $new_otps = array();
         foreach ($otps as $otp)
         {
            if (!empty($otp) && $otp === $pass)
            {
               $use_otp = TRUE;
               continue; // omit from new list
            }
            $new_otps[] = $otp;
         }
         $otps = $new_otps;
      }

   }


   // if an OTP was found, update the OTP list
   // (which should already have the used OTP
   // removed above) in the user prefs, then
   // substitute the real password and let the
   // login process proceed ... also set session
   // var that indicates user is logged in using a OTP
   //
   if ($use_otp)
   {
      put_user_otps($user, $otps, $otp_usernames);
      $$password_variable = get_user_real_pwd($user);
      $$username_variable = $user;
      if (empty($otps))
         put_user_real_pwd($user, TRUE);
      sqsession_register(1, 'logged_in_with_otp');
   }

}



/**
  * Put OTP block on options page
  *
  */
function otp_option_block_do() 
{
   global $optpage_blocks;


   // remove any option blocks as needed when
   // user is logged in with a temporary password
   //
   if (sqGetGlobalVar('logged_in_with_otp', $logged_in_with_otp, SQ_SESSION)
    && $logged_in_with_otp)
   {

      global $otp_locked_out_option_pages;
      otp_init();
      foreach ($otp_locked_out_option_pages as $page)
         unset($optpage_blocks[$page]);

   }

   sq_change_text_domain('otp');

   $optpage_blocks[] = array(
      'name' => _("Manage One-Time Passwords"),
      //'name' => _("Login Ticket Administration"),
      'url' => sqm_baseuri() . 'src/options.php?optpage=otp',
      'desc' => _("You can create passwords for one-time use when logging into your account via untrusted computers."),
      //'desc' => _("You can create login tickets (one-time logins) to login from untrusted computers without using your real password."),
      'js' => FALSE
   );

   sq_change_text_domain('squirrelmail');
}



/**
  * Allow removal of certain options on
  * various options pages
  *
  */
function otp_remove_options_do()
{

   // remove any options or option pages as needed when
   // user is logged in with a temporary password
   //
   if (sqGetGlobalVar('logged_in_with_otp', $logged_in_with_otp, SQ_SESSION)
    && $logged_in_with_otp)
   {

      global $otp_locked_out_option_pages, $optpage,
             $otp_locked_out_options, $optpage_data;
      otp_init();


      // make sure user isn't trying to access/submit
      // a disallowed option page
      //
      // 0 = Personal Information = personal
      // 1 = Display Preferences  = display
      // 2 = Message Highlighting = highlight
      // 3 = Folder Preferences   = folder
      // 4 = Index Order          = order
      // 5 = Compose Preferences  = compose
      //
      switch ($optpage)
      {
         case 'personal':
            if (in_array(0, $otp_locked_out_option_pages))
               exit;  // don't print anything out to help attackers
            break;
         case 'display':
            if (in_array(1, $otp_locked_out_option_pages))
               exit;  // don't print anything out to help attackers
            break;
         case 'highlight':
            if (in_array(2, $otp_locked_out_option_pages))
               exit;  // don't print anything out to help attackers
            break;
         case 'folder':
            if (in_array(3, $otp_locked_out_option_pages))
               exit;  // don't print anything out to help attackers
            break;
         case 'order':
            if (in_array(4, $otp_locked_out_option_pages))
               exit;  // don't print anything out to help attackers
            break;
         case 'compose':
            if (in_array(5, $otp_locked_out_option_pages))
               exit;  // don't print anything out to help attackers
            break;
      }


      // now remove options off any of the standard option pages
      //
      foreach ($otp_locked_out_options as $removeOptionName)
         foreach ($optpage_data['vals'] as $groupNumber => $optionGroup)
            foreach ($optionGroup as $optionNumber => $optionItem)
               if (isset($optionItem['name']) && $optionItem['name'] == $removeOptionName)
                  unset($optpage_data['vals'][$groupNumber][$optionNumber]);


      // check if no other elements are left in the option group
      // (except hidden elements)...  remove the whole group if
      // so (are there any hidden elements that we WANT to retain??)
      //
      foreach ($optpage_data['vals'] as $groupNumber => $optionGroup)
      {
         $group_is_empty = TRUE;
         foreach ($optionGroup as $optionNumber => $optionItem)
            if (isset($optionItem['type']) && $optionItem['type'] != SMOPT_TYPE_HIDDEN)
            {
               $group_is_empty = FALSE;
               break;
            }
         if ($group_is_empty)
         {
            unset($optpage_data['vals'][$groupNumber]);
            unset($optpage_data['grps'][$groupNumber]);
         }
      }
   }

}



/**
  * Set up custom options page
  *
  */
function otp_setup_options_page_do()
{

   global $optpage, $optpage_name, $optpage_file,
          $optpage_loader, $optpage_loadhook;

   if ($optpage == 'otp')
   {

      $previous_text_domain = sq_change_text_domain('otp');
      $optpage_name = _("Manage One-Time Passwords");
      //$optpage_name = _("Login Ticket Administration");
      $optpage_file = SM_PATH . 'plugins/otp/functions.php';
      $optpage_loader = 'load_optpage_data_otp';
      $optpage_loadhook = 'optpage_loadhook_otp';
      sq_change_text_domain($previous_text_domain);

   }

}



/**
  * This function builds an array with all the information about
  * the options to be displayed on this option page.  The options
  * are grouped by the groups in which they are displayed.
  *
  * @return array The options in SquirrelMail option API format
  *
  */
function load_optpage_data_otp()
{

   // make sure plugin is activated!
   //
   global $plugins;
   if (!in_array('otp', $plugins))
      return;


   // get current list of passwords
   // and calculate how many new ones are allowed
   //
   global $otp_strict_order, $otp_max_at_once, $otp_password_length,
          $otp_password_chars, $username, $data_dir, $otp_usernames,
          $otp_username_chars, $otp_username_length;

   otp_init();

   // @ and | and ; are used as separators internally, 
   // so strip them out of the valid character lists
   //
   $otp_password_chars = str_replace(array('@', '|', ';'), '', $otp_password_chars);
   $otp_username_chars = str_replace(array('@', '|', ';'), '', $otp_username_chars);

   sqGetGlobalVar('logged_in_with_otp', $logged_in_with_otp, SQ_SESSION);
   sq_change_text_domain('otp');
   $allow_otp = getPref($data_dir, $username, 'allow_otp', 0);

   $otps = get_user_otps($username, $otp_usernames);
   $max_otps_allowed = max(0, $otp_max_at_once - sizeof($otps));
   $allowable_otps = array();
   for ($i = 0; $i <= $max_otps_allowed; $i++)
      $allowable_otps[$i] = $i;


   // handle allow checkbox
   //
   if (sqGetGlobalVar('new_look_for_allow_otp_checkbox', $new_look_for_allow_otp_checkbox, SQ_POST)
    && $new_look_for_allow_otp_checkbox)
   {
      if (sqGetGlobalVar('new_allow_otp', $new_allow_otp, SQ_POST)
       && $new_allow_otp)
         $allow_otp = 1;
      else
         $allow_otp = 0;
      setPref($data_dir, $username, 'allow_otp', $allow_otp);
   }


   // handle delete password request if needed
   //
   if (sqGetGlobalVar('delete_delete_otps', $delete_otps_btn, SQ_POST)
    && $delete_otps_btn
    && sqGetGlobalVar('new_delete_otps', $delete_otps, SQ_POST)
    && !empty($delete_otps)
    && is_array($delete_otps))
   {

      $new_otps = array();
      foreach ($otps as $i => $pass)
      {
         if ($otp_usernames)
         {
            $pos = strpos($pass, ';');
            $test_value = sprintf(_("Username = %s Password = %s"),
                                  substr($pass, 0, $pos),
                                  substr($pass, $pos + 1));
         }
         else
            $test_value = $pass;
         foreach ($delete_otps as $j => $pwd)
            if ($i == $j && $pwd == $test_value)
               continue 2;
         $new_otps[] = $pass;
      }
      $otps = $new_otps;

      put_user_otps($username, $otps, $otp_usernames);
      if (empty($otps))
         put_user_real_pwd($username, TRUE);
      else
         put_user_real_pwd($username, FALSE);

   }


   // handle create password request if needed
   //
   $bad_pwd_verify = FALSE;
   if (sqGetGlobalVar('new_number_of_otp', $new_number_of_otp, SQ_POST)
    && $new_number_of_otp)
   {

      if (!sqGetGlobalVar('new_verify_real_pwd', $new_verify_real_pwd, SQ_POST)
       || !$new_verify_real_pwd 
       || $new_verify_real_pwd !== sqauth_read_password())
         $bad_pwd_verify = TRUE;

      else
      {

         // when a user first creates new OTPs (previously there
         // were none), automatically turn on the "allow" checkbox
         //
         if (sizeof($otps) < 1 && $new_number_of_otp > 0)
         {
            $allow_otp = 1;
            setPref($data_dir, $username, 'allow_otp', $allow_otp);
         }

         $new_otps = array();
         for ($i = 1; $i <= $new_number_of_otp && $i <= $max_otps_allowed; $i++)
            if ($otp_usernames)
            {
               // make sure same user/pwd pair wasn't already created
               //
               $user = GenerateRandomString($otp_username_length, $otp_username_chars);
               $pass = GenerateRandomString($otp_password_length, $otp_password_chars);
               $j = 0;
               $max_tries = 1000;
               while (find_otp_username_password($user, $pass) !== FALSE)
               {
                  if (++$j > $max_tries)
                  {
                     echo '<html><body><font color="red"><b>One-Time Password has encountered a fatal error trying to create a unique one-time username and password</b></font></body></html>';
                     exit;
                  }
                  $user = GenerateRandomString($otp_username_length, $otp_username_chars);
                  $pass = GenerateRandomString($otp_password_length, $otp_password_chars);
               }
               $new_otps[] = $user . ';' . $pass;
            }
            else
               $new_otps[] = GenerateRandomString($otp_password_length, $otp_password_chars);
         $otps = array_merge($otps, $new_otps);

         put_user_otps($username, $otps, $otp_usernames);
         if (empty($otps))
            put_user_real_pwd($username, TRUE);
         else
            put_user_real_pwd($username, FALSE);
      }

   }
   // it sucks that this is the only way to reset the defaults...
   $_POST['new_verify_real_pwd'] = '';
   if (!$bad_pwd_verify)
      $_POST['new_number_of_otp'] = 0;


   // recalculate number of allowable new OTPs
   //
   $max_otps_allowed = max(0, $otp_max_at_once - sizeof($otps));
   $allowable_otps = array();
   for ($i = 0; $i <= $max_otps_allowed; $i++)
      $allowable_otps[$i] = $i;


   // make $otps displayable if it contains usernames too
   //
   if ($otp_usernames)
      foreach ($otps as $i => $otp)
      {
         $pos = strpos($otp, ';');
         $otps[$i] = sprintf(_("Username = %s Password = %s"),
                             substr($otp, 0, $pos),
                             substr($otp, $pos + 1));
      }


   // set up page elements
   //
   $optgrps = array();
   $optvals = array();


   if (check_sm_version(1, 5, 2))
   {
      if ($bad_pwd_verify)
      {
         global $br; 
         $optgrps['info'] = _("Manage One-Time Passwords")
                          . $br
                          . _("ERROR: Password verification failed - please try again");
      }
      else
         $optgrps['info'] = _("Manage One-Time Passwords");
         //$optgrps['info'] = _("Login Ticket Administration");
   }
   else // 1.4.x
   {
      if ($bad_pwd_verify)
         $optgrps['info'] = _("ERROR: Password verification failed - please try again");
      else
         $optgrps['info'] = '';
   }


   $optvals['info'] = array(

      // intro text
      //
      array(
         'name'          => 'info1',
         'caption'       => '',
         'type'          => SMOPT_TYPE_INFO,
         'initial_value' => ($otp_usernames
                          ? _("When using a computer you don't trust (such as one in an Internet cafe), there is an increased risk that your username and/or password could be stolen. To help avoid this problem, you can create temporary usernames and passwords here that only work once each.") . ($otp_strict_order ? ' ' . _("You will need to use these username/password pairs in the order they are created and displayed here - you cannot use the last one before you use the first one, and each credential pair must be kept together - you cannot use a password with any other username except the one it is listed with.") : ' ' . _("Each credential pair must be kept together - you cannot use a password with any other username except the one it is listed with."))
                          : _("When using a computer you don't trust (such as one in an Internet cafe), there is an increased risk that your password could be stolen. To help avoid this problem, you can create temporary passwords here that only work once each.") . ($otp_strict_order ? ' ' . _("You will need to use the passwords in the order they are created and displayed here - you cannot use the last one before you use the first one.") : '')),
         'refresh'       => SMOPT_REFRESH_NONE,
      ),

   );

   if ($logged_in_with_otp)
   {

      // intro text
      //
      $optvals['info'][] = 
         array(
            'name'          => 'info2',
            'caption'       => '',
            'type'          => SMOPT_TYPE_INFO,
            'initial_value' => ($otp_usernames
                             ? _("However, while you are logged in with temporary credentials, you cannot view other ones, nor can you create more.  Please log in with your real username and password if you need to do so.")
                             : _("However, while you are logged in with a temporary password, you cannot view other temporary passowrds, nor can you create more.  Please log in with your real password if you need to do so.")),
            'refresh'       => SMOPT_REFRESH_NONE,
            'save'          => 'otp_save_prefs',
         );

   }

   else
   {

      // intro text
      //
      $optvals['info'][] = 
         array(
            'name'          => 'info3',
            'caption'       => '',
            'type'          => SMOPT_TYPE_INFO,
            'initial_value' => _("Please be careful about who can see your screen when you are viewing this page."),
            'refresh'       => SMOPT_REFRESH_NONE,
            'save'          => 'otp_save_prefs',
         );

      // spacer
      //
      $optvals['info'][] = 
         array(
            'name'           => 'separator_1',
            'caption'        => '',
            'comment'        => '',
            'type'           => SMOPT_TYPE_COMMENT,
            'save'           => 'otp_save_prefs',
         );

      // spacer
      //
      $optvals['info'][] = 
         array(
            'name'           => 'separator_2',
            'caption'        => '',
            'comment'        => '',
            'type'           => SMOPT_TYPE_COMMENT,
            'save'           => 'otp_save_prefs',
         );

      if (!empty($otps))
      {
         // show current OTPs (yikes!)
         //
         $optvals['info'][] = 
            array(
               'name'          => 'look_for_allow_otp_checkbox',
               'caption'       => '',
               'type'          => SMOPT_TYPE_HIDDEN,
               'refresh'       => SMOPT_REFRESH_NONE,
               'initial_value' => 1,
               'save'          => 'otp_save_prefs',
            );
         $optvals['info'][] = 
            array(
               'name'          => 'allow_otp',
               'caption'       => ($otp_usernames
                                ? _("Allow logins using temporary credentials")
                                : _("Allow logins using temporary passwords")),
               'type'          => SMOPT_TYPE_BOOLEAN,
               'refresh'       => SMOPT_REFRESH_NONE,
               'initial_value' => $allow_otp,
               'save'          => 'otp_save_prefs',
            );

         // show current OTPs (yikes!)
         //
         if (sizeof($otps) > 25) $layout_type = SMOPT_EDIT_LIST_LAYOUT_SELECT;
         else $layout_type = SMOPT_EDIT_LIST_LAYOUT_LIST;
//FIXME: if the layout is a select list, the POST variables are different, so the code that manages deletions (and maybe other) will have to work differently... so for now, we have to hard code to LIST style:
$layout_type = SMOPT_EDIT_LIST_LAYOUT_LIST;
         $optvals['info'][] = 
            array(
               'name'           => 'delete_otps',
               'type'           => SMOPT_TYPE_EDIT_LIST,
               'layout_type'    => $layout_type,
               'use_add_widget' => FALSE,
               //'size'           => SMOPT_SIZE_SMALL,
               'caption'        => ($otp_usernames
                                 ? _("Current credentials")
                                 : _("Current passwords")),
               'refresh'        => SMOPT_REFRESH_NONE,
               'posvals'        => $otps,
               'save'           => 'otp_save_prefs',
            );
      }

      if ($max_otps_allowed > 0)
      {
         // verify password to create new OTPs
         //
         $optvals['info'][] = 
            array(
               'name'    => 'verify_real_pwd',
               'type'    => SMOPT_TYPE_PASSWORD,
               'caption' => _("Verify real password"),
               'refresh' => SMOPT_REFRESH_NONE,
               'save'    => 'otp_save_prefs',
            );

         // number of OTPs to create
         //
         $optvals['info'][] = 
            array(
               'name'    => 'number_of_otp',
               'type'    => SMOPT_TYPE_STRLIST,
               'caption' => ($otp_usernames
                          ? _("Number of credential pairs to create")
                          : _("Number of passwords to create")),
               'refresh' => SMOPT_REFRESH_NONE,
               'posvals' => $allowable_otps,
               'save'    => 'otp_save_prefs',
            );
      }
   }


   // return final options groups
   //
   $result = array(
      'grps' => $optgrps,
      'vals' => $optvals,
      'xtra' => ''
   );
   sq_change_text_domain('squirrelmail');
   return ($result);

}



/**
  * Placeholder for saving settings
  *
  * This just makes sure we reload the same page
  *
  */
function otp_save_prefs()
{
   global $optmode;
   $optmode = SMOPT_MODE_DISPLAY;
}



/**
  * Retrieves (from user preferences) and
  * decrypts a user's list of OTPs; if usernames
  * are also being stored, the list is retrieved
  * from a global preferences set prefixed by the
  * real username
  *
  * @param string  $user         The user for which to get the OTPs
  * @param boolean $otp_usenames Whether or not usernames are also
  *                              part of OTPs
  *
  * @return array A list of the unencrypted OTPs; if usernames
  *               are also included, each OTP will contain a
  *               username and then password (in that order),
  *               separated by a ; character
  *
  */
function get_user_otps($user, $otp_usernames)
{
   global $data_dir;
   if (!$otp_usernames)
   {
      $otps = getPref($data_dir, $user, 'one_time_passwords', '');
      if (!empty($otps))
      {
         $otps = otp_crypto($otps, 'decrypt');
         if (($otps = @unserialize($otps)) === FALSE)
            $otps = array();
         else
            $otps = explode('|', $otps);
      }
      else
         $otps = array();
   }
   else
   {
      // first make sure main SM prefs cache is empty
      //
      global $prefs_cache, $prefs_are_cached;
      $prefs_cache = FALSE;
      $prefs_are_cached = FALSE;
      sqsession_unregister('prefs_are_cached');
      sqsession_unregister('prefs_cache');


      // retrieve system-wide OTP list
      //
      $all_otps = getPref($data_dir, 'one_time_passwords', 'one_time_passwords', '');
      if (!empty($all_otps))
      {
         $all_otps = otp_crypto($all_otps, 'decrypt');
         if (($all_otps = @unserialize($all_otps)) === FALSE)
            $all_otps = array();
         else
            $all_otps = explode('|', $all_otps);
      }
      else
         $all_otps = array();


      // now retrieve all OTPs for the current user
      //
      $otps = array();
      foreach ($all_otps as $otp)
      {
         if (strpos($otp, $user . ';') === 0)
            $otps[] = substr($otp, strlen($user) + 1);
      }


      // make sure main SM prefs cache is empty again
      //
      $prefs_cache = FALSE;
      $prefs_are_cached = FALSE;
      sqsession_unregister('prefs_are_cached');
      sqsession_unregister('prefs_cache');
   }
   return $otps;
}



/**
  * Encrypts and stores a user's current
  * list of OTPs in user preferences; if usernames
  * are also being stored, they go into a global
  * preferences set prefixed by the real username
  *
  * @param string  $user         The user for which to store OTPs
  * @param array   $otps         A list of the unencrypted OTPs
  * @param boolean $otp_usenames Whether or not usernames are also
  *                              part of OTPs
  *
  */
function put_user_otps($user, $otps, $otp_usernames)
{
   global $data_dir;
   if (empty($otps))
   {
      setPref($data_dir, $user, 'allow_otp', 0);
      if (!$otp_usernames)
         removePref($data_dir, $user, 'one_time_passwords');
      else
      {
         // first make sure main SM prefs cache is empty
         //
         global $prefs_cache, $prefs_are_cached;
         $prefs_cache = FALSE;
         $prefs_are_cached = FALSE;
         sqsession_unregister('prefs_are_cached');
         sqsession_unregister('prefs_cache');


         // retrieve system-wide OTP list
         //
         $all_otps = getPref($data_dir, 'one_time_passwords', 'one_time_passwords', '');
         if (!empty($all_otps))
         {
            $all_otps = otp_crypto($all_otps, 'decrypt');
            if (($all_otps = @unserialize($all_otps)) === FALSE)
               $all_otps = array();
            else
               $all_otps = explode('|', $all_otps);
         }
         else
            $all_otps = array();


         // now remove all OTPs for the current user
         //
         $new_all_otps = array();
         foreach ($all_otps as $otp)
         {
            if (strpos($otp, $user . ';') === 0)
               continue;
            $new_all_otps[] = $otp;
         }
         $new_all_otps = serialize(implode('|', $new_all_otps));
         $new_all_otps = otp_crypto($new_all_otps, 'encrypt');
         setPref($data_dir, 'one_time_passwords', 'one_time_passwords', $new_all_otps);


         // make sure main SM prefs cache is empty again
         //
         $prefs_cache = FALSE;
         $prefs_are_cached = FALSE;
         sqsession_unregister('prefs_are_cached');
         sqsession_unregister('prefs_cache');
      }
   }
   else
   {
      if (!$otp_usernames)
      {
         $otps = serialize(implode('|', $otps));
         $otps = otp_crypto($otps, 'encrypt');
         setPref($data_dir, $user, 'one_time_passwords', $otps);
      }
      else
      {
         // first make sure main SM prefs cache is empty
         //
         global $prefs_cache, $prefs_are_cached;
         $prefs_cache = FALSE;
         $prefs_are_cached = FALSE;
         sqsession_unregister('prefs_are_cached');
         sqsession_unregister('prefs_cache');


         // retrieve system-wide OTP list
         //
         $all_otps = getPref($data_dir, 'one_time_passwords', 'one_time_passwords', '');
         if (!empty($all_otps))
         {
            $all_otps = otp_crypto($all_otps, 'decrypt');
            if (($all_otps = @unserialize($all_otps)) === FALSE)
               $all_otps = array();
            else
               $all_otps = explode('|', $all_otps);
         }
         else
            $all_otps = array();


         // now remove all OTPs for the current user
         //
         $new_all_otps = array();
         foreach ($all_otps as $otp)
         {
            if (strpos($otp, $user . ';') === 0)
               continue;
            $new_all_otps[] = $otp;
         }


         // now re-add OTPs for the current user
         //
         foreach ($otps as $otp)
         {
            $new_all_otps[] = $user . ';' . $otp;
         }
         $new_all_otps = serialize(implode('|', $new_all_otps));
         $new_all_otps = otp_crypto($new_all_otps, 'encrypt');
         setPref($data_dir, 'one_time_passwords', 'one_time_passwords', $new_all_otps);


         // make sure main SM prefs cache is empty again
         //
         $prefs_cache = FALSE;
         $prefs_are_cached = FALSE;
         sqsession_unregister('prefs_are_cached');
         sqsession_unregister('prefs_cache');
      }
   }
}



/**
  * Check for a matching OTP username and password
  *
  * ONLY for use when $otp_usernames is enabled!
  *
  * @param string  $user       The username to check
  * @param string  $pass       The password to check
  * @param boolean $only_first Whether or not to only
  *                            check the first OTP for
  *                            each real user (OPTIONAL;
  *                            default = FALSE)
  *
  * @return mixed If no match was found, FALSE is returned.
  *               If a match was found, an array is returned,
  *               where the first element is the matching
  *               user's real username and the second element
  *               is an updated OTP list without the matching
  *               OTP.
  *
  */
function find_otp_username_password($user, $pass, $only_first=FALSE)
{ 

   $return_value = FALSE;


   // first make sure main SM prefs cache is empty
   //
   global $data_dir, $prefs_cache, $prefs_are_cached;
   $prefs_cache = FALSE;
   $prefs_are_cached = FALSE;
   sqsession_unregister('prefs_are_cached');
   sqsession_unregister('prefs_cache');


   // retrieve system-wide OTP list
   //
   $all_otps = getPref($data_dir, 'one_time_passwords', 'one_time_passwords', '');
   if (!empty($all_otps))
   {
      $all_otps = otp_crypto($all_otps, 'decrypt');
      if (($all_otps = @unserialize($all_otps)) === FALSE)
         $all_otps = array();
      else
         $all_otps = explode('|', $all_otps);
   }
   else
      $all_otps = array();


   // now look for an OTP that matches
   //
   $matched = FALSE;
   $new_otps = array();
   $last_user = '';
   foreach ($all_otps as $otp)
   {
      $pos = strpos($otp, ';');
      $current_user = substr($otp, 0, $pos);

      if ($current_user != $last_user)
         if ($matched)
            break;
         else
            $new_otps = array();

      else if ($only_first)
      {
         $new_otps[] = substr($otp, $pos + 1);
         continue;
      }

      $last_user = $current_user;

      if (!$matched && preg_match('/^.+;' . $user . ';' . $pass . '$/', $otp))
      {
         $return_value = array($current_user);
         $matched = TRUE;
      }
      else
         $new_otps[] = substr($otp, $pos + 1);
   }
   if ($matched)
      $return_value[] = $new_otps;


   // make sure main SM prefs cache is empty again
   //
   $prefs_cache = FALSE;
   $prefs_are_cached = FALSE;
   sqsession_unregister('prefs_are_cached');
   sqsession_unregister('prefs_cache');


   return $return_value;

}



/**
  * Retrieves (from user preferences) and
  * decrypts a user's real password
  *
  * @param string The user for which to get the real password
  *
  * @return string The unencrypted real user password
  *
  */
function get_user_real_pwd($user)
{ 
   global $data_dir;
   $real_pwd = getPref($data_dir, $user, 'real_password', '');
   return otp_crypto($real_pwd, 'decrypt');
}



/**
  * Encrypts and stores the CURRENT
  * LOGGED IN user's real password
  * in user preferences
  *
  * @param string  $user   The user for which to store
  *                        the real password (must match
  *                        the username of the currently
  *                        logged in user unless the
  *                        password is being removed)
  * @param boolean $remove When TRUE, removes the password
  *                        from the user's preferences
  *                        (OPTIONAL; default = TRUE).
  *
  */
function put_user_real_pwd($user, $remove=TRUE)
{
   global $data_dir;
   if ($remove)
   {
      removePref($data_dir, $user, 'real_password');
   }
   else
   {
      global $username;
      if (!sqGetGlobalVar('user_is_logged_in', $user_is_logged_in, SQ_SESSION)
       || !$user_is_logged_in || $user !== $username)
      {
         echo '<html><body><font color="red"><b>INCORRECT AND INSECURE USE OF One-Time Password put_user_real_pwd() FUNCTION!</b></font></body></html>';
         exit;
      }
      $real_pwd = otp_crypto(sqauth_read_password(), 'encrypt');
      setPref($data_dir, $user, 'real_password', $real_pwd);
   }
}



/**
  * Built-in PHP RC4 Encryption/Decryption Routine
  * 
  * Copied from the Vadmin plugin
  * 
  * This function stolen nearly verbatim from hastymail.
  * http://hastymail.sf.net/
  * 
  */
function otp_rc4_crypt($input, $key)
{
    $k_tmp = preg_split('//', $key, -1, PREG_SPLIT_NO_EMPTY);
    foreach($k_tmp as $char) {
        $k[] = ord($char);
    }
    unset($k_tmp);
    $message = preg_split('//', $input, -1, PREG_SPLIT_NO_EMPTY);
    $rep = count($k);
    for ($n=0;$n<$rep;$n++) {
        $s[] = $n;
    }
    $i = 0;
    $f = 0;
    for ($i = 0;$i<$rep;$i++) {
        $f = (($f + $s[$i] + $k[$i]) % $rep);
        $tmp = $s[$i];
        $s[$i] = $s[$f];
        $s[$f] = $tmp;
    }
    $i = 0;
    $f = 0;
    foreach($message as $letter) {
        $i = (($i + 1) % $rep);
        $f = (($f + $s[$i]) % $rep);
        $tmp = $s[$i];
        $s[$i] = $s[$f];
        $s[$f] = $tmp;
        $t = $s[$i] + $s[$f];
        $done = ($t^(ord($letter)));
        $i++;
        $f++;
        $enc_array[] = chr($done);
    }
    $coded = implode('', $enc_array);
    return $coded;
}



/**
  * Encrypt/Decrypt Data
  * 
  * Copied (and slightly modified) from the Vadmin plugin
  * 
  * This function does the encryption and decryption of sensitive
  * data stored on the HDD. It uses the CRYPTO_HASH_LINE as key.
  * See docs for more info and/or explanation of how this works.
  *
  * @param $input  The contents to encrypt/decrypt.
  * @param $mode   A string that can be either "decrypt" or "encrypt"
  *                depending on which action needs to be performed.
  * @return string The results of encryption/decryption.
  */
function otp_crypto($input, $mode)
{
    $CRYPTO_HASH_LINE   = get_otp_crypto_hash_phrase();
    $MCRYPT_ALGO        = get_otp_crypto_algorithm();

    if ($CRYPTO_HASH_LINE == 'Llama')
    {
        echo '<html><body><font color="red"><b>FATAL ERROR: '
           . _("The One-Time Password plugin's apache.conf is misconfigured: the cryptographic hash phrase has not been changed from its default value")
           . '</b></font></body></html>';
        exit;
    }

    /**
     * See if we have everything needed for
     * encryption/decryption. This includes checking for the
     * encryption functions, the algorithm, and the hash line.
     */
    if (!$CRYPTO_HASH_LINE || !$MCRYPT_ALGO){
        $message = '';
        if (!$CRYPTO_HASH_LINE){
            $message .= _("Could not find cryptographic hash phrase.");
        }
        if (!$MCRYPT_ALGO){
            if (!empty($message)) $message .= ' ';
            $message .= _("Could not find cryptographic algorithm.");
        }
        echo '<html><body><font color="red"><b>FATAL ERROR: '
           . sprintf(_("The One-Time Password plugin is misconfigured: %s"),$message)
           . '</b></font></body></html>';
        exit;
    }
    $key = $CRYPTO_HASH_LINE;
    if ($MCRYPT_ALGO == 'rc4_builtin'){
        switch($mode){
        case 'encrypt':
            $endresult = base64_encode(otp_rc4_crypt($input, $key));
            break;
        case 'decrypt':
            $endresult = otp_rc4_crypt(base64_decode($input), $key);
            break;
        }
    } else {
        if (!function_exists('mcrypt_generic')){
            $message = _("An algorithm other than 'rc4_builtin' specified, but mcrypt support not found.");
            echo '<html><body><font color="red"><b>FATAL ERROR: '
               . sprintf(_("The One-Time Password plugin is misconfigured: %s"),$message)
               . '</b></font></body></html>';
            exit;
        }
        $td = mcrypt_module_open($MCRYPT_ALGO, '', MCRYPT_MODE_ECB, '');
        $iv = mcrypt_create_iv(mcrypt_enc_get_iv_size ($td), MCRYPT_RAND);
        @mcrypt_generic_init($td, $key, $iv);
        switch ($mode){
        case 'encrypt':
            $endresult = base64_encode(mcrypt_generic($td, $input));
            break;
        case 'decrypt':
            $endresult = mdecrypt_generic($td, base64_decode($input));
            /**
             * There will be trailing nul's on the end. Remove
             * these. They suck.
             */
            $endresult = rtrim($endresult);
            break;
        }
        mcrypt_generic_deinit($td);
    }
    return $endresult;
}



