<?php
/**
 * gpg_key_functions.php
 * -----------
 * GPG plugin functions file, as defined by the SquirrelMail-1.2 API.
 * Updated for the SM 1.3/1,4 API
 *
 * Copyright (c) 1999-2003 The SquirrelMail development team
 * Licensed under the GNU GPL. For full terms see the file COPYING.
 *
 * Copyright (c) 2002-2003 Braverock Ventures
 *
 * $Id: gpg_key_functions.php,v 1.9 2003/04/02 21:52:19 brian Exp $
 *
 */
/*********************************************************************/

/**
 * function gpg_list_keys
 * This function is the generic key lister for the plugin
 * it is used for trusted_key operations, as well as key signing
 *
 *
 * Add switches to this function to modify the gpg command line
 *
 * The format of the --with-colons parameter is described in detail in
 * the file named DETAILS in the gpg distribution.
 *
 * @param $debug, $search_string, $with_colons, $with_fingerprint
 * @return string with output, parse string on the UI.
 * Note the departure from the convention of putting the $debug flag last:
 * $debug is first in this function to allow easy adding of additional switches
 * without breaking things.
 */
function gpg_list_keys($debug, $search_string, $with_colons, $with_fingerprint, $keyring_type='public'){

  /**
   * The long key id is retrieved by using
   * gpg --list-keys --with-colons
   * pub:-:1024:17:257F139F72E1465E:2001-09-27:::-:Brian G. Peterson <brian@braverock.com>::scESC:
   */

  global $gpg_key_file;
  global $gpg_key_dir;
  global $path_to_gpg;

  $command = "$path_to_gpg --homedir $gpg_key_dir ";

  /**
   * We will us the $keyring_type parameter to determine
   * what to add the the $command string.
   *
   * Legal Values are 'public', 'private', and 'all'
   *
   * From the gpg man page
   *  --list-keys [names]
   *
   *  --list-public-keys [names]
   *        List  all  keys  from  the  public keyrings, or just the ones
   *        given on the command line.
   *
   *  --list-secret-keys [names]
   *        List all keys from the secret  keyrings,  or  just  the  ones
   *        given on the command line.
   */
  switch ($keyring_type) {
    case 'all':
        $command .= '--list-keys ';
        break;
    case 'public':
        $command .= '--list-public-keys ';
        break;
    case 'private':
        $command .= '--list-secret-keys ';
        break;
  };

  if ($with_colons == 'true') {
      $command .= '--with-colons ';
  };

  if ($with_fingerprint == 'true') {
      $command .= '--with-fingerprint ';
  };

  if ($search_string != '') {
      $search_string = escapeshellarg($search_string);
      $command .= "$search_string";
  }

  // wrap it up by redirecting the output
  // to stderr using 2>&1
  $command .= " 2>&1";

  if ($debug) {
   	  echo $command;
  };

  exec($command, $list_text, $returnval);


  //clean this up later, but for now just return the array
  return $list_text;

};

/*********************************************************************/
/**
 * function gpg_keyserver_findkey
 *
 * This function will search for keys on a public keyserver
 *
 * Ideally, we would use a command like:
 * gpg --keyserver wwwkeys.pgp.net --search-keys brian@braverock.com
 *
 * BUT: this command is interactive, expecting input, and I can't
 * find anything about using it in batch mode. if you try --batch
 * you get : gpg: Sorry, we are in batchmode - can't get input
 * so we would have to use a cycle to get the output and parse it
 * then use the gpg_recv_key function
 *
 * SO: until we figure out how to parse the HKP output,
 *     or use the LDAP interface
 *
 * We will first take an email address or keyid $search_keyid
 * and look on the keyserver using http, like this:
 *
 * http://pgp.mit.edu:11371/pks/lookup?op=index&search=$search_keyid
 * http://stinkfoot.org:11371/pks/lookup?op=index&search=$search_keyid
 * http://wwwkeys.pgp.net:11371/pks/lookup?op=index&search=$search_keyid
 * http://wwwkeys.eu.pgp.net:11371/pks/lookup?op=index&search=$search_keyid
 *
 * Then, we will parse the output, and place it in an array
 * for display and selection.
 *
 * Information on public keyservers may be found at
 *
 * http://www.vcnet.com/~rossde/pgp_keyserv.html
 *
 * Finally, we will return an array of the key(s) to the interface
 * so the user may select one or more for import.
 *
 * It would be nice to use the HKP or LDAP interfaces too,
 * but this will work for now.
 *
 * @param $search_keyid,$keyserver,$debug
 * @return array $returnkeys
 */
function gpg_keyserver_findkey($search_keyid,$debug) {

	$keyserver = "";
	// get the user's prefered keyserver
	$keyserver = getPref($data_dir,$username,'keyserver');

	// set a default keyserver if we don't pass one in
	if (!$keyserver) {
		$keyserver="pgp.mit.edu:11371";
	};

	//process the search_keyid, hack this for now
	//eventaully, break search_keyid into multiple searches.


	//start our output formatting
	$f = fopen("http://" . escapeshellcmd($keyserver) . "/pks/lookup?op=index&search=" . urlencode($search_keyid), "r");
	if ($debug) {
	echo "http://" . escapeshellcmd($keyserver) . "/pks/lookup?op=index&search=" . urlencode($search_keyid);
	};
        $lastkey = "";
	// The first 4 lines contain header info, skip these
        fgets($f,512);
        fgets($f,512);
        fgets($f,512);
        fgets($f,512);

        $returnkeys = array();
	$ret = array();

        while($line=fgetss($f,512)) {
                $line=str_replace("&gt;",">",$line);
                $line=str_replace("&lt;","<",$line);
                $line=str_replace("&amp;",'"',$line);
                $line=str_replace("&quot;","&",$line);

		  if (ereg("^pub[[:space:]]+([[:digit:]]+[R|D])/([[:alnum:]]+)[[:space:]]+([[:digit:]]+)/([[:digit:]]+)/([[:digit:]]+)[[:space:]]+(.*)", $line, $tmp)) {
			// foreach ($tmp as $key => $value) echo "<br>regex $key = $value";
                        $lastkey = $tmp[2];
                        $lastdate = $tmp[3] . "-" . $tmp[4] . "-" . $tmp[5];
                        $line = $tmp[6]; # fool myself

                }
                if (ereg("(.+)<(.+@.+)>", $line, $t)) {
                        $tmpemail = $t[2];
                        $tmpname = $t[1];

			$ret['tmpname'] = $tmpname;
			$ret['tmpemail'] = $tmpemail;
			$ret['lastkey'] = $lastkey;
			$ret['lastdate'] = $lastdate;
			$returnkeys[] = $ret;
		}
        }

	if ($debug) {
		foreach ($returnkeys as $key) {
	  		echo "<br>-";
	  		foreach ($key as $field => $value) echo "<br>$field=$value";
		};
	};

	return ($returnkeys);

};

/*********************************************************************/

/**
 * function gpg_import_key
 * This function imports PGP or GPG
 * ASCII Armored Keys
 *
 * @param $keystring,$debug
 * @return $key_import_output_str
 */
function gpg_import_key($keystring,$debug){


  //set the keyring file to be $username.gpgpubring
  $username = $_SESSION['username'];
  global $gpg_key_dir;
  global $path_to_gpg;

  //$gpg_key_file="$username.gpgpubkeyring";

  if ($debug) {
	echo "<br>Username: $username";
  	echo "<br>DataDir: $gpg_key_dir";
  	echo "<br>Keyfile: $gpg_key_dir/$gpg_key_file";
  };

  // clean input
  // make sure there aren't any funny characters in here
  $keystring = escapeshellarg ($keystring);

  // import
  // the following line redirects the output to stderr: 2>&1
  // use --import
  //$command = "echo $keystring | $path_to_gpg --import --batch --no-tty --homedir $gpg_key_dir --no-default-keyring --keyring $data_dir/$gpg_key_file 2>&1";
  $command = "echo $keystring | $path_to_gpg --allow-secret-key-import --import --batch --no-tty --homedir $gpg_key_dir 2>&1";

  if ($debug) {
	echo "<hr>$command<hr>";
  };

  exec($command, $output, $returnval);

  // make the result a string
  if (is_array($output)) {
      $key_import_output_str = implode($output,"\n");
    };

  if ($debug) {
	echo "$key_import_output_str";
	echo " returnvalue= $returnval";
  	echo "end of gpg_import_key";
  };

  //return the output string to our calling function
  return ($key_import_output_str);
};

/*********************************************************************/

/**
 * function gpg_recv_key
 * This function imports PGP or GPG
 * keys from a public keyserver
 *
 * @param $searchkeyid,$debug,$keyserver
 * @return $recv_key_output_str
 */
function gpg_recv_key($searchkeyid,$debug,$keyserver){


  //set the keyring file to be $username.gpgpubring
  $username = $_SESSION['username'];
  global $gpg_key_dir;
  global $path_to_gpg;

  //$gpg_key_file="$username.gpgpubkeyring";

  if ($debug) {
	echo "<br>Username: $username";
  	echo "<br>DataDir: $gpg_key_dir";
  	echo "<br>Keyfile: $gpg_key_dir/$gpg_key_file";
  };

  // clean input
  // make sure there aren't any funny characters in here
  $searchkeyid = escapeshellarg ($searchkeyid);
  if (!$keyserver) {
    $keyserver = getPref($data_dir,$username,'keyserver');
  }
  // import
  // the following line redirects the output to stderr: 2>&1
  // use --import
  $command = "$path_to_gpg --batch --no-tty --homedir $gpg_key_dir --keyserver hkp://$keyserver --recv-key $searchkeyid 2>&1";

  if ($debug) {
	echo "<hr>$command<hr>";
  };

  exec($command, $output, $returnval);

  // make the result a string
  if (is_array($output)) {
      $recv_key_output_str = implode($output,"\n");
    };

  if ($debug) {
	echo "$recv_key_output_str";
	echo " returnvalue= $returnval";
  	echo "end of gpg_recv_key";
  };

  //return the output string to our calling function
  return ($recv_key_output_str);
};

/*********************************************************************/

/**
 * function gpg_import_keyring
 * This function imports PGP or GPG keyrings
 *
 * @param file handle $p_keyring
 * @return
 */
function gpg_import_keyring($imp_keyring,$debug){
  /**
   * Import keyrings contained in $imp_keyring
   *
   */

  //pull some global variables that we need
  global $gpg_key_dir;
  global $path_to_gpg;
  $username = $_SESSION['username'];

  if ($debug) {
 	echo "<br>Username: $username";
	echo "<br>DataDir: $data_dir";
	echo "<br>Keyfile: $data_dir/$gpg_key_file";
  };

  // clean input
  // make sure there aren't any funny characters in here
  $imp_keyring = escapeshellarg ($imp_keyring);

  if ($debug) {
	echo "Temporary Keyring Path: $imp_keyring";
  };

  // import
  // the following line redirects the output to stderr: 2>&1
  // use --import
  $command = "$path_to_gpg --batch --no-tty --homedir $gpg_key_dir --allow-secret-key-import --import $imp_keyring 2>&1";

  if ($debug) {
	echo "<hr>$command<hr>";
  };

  exec($command, $output, $returnval);

  // make the result a string
  if (is_array($output)) {
      $keyring_import_output_str = implode($output,"\n");
    };

  if ($debug) {
	echo" <textarea cols=80 rows=25 name=output>$output_str</textarea>";
	echo "<br>$keyring_import_output_str";

	echo "<br>returnvalue= $returnval";
	echo "<br>end of gpg_import_keyring";
  };
  return ($keyring_import_output_str);

};

/*********************************************************************/

/**
 * function gpg_generate_keypair
 * This function generates a keypair
 *
 * @param
 * @return
 *
 * Unattended key generation
 * =========================
 * This feature allows unattended generation of keys controlled by a
 * parameter file.  To use this feature, you use --gen-key together with
 * --batch and feed the parameters either from stdin or from a file given
 * on the commandline.
 *
 * The format of this file is as follows:
 *   o Text only, line length is limited to about 1000 chars.
 *   o You must use UTF-8 encoding to specify non-ascii characters.
 *   o Empty lines are ignored.
 *   o Leading and trailing spaces are ignored.
 *   o A hash sign as the first non white space character indicates a comment line.
 *   o Control statements are indicated by a leading percent sign, the
 *     arguments are separated by white space from the keyword.
 *   o Parameters are specified by a keyword, followed by a colon.  Arguments
 *     are separated by white space.
 *   o The first parameter must be "Key-Type", control statements
 *     may be placed anywhere.
 *   o Key generation takes place when either the end of the parameter file
 *     is reached, the next "Key-Type" parameter is encountered or at the
 *     control statement "%commit"
 *   o Control statements:
 *     %echo <text>
 * 	Print <text>.
 *     %dry-run
 * 	Suppress actual key generation (useful for syntax checking).
 *     %commit
 * 	Perform the key generation.  An implicit commit is done
 * 	at the next "Key-Type" parameter.
 *     %pubring <filename>
 *     %secring <filename>
 * 	Do not write the key to the default or commandline given
 * 	keyring but to <filename>.  This must be given before the first
 * 	commit to take place, duplicate specification of the same filename
 * 	is ignored, the last filename before a commit is used.
 * 	The filename is used until a new filename is used (at commit points)
 * 	and all keys are written to that file.	If a new filename is given,
 * 	this file is created (and overwrites an existing one).
 * 	Both control statements must be given.
 *    o The order of the parameters does not matter except for "Key-Type"
 *      which must be the first parameter.  The parameters are only for the
 *      generated keyblock and parameters from previous key generations are not
 *      used. Some syntactically checks may be performed.
 *      The currently defined parameters are:
 *      Key-Type: <algo-number>|<algo-string>
 * 	Starts a new parameter block by giving the type of the
 * 	primary key. The algorithm must be capable of signing.
 * 	This is a required parameter.
 *      Key-Length: <length-in-bits>
 * 	Length of the key in bits.  Default is 1024.
 *      Key-Usage: <usage-list>
 *         Space or comma delimited list of key usage, allowed values are
 *         "encrypt" and "sign".  This is used to generate the key flags.
 *         Please make sure that the algorithm is capable of this usage.
 *      Subkey-Type: <algo-number>|<algo-string>
 * 	This generates a secondary key.  Currently only one subkey
 * 	can be handled.
 *      Subkey-Length: <length-in-bits>
 * 	Length of the subkey in bits.  Default is 1024.
 *      Subkey-Usage: <usage-list>
 *         Similar to Key-Usage.
 *      Passphrase: <string>
 * 	If you want to specify a passphrase for the secret key,
 * 	enter it here.	Default is not to use any passphrase.
 *      Name-Real: <string>
 *      Name-Comment: <string>
 *      Name-Email: <string>
 * 	The 3 parts of a key. Remember to use UTF-8 here.
 * 	If you don't give any of them, no user ID is created.
 *      Expire-Date: <iso-date>|(<number>[d|w|m|y])
 * 	Set the expiration date for the key (and the subkey).  It
 * 	may either be entered in ISO date format (2000-08-15) or as
 * 	number of days, weeks, month or years. Without a letter days
 * 	are assumed.
 *      Preferences: <string>
 *         Set the cipher, hash, and compression preference values for
 * 	this key.  This expects the same type of string as "setpref"
 * 	in the --edit menu.
 *
 * Here is an example:
 * $ cat >foo <<EOF
 *      %echo Generating a standard key
 *      Key-Type: DSA
 *      Key-Length: 1024
 *      Subkey-Type: ELG-E
 *      Subkey-Length: 1024
 *      Name-Real: Joe Tester
 *      Name-Comment: with stupid passphrase
 *      Name-Email: joe@foo.bar
 *      Expire-Date: 0
 *      Passphrase: abc
 *      %pubring foo.pub
 *      %secring foo.sec
 *      # Do a commit here, so that we can later print "done" :-)
 *      %commit
 *      %echo done
 * EOF
 * $ gpg --batch --gen-key -a foo
 *  [...]
 *
 *
 * echo "Key-Type: DSA Key-Length: 1024 Subkey-Type: ELG-E Subkey-Length: 2048 \
 * 		Name-Real: Joe Tester Name-Email: joe@foo.bar Passphrase: abc" \
 *		| gpg --batch --gen-key --armor --homedir $gpg_key_dir
 *
 */
function gpg_generate_keypair($debug, $real_name, $email, $passphrase, $comment = '',
                         $keylength = 1024)
    {
    	// set up our return error and warning arrays.
    	$return['warnings'] = array();
    	$return['errors'] = array();

        /* Check for secure connection.
         *
         * If we don't have a secure connection, return an error
         * string to be displayed by MakePage
         */
		$https_check=0;
		$https_check=gpg_https_connection ();
		if (!https_check) {
		   $line = "You are not using a secure connection. SSL connection required to use passphrase functions.";
		   $return['errors'][] = $line;
		   return ($return);
		   exit;
		};


         $pipes = array(
		    0 => array("pipe", "r"),  // stdin is a pipe that the child will read from
		    1 => array("pipe", "w"),  // stdout is a pipe that the child will write to
		    2 => array("pipe", "a"), // stderr is a file to write to
			);


        /* Create the config file necessary for GnuPG to run in batch mode.
         * there is a difference between using 'r' and 'w' in popen
         * the Horde team uses 'w', but I think that we can use 'r' to
         * get the output of the command without having to write to temp
         * files.  Even without using 'r', we should be able to check and
         * see if a new key was generated and added to the default keyring.
         */

        $fp = popen($path_to_gpg . " --gen-key --batch --armor --homedir $gpg_key_dir", 'r');
        /* Key-Type 20 = El Gamal + DSA Key */
        fputs($fp, "Key-Type: 20\n");
        fputs($fp, "Key-Length: " . $keylength . "\n");
        fputs($fp, "Name-Real: " . $real_name . "\n");
        fputs($fp, "Name-Comment: Key generated on public webmail server". "\n");
        fputs($fp, "Name-Email: " . $email . "\n");
        fputs($fp, "Expire-Date: 0\n");
        fputs($fp, "Passphrase: " . $passphrase . "\n");
        fputs($fp, "%commit\n");
        $output = fread($fp);
        pclose($fp);

        /*
         * Check the $output to see if it contains what we want.
         *
         * Here is where we need some parsing code...
         *
         */

         //return a string that indicates success or an error message to be displayed to the user.
         return ($output);

         //return array('public' => $public_key, 'private' => $secret_key);
    }

/***************************************************/
/*
 * $Log: gpg_key_functions.php,v $
 * Revision 1.9  2003/04/02 21:52:19  brian
 * - applied patch to use --allow-secret-key-import in functions
 *   - gpg_import_key
 *   - gpg_import_keyring
 * - patch tested under gpg v 1.0.7 and 1.2.1 before cvs commit
 *   - no errors or warnings
 *   - import of secret keys worked as expected
 *   - patch deemed safe for inclusion in main code trunk
 * Bug 16
 *
 * Revision 1.8  2003/03/27 20:56:32  brian
 * added https check in key generation function
 *
 * Revision 1.7  2003/03/27 12:53:24  brian
 * updates to key generation function
 *
 * Revision 1.6  2003/03/25 16:12:44  brian
 * updates to key generation functions.
 *
 * Revision 1.5  2003/03/12 23:46:27  brian
 * fixed --list-secret-keys
 * from  --list-private-keys (bug)
 *
 * Revision 1.4  2003/03/12 22:16:45  tyler
 * - gpg_recv_key function modified to honor the chosen keyserver
 *
 * Revision 1.3  2003/03/12 17:07:55  brian
 * added optional $keyring_type parameter to gpg__list_keys function
 *
 * Revision 1.2  2003/03/12 16:27:12  brian
 * added $Log directive to broken out functions file.
 *
 */
?>
