<?php
/**
 * gpg_encrypt_functions.php
 * -----------
 * GPG plugin functions file, as defined by the SquirrelMail-1.2 API.
 * Updated for the SM 1.3/1,4 API
 *
 * Copyright (c) 1999-2003 The SquirrelMail development team
 * Licensed under the GNU GPL. For full terms see the file COPYING.
 *
 * Copyright (c) 2002-2003 Braverock Ventures
 *
 * $Id: gpg_encrypt_functions.php,v 1.22 2003/04/08 16:26:16 brian Exp $
 *
 */

/*********************************************************************/
/**
 * function gpg_parse_address
 *
 * This function will parse the address correctly into a
 * recipient list for use by the calling page.
 *
 * Squirrelmail v1.2.x's address parsing functions fail silently on
 * malformed or otherwise obnoxious email addresses.
 * We use them for compatibility and hope they will be upgraded later.
 *
 * Be careful what you wish for.  SM 1.4 has made the address
 * parsing more robust, at the cost of simplicity and backwards compatibility.
 * Check which version we are running on, and call the correct functions.
 *
 * @param $send_to, $send_to_cc, $send_to_bcc, $debug
 * @return array with results
 */
function gpg_parse_address ($send_to, $send_to_cc, $send_to_bcc, $debug){

    if ($debug) {
		echo '<br> Entering Address Parsing:<br>';
    }
    global $version;

	if (substr($version, 2,4) >= 3.1) {
		//parse using SM 1.3.1+ functions from rfc822header

		$valid_addresses = array();

		$abook = addressbook_init(false, true);
		$rfc822_header = new Rfc822Header;
		$rfc822_header->to = $rfc822_header->parseAddress($_POST['send_to'],
			   true,array(), '', $domain, array(&$abook,'lookup'));
		$rfc822_header->cc = $rfc822_header->parseAddress($_POST['send_to_cc'],
			   true,array(), '',$domain,array(&$abook,'lookup'));
		$rfc822_header->bcc = $rfc822_header->parseAddress($_POST['send_to_bcc'],
			   true,array(), '',$domain, array(&$abook,'lookup'));

	    $to = array();
	    $cc = array();
	    $bcc = array();

		foreach (($rfc822_header->to ) as $value) $to[] = $value->mailbox . "@" . $value->host;
		foreach (($rfc822_header->cc ) as $value) $cc[] = $value->mailbox . "@" . $value->host;
		foreach (($rfc822_header->bcc) as $value) $bcc[] = $value->mailbox . "@" . $value->host;

		$parsed_addr = array_merge($to, $cc, $bcc);

	    $valid_addresses = $parsed_addr;

	    //fix display under SM 1.4.0
	    $send_to_str = htmlspecialchars ($_POST['send_to']);
	    $send_to_cc_str = htmlspecialchars ($_POST['send_to_cc']);
	    $send_to_bcc_str = htmlspecialchars ($_POST['send_to_bcc']);
	    gpg_setglobal ( 'send_to' , $send_to_str );
	    gpg_setglobal ( 'send_to_cc' , $send_to_cc_str );
	    gpg_setglobal ( 'send_to_bcc' , $send_to_bcc_str );


		//TODO: add debug display for SM 1.4 code
		//end SM v >=1.3.1 processing
    } else {
		//parse using SM 1.2.x functions

		// show the results of the address expansion if debug is on
		if ($debug) {
			echo '<br>Debug address Expansion for SM v 1.2.x <br>';
			$debug_to_addr  = explode (',', $send_to);
			$debug_cc_addr  = explode (',', $send_to_cc);
			$debug_bcc_addr = explode (',', $send_to_bcc);
			$debug_addr = array_merge($debug_to_addr, $debug_cc_addr, $debug_bcc_addr);

			$j = parseAddrs($debug_addr);
			foreach ($j as $val) echo '<br>Parse ', $val;
			$k = expandRcptAddrs(parseAddrs($debug_addr));
			foreach ($k as $val) echo '<br>Expand ', $val;
			echo '<br><hr>';
		};

		//call expand and parse for real
		$to_addr = expandRcptAddrs(parseAddrs($send_to));
		$cc_addr = expandRcptAddrs(parseAddrs($send_to_cc));
		$bcc_addr = expandRcptAddrs(parseAddrs($send_to_bcc));
		$parsed_addr = array_merge($to_addr, $cc_addr, $bcc_addr);

		//create the $valid_addresses array.
		foreach ($parsed_addr as $key => $value) {
			if (eregi("<(.+)@(.+)>", $value, $matches)) {
				$valid_addresses[] =
				escapeshellarg($matches[1] . "@" . $matches[2]);
			}; //end eregi processing
		}; //end foreach
	    //end SM v 1.2.x processing
	};

	return $valid_addresses;

};

/*********************************************************************/
/**
 * function gpg_encrypt
 * This function does the encryption
 * This is the workhorse of the encryption side of the plugin
 *
 * Add code here to use user preferences to modify the gpg command line
 *
 * @param $body,$send_to
 * @return array with results
 */
function gpg_encrypt($body,$send_to_list,$debug){

	// set up globals
	global $trusted_key_id;
	global $gpg_key_file;
	global $gpg_key_dir;
	global $path_to_gpg;

	$username = $_SESSION['username'];

	if ($debug) {
		echo "<br>Global Key Dir: " . $gpg_key_dir;
		echo "<br>Username: $username";
	};


	/**
	* Hack the long key id
	*
	* Gotten by using
	* gpg --fingerprint --with-colons
	* pub:-:1024:17:257F139F72E1465E:2001-09-27:::-:Brian G. Peterson <brian@braverock.com>::scESC:
	* fpr:::::::::0C9E9AE5EB53699A77A9968E257F139F72E1465E:
	*/
	// Trusted Key Test
	$trusted_key_id = ''; //initialize empty
	$use_trusted_key_id = getPref($data_dir,$username,'use_trusted_key_id');
	if ($use_trusted_key_id == 'true') {
		 $trusted_key_id = escapeshellarg(getPref($data_dir,$username,'trusted_key_id'));
		 if ($debug) {
			   echo '<BR>use_trusted_key_id = true';
			   echo "<br>Trusted Key ID: $trusted_key_id";
			   echo '<hr>';
		 };
	} else {
		 if ($debug) {
			 echo '<BR>use_trusted_key_id = false';
		 };
	};

	//Encrypt to Self Test
	$encrypt_to_self=getPref($data_dir, $username, 'encrypt_to_self');
	if ($encrypt_to_self=='true') {
		 $self_encr_email = escapeshellarg(getPref($data_dir, $username, 'self_encr_email'));
		 // add the selected email address to the recipient list
		 $send_to_list .= " -r $self_encr_email";
		 if ($debug) {
			  echo '<BR>encrypt_to_self = true';
			  echo "<BR>Self Encrypt Email: $self_encr_email";
		 };
	} else {
	 if ($debug) {
		 echo '<BR>encrypt_to_self = false';
	 };
	};

	// add stuff in here to re-activate the 'corporate' shared keyring


	//plaintext debug box
	if ($debug) {
		 echo "<textarea cols=80 rows=10 name=plaintext>$body</textarea>";
		 echo '<hr>';
	};

	// clean the body string that is passed in
	// make sure that funny characters get
	// bracketed by single quotes and backslashes
	$body = escapeshellarg ($body);

	/********* gpg_encrypt Command String *********/
	/**
	* Build the command string in pieces, checking for the
	* existance of various preferences, and modifying the
	* command string accordingly
	*/
	//set up the base command
	$command = "echo $body | $path_to_gpg --batch --no-tty --encrypt --armor --homedir $gpg_key_dir";

	if ($use_trusted_key_id == 'true' && $trusted_key_id != '') {
    	$command .= " --trusted-key $trusted_key_id ";
	} else {
    	$command .= ' --always-trust ';
	};

	// wrap it up by setting the recipients to the sender list using -r
	// and redirect the output to stderr using 2>&1
	$command .= " -r $send_to_list 2>&1";


	if ($debug) {
		echo "<hr>Command String: $command<hr>";
	};

	exec($command, $cyphertext, $returnval);

	// make the result a string
	if (is_array($cyphertext)) {
	    $cyphertext_str = implode($cyphertext,"\n");
	};

	if ($debug) {
		echo " <textarea cols=80 rows=25 name=cyphertext>$cyphertext_str</textarea>";
		echo " returnvalue= $returnval";
	};

	//now parse the return value and return an array with some useful contents.
	$sep = '-----BEGIN PGP MESSAGE-----';

	list ($front, $cyphertext_tail) = explode ($sep, $cyphertext_str);

	if ($debug) {
		echo "<hr>Front: $front";
		echo "Back: $cyphertext_tail";
	};

	$returntext = "$sep $cyphertext_tail";

	$return['cyphertext'] = $returntext;
	$return['warnings'] = array();
	$return['errors'] = array();
	$return['skipped_keys'] = array();

	foreach ($cyphertext as $line) {
		if ($line == "-----BEGIN PGP MESSAGE-----") break;
		$j = substr_count (" " . $line, 'encryption failed');
		if ($j) {
		  $return['errors'][] = $line;
		}
		$j = substr_count (" " . $line, 'gpg: Warning:');
		if ($j) {
		  $return['warnings'][] = $line;
		};
		$j = substr_count (" " . $line, 'gpg: Error:');
		if ($j) {
		  $return['errors'][] = $line;
		};
		$j = substr_count (" " . $line, 'skipped: public key not found');
		if ($j) {
		  $return['skipped_keys'][] = $line;
		};
		$j = substr_count (" " . $line, 'gpg: Missing argument for option');
		if ($j) {
		  $return['errors'][] = $line;
		};
	};

	if ($debug) {
		echo "<hr>";
		echo "Cyphertext /n<pre>", $return['cyphertext'], "</pre>";
		foreach ($return['warnings'] as $warn) echo "<br>Warning $warn";
		foreach ($return['errors'] as $error) echo "<br>Error $error";
		foreach ($return['skipped_keys'] as $error) echo "<br>Skipped Keys $error";
	};
	return ($return);
	//should add code to filter out the errors/warnings we expect.

};

/*********************************************************************/
/**
 * function gpg_decrypt
 * This function does the decryption
 * This is the workhorse of the decryption side of the plugin
 *
 * Add code here to use user preferences to modify the gpg command line
 *
 * @param $body,$passphrase
 * @return array with results
 */
function gpg_decrypt($body,$passphrase,$debug){

	// set up globals
	global $gpg_key_file;
	global $gpg_key_dir;
	global $path_to_gpg;
	$no_signing_passwd = getPref ($data_dir, $username, 'no_signing_passwd');

	$username = $_SESSION['username'];

	if ($debug) {
		echo "<br>Global Key Dir: " . $gpg_key_dir;
		echo "<br>Username: $username";
	};

	// clean the body string that is passed in
	// make sure that funny characters get
	// bracketed by single quotes and backslashes
	$body = escapeshellarg ($body);

	// set up our return error and warning arrays.
	$return['warnings'] = array();
	$return['errors'] = array();

	/* Check for secure connection.
	 *
	 * If we don't have a secure connection, return an error
	 * string to be displayed by MakePage
	 */
	$https_check=0;
	$https_check=gpg_https_connection ();
	if (!https_check) {
	   $line = "You are not using a secure connection. SSL connection required to use passphrase functions.";
	   $return['errors'][] = $line;
	   return ($return);
	   exit;
	};

	/********* gpg_decrypt Command String *********/
	/**
	* Build the command string in pieces, checking for the
	* existance of various preferences, and modifying the
	* command string accordingly
	*/
	//set up the base command
	//build a command without the passphrase incase debug is set so
	//we dont go displaying the passphrase in a debug window
	$pre_pass =" | $path_to_gpg --passphrase-fd 0 --batch --no-tty --decrypt --homedir $gpg_key_dir 2>&1";
	$without_pass =" | $path_to_gpg --batch --no-tty --decrypt --homedir $gpg_key_dir 2>&1";

    //first check to see if they don't need a passphrase and set command accordingly
    if ($no_signing_passwd == 'true') {
	     //this will die an ugly death if the message is encrypted to a key
	     //that requires a passphrase
	     if ($debug) {
		 			echo "<br>No passphrase provided.\n";
		 			echo "<br>Command string: echo [Body]$without_pass<br>\n";
		 };
		 $command = "echo $body$without_pass";
	}

    //then check and see if they provided a passphrase anyway, and set if they did
	if ($passphrase) {
		 if ($debug) {
			echo "<br>Got passphrase.\n";
			echo "<br>Command string: echo [PassPhrase][Body]$pre_pass<br>\n";
		 };
		 //Make sure we escape naughty characters before passing to the shell
		 $passphrase = escapeshellarg($passphrase . "\n");
		 $command = "echo $passphrase$body$pre_pass";
	} elseif ($no_signing_passwd == 'false') {
		 // we should reopen the window or check for passphrase with javascript
		 // instead of just throwing an error here
           $line = "You did not provide a passphrase!";
     	   $return['errors'][] = $line;
    	   return ($return);
    	   exit;
	}

	exec($command, $plaintext, $returnval);

	// make the result a string
	if (is_array($plaintext)) {
	    $plaintext_str = implode($plaintext,"\n");
	};

	if ($debug) {
		echo " <textarea cols=80 rows=25 name=plaintext>$plaintext_str</textarea>";
		echo " returnvalue= $returnval";
	};

	/**
	 * now parse the return value and return an array with some useful contents.
	 * if there are no errors, and the message is not signed,
	 * we will only have the plaintext.
	 */

	$return['plaintext'] = $plaintext_str;
	$return['warnings'] = array();
	$return['errors'] = array();
	$return['info'] = array();

	foreach ($plaintext as $line) {
		if (substr_count ($line, 'Signature Status')) break;
		if (substr_count ($line, 'gpg: Good signature')) break;
		$j = 0;
		$j = substr_count ($line, 'gpg: encrypted with');
		if ($j) {
		  $return['info'][] = $line;
		};
		$j = substr_count ($line, 'decryption failed');
		if ($j) {
		  $return['errors'][] = $line;
		}
		$j = substr_count ($line, 'gpg: Warning:');
		if ($j) {
		  $return['warnings'][] = $line;
		};
		$j = substr_count ($line, 'gpg: Error:');
		if ($j) {
		  $return['errors'][] = $line;
		};
		$j = substr_count ($line, 'gpg: decryption failed:');
		if ($j) {
		  $return['errors'][] = $line;
		};
		$j = substr_count ($line, 'gpg: no valid OpenPGP data found.');
		if ($j) {
		  $return['errors'][] = $line;
		};
		$j = substr_count ($line, 'gpg: decrypt_message failed');
		if ($j) {
		  $return['errors'][] = $line;
		};
		$j = substr_count ($line, 'gpg: invalid radix64 character');
		if ($j) {
		  $return['warnings'][] = $line;
		};
		$j = substr_count ($line, 'gpg: CRC error');
		if ($j) {
		  $return['errors'][] = $line;
		};
		$j = substr_count ($line, 'invalid packet');
		if ($j) {
		  $return['errors'][] = $line;
		};
		$j = substr_count ($line, 'gpg: out of secure memory while allocating');
		if ($j) {
		  $return['errors'][] = $line;
		};
		$j = substr_count ($line, 'gpg: (this may be caused by too many');
		if ($j) {
		  $return['errors'][] = $line;
		};
	};

	if ($debug) {
		echo "<hr>";
		echo "Plaintext /n<pre>", $return['plaintext'], "</pre>";
		foreach ($return['warnings'] as $warn) echo "<br>Warning: $warn";
		foreach ($return['errors'] as $error) echo "<br>Error:  $error";
		foreach ($return['info'] as $info) echo   "<br>Info:   $info";
	};
	return ($return);
	//should add code to filter out the errors/warnings we expect.

};

/*********************************************************************/
/*
 * $Log: gpg_encrypt_functions.php,v $
 * Revision 1.22  2003/04/08 16:26:16  brian
 * added htmlspecialchars to gpg_parse_addr funtion to fix display bug in SM 1.4.0
 *
 * Revision 1.21  2003/04/08 15:16:52  brian
 * - added more debug statements to gpg_decrypt
 * - load no_signing_passwd pref inside the gpg_decrypt fn
 * Bug 21
 *
 * Revision 1.20  2003/04/07 01:16:37  brian
 * - mangled command string in gpg_decrypt to omit --passphrase fd0 if user has
 *   no_signing_passphrase = true
 * Bug 21
 *
 * Revision 1.19  2003/04/04 18:56:41  brian
 * separated out if statement to change command if the user has no_signing_passphrase set to true.
 * Bug 21
 *
 * Revision 1.18  2003/04/04 17:19:03  brian
 * fixed typo in elseif definition in gpg_decrypt function
 *
 * Revision 1.17  2003/04/04 05:22:52  brian
 * - added check for no_signing_passphrase to gpg_decrypt fn.
 * - Should execute without errors if user has no_signing_passphrase turned on.
 * Bug 21
 *
 * Revision 1.16  2003/04/02 18:26:40  brian
 * added check for no_signing_passwd  in fn gpg_decrypt
 *  - the user will not see an error if they have selected
 *    no_signing_passwd and submit without a passphrase
 *
 * Revision 1.15  2003/04/01 15:53:01  brian
 * improved error handling in gpg_encrypt and gpg_decrypt functions by
 * replacing strpos checks with substr_count checks
 * Bug 8
 *
 * Revision 1.14  2003/04/01 05:24:55  brian
 * improved error handling strings
 *
 * Revision 1.13  2003/03/31 15:43:25  brian
 * fixed typo on line 389, missign second right paren before 'break'
 *
 * Revision 1.12  2003/03/30 22:13:59  brian
 * Convert gpg_decrypt fn to use plaintext language instead ofcyphertext lang.
 * Bug 8
 *
 * Revision 1.11  2003/03/29 19:57:58  brian
 * clean up comments in gpg_parse_address fn
 * Bug 3
 *
 * Revision 1.10  2003/03/28 21:09:16  brian
 * address parsing in gpg_parse_address function now works for sm 1.2.x and sm >=1.3.x
 * tested on sm 1.4rc2a
 * fixes bug# 3
 * Bug 3
 *
 * Revision 1.9  2003/03/27 21:03:40  brian
 * fixed syntax error in gpg_decrypt, extra curly brace
 *
 * Revision 1.8  2003/03/27 19:21:11  brian
 * manual fix of $Log: entries
 *----------------------------
 * Revision 1.7 2003/03/27 19:11:38  brian
 * basics of decryption
 * gpg_decrypt function should work when called with body and passphrase
 *
 * Revision 1.6 2003/03/25 16:14:22  brian
 * Add to SM 1.4 parsing code
 *
 * Revision 1.5 2003/03/17 18:58:30  brian
 * - progress towards SM v >=1.3.1 compatibility
 * - path selection for includes now works on both
 *   SM 1.2.x and SM >= 1.3.1
 * - path selection for version check in gpg_parse_address
 *   now correct
 * - fixed skipped_keys bug in fn gpg_encrypt
 *
 * Revision 1.4 2003/03/15 21:04:46  brian
 * added version checknig code to gpg_parse_address
 * we now run code based on SM version
 *
 * Revision 1.3 2003/03/15 20:50:30  brian
 * fixes gpg_parse_address function to only return an array, not process it into a recipient list
 *
 * Revision 1.2 2003/03/15 20:43:16  brian
 * added gpg_parse_address function
 *
 * Revision 1.1 2003/03/11 23:30:30  tyler
 * - Initial breakout of the *_functions.php file
 *
 * This file was broken out from gpg_functions.php by tyler.
 * Early details on these functions may be found in the log
 * entries in gpg_functions.php - Brian
 */

?>
