<?php
  
/**
  * SquirrelMail Empty Trash Plugin
  *                       
  * Copyright (c) 2004-2011 Paul Lesniewski <paul@squirrelmail.org>
  * Copyright (c) 2002-2003 Derek Battams <derek@battams.ca>
  *
  * Licensed under the GNU GPL. For full terms see the file COPYING.
  *
  * @package plugins
  * @subpackage empty_trash
  *
  */



/**
  * Initialize this plugin (load config values)
  *
  * @param boolean $debug When TRUE, do not suppress errors when including
  *                       configuration files (OPTIONAL; default FALSE)
  *
  * @return boolean FALSE if no configuration file could be loaded, TRUE otherwise
  *
  */
function empty_trash_init($debug=FALSE)
{

   if ($debug)
   {
      if (!include_once(SM_PATH . 'config/config_empty_trash.php'))
         if (!include_once(SM_PATH . 'plugins/empty_trash/config.php'))
            if (!include_once(SM_PATH . 'plugins/empty_trash/config_default.php'))
               return FALSE;
   }
   else
   {
      if (!@include_once(SM_PATH . 'config/config_empty_trash.php'))
         if (!@include_once(SM_PATH . 'plugins/empty_trash/config.php'))
            if (!@include_once(SM_PATH . 'plugins/empty_trash/config_default.php'))
               return FALSE;
   }


   return TRUE;

}



/**
  * Increment a counter each time a user signs into SquirrelMail
  *
  */ 
function empty_trash_counter_increment()
{
   global $data_dir, $username;

   $logins = getPref($data_dir, $username, 'logins');
   setPref($data_dir, $username, 'logins', ++$logins);
   setPref($data_dir, $username, 'doEmpty', 1);
}



/**
  * Redirects folder list page to the empty_trash.php
  * script in order to empty out a user's trash folder.
  *
  * Uses JavaScript to immediately redirect the browser.
  *
  */
function empty_trash_redirect()
{
   echo '<script type="text/javascript" language="JavaScript">'
      . ' window.location="empty_trash.php"; '
      . '</script>';
}



/**
  * Decide if it's time to delete the user's
  * trash folder and if it is then do it!
  *
  */
function check_empty_trash()
{

   global $empty_upon_login, $data_dir, $username;


   // don't waste cycles if not just logging in
   //
   $doEmpty = getPref($data_dir, $username, 'doEmpty');
   if (empty($doEmpty))
      return;


   empty_trash_init();


   // don't waste cycles if configured to empty at logout instead
   //
   if (!$empty_upon_login)
      return;


   // if not using a trash folder, can't do anything
   //
   $trash_folder = getPref($data_dir, $username, 'trash_folder');
   global $trash_folder;
   if ($trash_folder == '' || $trash_folder == 'none')
      return;


   if (check_empty_trash_limits())
   {
      setPref($data_dir, $username, 'doEmpty', 0);
      setPref($data_dir, $username, 'lastDel', time());
      empty_trash();
      //empty_trash_redirect();
   }

}



/**
  * Empty trash folder upon logout if needed
  *
  */
function empty_trash_on_logout()
{

   // Check if there is a current login session -
   // don't waste our time if not
   //
   if (!sqsession_is_registered('user_is_logged_in'))
      return;


   global $username, $data_dir, $empty_upon_login;


   // if not using a trash folder, can't do anything
   //
   $trash_folder = getPref($data_dir, $username, 'trash_folder');
   global $trash_folder;
   if ($trash_folder == '' || $trash_folder == 'none')
      return;


   empty_trash_init();


   // Only empty if configured to do so
   //
   if (!$empty_upon_login && check_empty_trash_limits())
   {
      setPref($data_dir, $username, 'doEmpty', 0);
      setPref($data_dir, $username, 'lastDel', time());
      empty_trash();
   }

}



/**
  * Determine if login or number of days
  * limit has been reached
  *
  * @return boolean TRUE if one of the limits has
  *                 been reached and the trash should
  *                 be emptied; FALSE otherwise
  *
  */
function check_empty_trash_limits()
{

   global $loginLimit, $dayLimit, $username, $data_dir;

   empty_trash_init();


   // get login and last deletion information
   //
   $lastDel = getPref($data_dir, $username, 'lastDel');
   $logins = getPref($data_dir, $username, 'logins');
   $now = time();


   // if had never been emptied by this plugin before,
   // we'll start our timer right now (as if it was
   // emptied just now)
   //
   if (empty($lastDel))
   {
      $lastDel = $now;
      setPref($data_dir, $username, 'lastDel', $lastDel);
   }


   // determine if/when the trash was last emptied
   // 
   $overLoginLimit = ($loginLimit > 0 && ($logins % $loginLimit) === 0);
   $overDayLimit = ($dayLimit > 0 && (($now - $lastDel) / 86400) > $dayLimit);


   if ($overLoginLimit || $overDayLimit) return TRUE;

   return FALSE;

}



/**
  * Empty out the trash folder.
  *
  * This code was copied from SquirrelMail STABLE
  * src/empty_trash.php on April 16, 2011
  *
  */
function empty_trash()
{

   global $imapServerAddress, $imapPort, $trash_folder;


require_once(SM_PATH . 'functions/display_messages.php');
require_once(SM_PATH . 'functions/imap.php');
require_once(SM_PATH . 'functions/tree.php');

sqgetGlobalVar('username', $username, SQ_SESSION);
sqgetGlobalVar('key', $key, SQ_COOKIE);
sqgetGlobalVar('delimiter', $delimiter, SQ_SESSION);
sqgetGlobalVar('onetimepad', $onetimepad, SQ_SESSION);



$imap_stream = sqimap_login($username, $key, $imapServerAddress, $imapPort, 0);

sqimap_mailbox_list($imap_stream);

$mailbox = $trash_folder;
$boxes = sqimap_mailbox_list($imap_stream);

/*
 * According to RFC2060, a DELETE command should NOT remove inferiors (sub folders)
 *    so lets go through the list of subfolders and remove them before removing the
 *    parent.
 */

/** First create the top node in the tree **/
$numboxes = count($boxes);
$foldersTree = array();
for ($i = 0; $i < $numboxes; $i++) {
    if (($boxes[$i]['unformatted'] == $mailbox) && (strlen($boxes[$i]['unformatted']) == strlen($mailbox))) {
        $foldersTree[0]['value'] = $mailbox;
        $foldersTree[0]['doIHaveChildren'] = false;
        continue;
    }
}
/*
 * Now create the nodes for subfolders of the parent folder
 * You can tell that it is a subfolder by tacking the mailbox delimiter
 *    on the end of the $mailbox string, and compare to that.
 */
$j = 0;
for ($i = 0; $i < $numboxes; $i++) {
    if (substr($boxes[$i]['unformatted'], 0, strlen($mailbox . $delimiter)) == ($mailbox . $delimiter)) {
        addChildNodeToTree($boxes[$i]['unformatted'], $boxes[$i]['unformatted-dm'], $foldersTree);
    }
}

// now lets go through the tree and delete the folders
walkTreeInPreOrderEmptyTrash(0, $imap_stream, $foldersTree);
sqimap_logout($imap_stream);

}



