<?php

/**
  * SquirrelMail Debugger
  * Copyright (C) 2004 Paul Lesneiwski <pdontthink@angrynerds.com>
  * This program is licensed under BSD (because some portions of
  * the source are already licensed under BSD; typical SquirrelMail
  * license is GPL). See COPYING for details
  *
  */



/**
  * Simple Error Handling Routine
  *
  * @param int $errorNumber
  * @param int $errorMessage
  * @param int $errorFile
  * @param int $errorLine
  * @param int $errorContext
  *
  */
function simpleErrorHandler($errorNumber, $errorMessage, $errorFile, $errorLine, $errorContext)
{

   $errType = array (
      1    => "PHP Error",
      2    => "PHP Warning",
      4    => "Parsing Error",
      8    => "PHP Notice",
      16   => "Core Error",
      32   => "Core Warning",
      64   => "Compile Error",
      128  => "Compile Warning",
      256  => "PHP User Error",
      512  => "PHP User Warning",
      1024 => "PHP User Notice"
   );


   echo '<html><body><pre>'
      . "<hr />\n\n<b>" 
      . (strpos($errType[$errorNumber], 'PHP') === 0 ? substr($errType[$errorNumber], 4) : $errType[$errorNumber]) 
      . '</b>: ' . $errorMessage . ' in <b>' . $errorFile 
      . '</b> on line <b>' . $errorLine
      . '</b><br />' . "\n<hr />\n</pre><b style='text-decoration:underline'>Stack Trace:</b><br><br>";


   echo getStackTrace(120, TRUE, '', 1);

   echo "<pre><hr />\n\n";

   include_once(SM_PATH . 'plugins/debugger/bug_report.php');
   global $body;
   echo $body;

   echo '</pre>';

   global $sm_debugger_fatal;
   if ($sm_debugger_fatal) 
   {
      echo '</body></html>';
      exit;
   }

}



/**
  * Get Stack Trace
  *
  * @param int $maxArgLength The maximum number of characters that
  *                          will be shown of any one function argument.
  *                          Set to zero to disable all limits.
  * @param boolean $indent If TRUE, indents successive stack traces.
  * @param string $basedir The name of the directory immediately
  *                        above the one where file paths should
  *                        start (everything up to this "basedir"
  *                        will be truncated and not shown).  Set
  *                        to an empty string to show full path.
  * @param int $extraStackPops Number of additional traces to remove
  *                            from stack trace, if any.
  * @param int $traceArgOffset Can be overridden to make arguments 
  *                            show up in correct function calls; this 
  *                            depends on quirkiness of PHP installation.
  *
  * @return string The HTML-ready prettified stack trace.
  *
  */
function getStackTrace($maxArgLength=0, $indent=TRUE, $basedir='', $extraStackPops=0, $traceArgOffset=0)
{
       
   if (!function_exists('debug_backtrace')) 
      return 'Stack trace unavailable.';
      
   $trace = debug_backtrace();
   array_shift($trace);
   for ($i = 0; $i < $extraStackPops; $i++) array_shift($trace);
   $out = '';
       
   $x = 0;
   foreach ($trace as $callNo => $callDetails)
   {

      // get file
      //
      if (!isset($callDetails['file']))
         $file = 'unknown';
      else if (!empty($basedir)) 
         $file = substr($callDetails['file'], strpos($callDetails['file'], $basedir) + strlen($basedir) + 1);
      else
         $file = $callDetails['file'];
        
      // get line number
      // 
      if (!isset($callDetails['line']))
         $line = 'unknown';
      else
         $line = $callDetails['line'];

      // get function and its arguments
      //
      $args = array();
      if (isset($trace[$callNo + $traceArgOffset]['args'])) foreach ($trace[$callNo + $traceArgOffset]['args'] as $arg)
      {
         if (is_null($arg)) $args[] = '';  // 'null';
         else if (is_array($arg)) $args[] = 'Array['.sizeof($arg).']';
         else if (is_object($arg)) $args[] = 'Object:'.get_class($arg);
         else if (is_bool($arg)) $args[] = $arg ? 'TRUE' : 'FALSE';
         else
         {
            $arg = (string) @$arg;
            $str = htmlspecialchars(($maxArgLength > 0 ? substr($arg, 0, $maxArgLength) : $arg));
            if ($maxArgLength > 0 && strlen($arg) > $maxArgLength) $str .= '...';
            $args[] = "\"" . $str . "\"";
         }
      }
      $function = '<b>' . $callDetails['function'] . '</b>(' . implode(', ', $args) . ')';



      // begin output
      //
      if ($indent) for ($i = 0; $i < ($x * 4); $i++) $out .= '&nbsp;';
            
      $out .= $function . '<br />';

      if ($indent) for ($i = 0; $i < ($x * 4); $i++) $out .= '&nbsp;';
            
      $out .= $file . '<br />';

      if ($indent) for ($i = 0; $i < ($x * 4); $i++) $out .= '&nbsp;';
            
      $out .= 'line <b>' . $line . '</b><br />';
             
      $out .= "<br />";
            
      $x++;
   }

   return $out;
   
}


?>
