<?php

/**
  * SquirrelMail Cleanup Plugin
  * Copyright (c) 2003 Jimmy Conner <jimmy@advcs.org>,
  * Copyright (c) 2008 Paul Lesniewski <paul@squirrelmail.org>,
  * Licensed under the GNU GPL. For full terms see the file COPYING.
  *
  * @package plugins
  * @subpackage cleanup
  *
  */



/**
  * Initialize this plugin (load config values)
  *
  * @return boolean FALSE if no configuration file could be loaded, TRUE otherwise
  *
  */
function cleanup_init()
{
   
   if (!@include_once(SM_PATH . 'config/config_cleanup.php'))
      if (!@include_once(SM_PATH . 'plugins/cleanup/config.php'))
         if (!@include_once(SM_PATH . 'plugins/cleanup/config_default.php'))
            return FALSE;

   return TRUE;

}



/**
  * Do cleanup
  *
  */
function cleanup_do()
{

   global $cleanup_frequency, $cleanup_attachments;
   cleanup_init();

   // grab our preference setting for last time of cleanup
   //
   global $data_dir, $prefs_cache, $prefs_are_cached;
   $prefs_cache = FALSE;
   $prefs_are_cached = FALSE;
   sqsession_unregister('prefs_are_cached');
   sqsession_unregister('prefs_cache');

   $last_time = getPref($data_dir, 'cleanup', 'last_cleanup_time', 0);


   // proceed only if it's time
   //
   if (time() > ($last_time + $cleanup_frequency * 86400)) // 86400 seconds in one day
   {
      if ($cleanup_attachments)
         cleanup_attachments();
   }


   // update last cleanup time
   //
   setPref($data_dir, 'cleanup', 'last_cleanup_time', time());


   $prefs_cache = FALSE;
   $prefs_are_cached = FALSE;
   sqsession_unregister('prefs_are_cached');
   sqsession_unregister('prefs_cache');

}



/**
  * Cleanup attachments directory
  *
  * Removes any files older than desired
  *
  * @param $directory The directory to start the cleanup from;
  *                   used for recursive calls when the attachments
  *                   directory is hashed (OPTIONAL; default empty,
  *                   in which case the base SquirrelMail attachments
  *                   directory is used)
  *
  */
function cleanup_attachments($directory='')
{

   global $attachment_dir, $save_files_patterns, $save_directories,
          $delete_attachment_files_older_than;
   cleanup_init();


   // always exclude "." and ".." directories
   //
   $save_directories[] = '.';
   $save_directories[] = '..';


   if (empty($directory))
      $directory = $attachment_dir;


   // make sure directory path has trailing slash
   //
   if ($directory{strlen($directory) - 1} != '/' && strpos($directory, '/') !== false)
      $directory .= '/';
   if ($directory{strlen($directory) - 1} != '\\' && strpos($directory, '\\') !== false)
      $directory .= '\\';

   // open the directory
   //
   $d = @dir($directory);
   if (!$d) return;


   // iterate through the files therein; recursively
   // scanning subdirectories and removing files older
   // than $delete_attachment_files_older_than days
   //
   while (false !== ($entry = $d->read()))
   {

      $delete = FALSE;
      if (@is_dir($directory . $entry))
      {
         foreach ($save_directories as $skip)
         {
            if (strtolower($entry) == strtolower($skip))
               continue 2;
         }
         cleanup_attachments($directory . $entry);
      }
      else if (@is_file($directory . $entry))
      {
         $delete = TRUE;
         foreach ($save_files_patterns as $skip)
         {
            if (strpos(strtolower($entry), $skip) !== false)
               $delete = FALSE;
         }
      }
      if ($delete
       && time() > @filemtime($directory . $entry) + $delete_attachment_files_older_than * 86400)
         @unlink($directory . $entry);

   }
   $d->close();

}



/**
  * Validate that this plugin is configured correctly
  *
  * @return boolean Whether or not there was a
  *                 configuration error for this plugin.
  *
  */
function cleanup_check_configuration_do()
{

   // make sure plugin is correctly configured
   //
   if (!cleanup_init())
   {
      do_err('Cleanup plugin is not configured correctly', FALSE);
      return TRUE;
   }


   // make sure the attachments directory is readable
   //
   global $attachment_dir;
   $directory = $attachment_dir;
   if ($directory{strlen($directory) - 1} != '/' && strpos($directory, '/') !== false)
      $directory .= '/';
   if ($directory{strlen($directory) - 1} != '\\' && strpos($directory, '\\') !== false)
      $directory .= '\\';
   $d = @dir($directory);
   if (!$d)
   {
      do_err('Cleanup plugin requires read access to the attachments directory (' . $attachment_dir . ')', FALSE);
      return TRUE;
   }


   return FALSE;

}



