<?php
/**
 * Change LDAP Password plugin functions
 * @copyright (c) 2000-2003 Simon Annetts <simon@ateb.co.uk>
 * @copyright (c) 2006-2007 The SquirrelMail Project Team
 * @copyright (c) 2007 The NaSMail Project
 * @license http://opensource.org/licenses/gpl-license.php GNU Public License
 * @version $Id: functions.php,v 1.12 2007/07/21 06:56:42 tokul Exp $
 * @package plugins
 * @subpackage change_ldappass
 */

/** load configuration */
include_once(SM_PATH . 'plugins/change_ldappass/load_config.php');

/**
 * Checks user input
 * @return array
 */
function change_ldappass_check() {
    global $lcp_crack_dict;

    $cp_oldpass = '';
    $cp_newpass = '';
    $cp_verify = '';
    sqgetGlobalVar('cp_oldpass',$cp_oldpass,SQ_POST);
    sqgetGlobalVar('cp_newpass',$cp_newpass,SQ_POST);
    sqgetGlobalVar('cp_verify',$cp_verify,SQ_POST);

    $Messages = array();
    if (! extension_loaded('ldap')) {
        array_push($Messages,_("PHP LDAP extension is not available."));
    }
    if ($cp_oldpass == '') {
        array_push($Messages, _("You must type in your old password."));
    }
    if ($cp_newpass == '') {
        array_push($Messages, _("You must type in a new password."));
    }

    if ($cp_verify == '') {
        array_push($Messages,_("You must also type in your new password in the verify box."));
    }

    if ($cp_newpass != '' && $cp_verify != $cp_newpass) {
        array_push($Messages, _("Your new password doesn't match the verify password."));
    }

    if (!preg_match("/^[A-Za-z0-9_%=:@#~,\\^\\*\\(\\)\\-\\+\\[\\]\\{\\}\\.\\?]+$/",$cp_newpass)) {
        // i18n: comma separated list of acceptable characters is listed in next line
        array_push($Messages,
                   _("Passwords can only contain the following characters:"),
                   "A-Z, a-z, 0-9, %^*()-_+=[]{}:@#~,.?");
    }

    // PHP crack support from NaSMail change_password plugin
    if (! empty($lcp_crack_dict)) {
        if (! function_exists('crack_opendict')) {
            $Messages[] = _("PHP Crack extension is not available.");
        } else {
            // if crack_opendict command fails, it outputs E_WARNING,
            // Suppress it and handle error internally. assume that it is dictionary open failure
            if ($dict = @crack_opendict($lcp_crack_dict)) {
                if (! crack_check($dict,$cp_newpass)) {
                    $Messages[] = lcp_translate_crack_msg(crack_getlastmessage());
                    $Messages[] = _("Please choose stronger password.");
                }
                crack_closedict($dict);
            } else {
                $Messages[] = sprintf(_("Could not open crack dictionary: %s"),$lcp_crack_dict);
            }
        }
    }

    if (count($Messages)) {
        return $Messages;
    }
    return change_ldappass_go($cp_oldpass, $cp_newpass);
}

/**
 * Changes password
 * @param string $cp_oldpass
 * @param string $cp_newpass
 */
function change_ldappass_go($cp_oldpass, $cp_newpass) {
    global $username, $ldapsmb_object, $force_smb_sync, $ldap_server,
        $ldap_protocol_version, $ldap_base_dn, $ldap_password_field,
        $ldap_user_field, $query_dn, $query_pw, $ldap_filter,
        $no_bind_as_user, $ldap_bind_as_manager, $ldap_manager_dn,
        $ldap_manager_pw, $change_ldapsmb, $mkntpwd,
        $ldapsmb_ntpassword, $ldapsmb_lmpassword, $change_smb,
        $smb_host, $smb_passwd, $debug;

    $Messages = array();
    if ($debug) {
        array_push($Messages, 'Connecting to LDAP Server');
    }

    $ds=ldap_connect($ldap_server);
    if (! $ds) {
        array_push($Messages, _("Can't connect to Directory Server, please try later!"));
        return $Messages;
    }

    if (function_exists( 'ldap_set_option')) {
        if (! @ldap_set_option($ds,LDAP_OPT_PROTOCOL_VERSION,$ldap_protocol_version)) {
            array_push($Messages, _("Unable to set LDAP bind protocol version."));
            if ($debug) {
                array_push($Messages, htmlspecialchars("LDAP bind protocol version: $ldap_protocol_version"));
            }
            return $Messages;
        } else {
            if ($debug) {
                array_push($Messages, htmlspecialchars("LDAP protocol version was set to $ldap_protocol_version"));
            }
        }
    }


    // first bind and try to determine the correct dn for the user with uid=$username
    if (! @ldap_bind($ds, $query_dn, $query_pw)) {
        array_push($Messages, _("LDAP bind failed."),
                   sprintf(_("Error: %s"),htmlspecialchars(ldap_err2str(ldap_errno($ds)))));
        if ($debug) {
            array_push($Messages,
                       htmlspecialchars("LDAP server: $ldap_server"),
                       htmlspecialchars('BIND-DN: ' . (! empty($query_dn) ? $query_dn : 'anonymous')));
        }
        return $Messages;
    } else {
        if ($debug) {
            array_push($Messages,
                       'LDAP bind successful.',
                       htmlspecialchars("LDAP server: $ldap_server"),
                       htmlspecialchars('BIND-DN: ' . (! empty($query_dn) ? $query_dn : 'anonymous')));
        }
    }

    if (!empty($ldap_filter)) {
        if (!preg_match('/^\\(.+\\)$/',$ldap_filter)) {
            $ldap_filter = '(' . $ldap_filter . ')';
        }
        $filter = "(&$ldap_filter($ldap_user_field=$username))";
    } else {
        $filter = "($ldap_user_field=$username)";
    }

    // Hide ldap_search notices
    $sr=@ldap_search($ds,$ldap_base_dn,$filter,array('dn')); //search for uid
    if (! $sr) {
        array_push($Messages, _("LDAP search failed."),
                   sprintf(_("Error: %s"),htmlspecialchars(ldap_err2str(ldap_errno($ds)))));
        if ($debug) {
            array_push($Messages,
                       htmlspecialchars("BASE DN: $ldap_base_dn"),
                       htmlspecialchars("Query: ($ldap_user_field=$username)"));
        }
        return $Messages;
    }

    if (ldap_count_entries($ds,$sr)>1) {
        array_push($Messages, _("Duplicate login entries detected, cannot change password!"));
        if ($debug) {
            array_push($Messages,ldap_debug_print_array(ldap_get_entries($ds,$sr)));
        }
        return $Messages;
    }

    if (ldap_count_entries($ds,$sr)==0) {
        array_push($Messages, _("Your login account was not found in the LDAP database, cannot change password!"));
        return $Messages;
    }

    $info=ldap_get_entries($ds,$sr);
    if ($debug) {
        array_push($Messages,ldap_debug_print_array($info));
    }
    $dn=$info[0]["dn"]; //finally get the full users dn

    // now rebind to the database as user to verify password.
    if (! $no_bind_as_user) {
        if (! @ldap_bind($ds,$dn,$cp_oldpass)) { //if we can't bind as the user then the old passwd must be wrong
            array_push($Messages, _("Your old password is not correct."));
            if ($debug) {
                array_push($Messages,
                           'LDAP bind failed.',
                           htmlspecialchars("BIND-DN: $dn"));
            }
            return $Messages;
        } else {
            if ($debug) {
                array_push($Messages,
                           'LDAP bind successful.',
                           htmlspecialchars("BIND-DN: $dn"));
            }
        }
    } elseif ($ldap_bind_as_manager) {
        // Now, if needed, we rebind as the manager so we can read passwords and make changes.
        if (! @ldap_bind($ds,$ldap_manager_dn,$ldap_manager_pw)) {
            array_push($Messages, _("LDAP bind failed."));
            if ($debug){
                array_push($Messages, htmlspecialchars("BIND-DN: $ldap_manager_dn"));
            }
            return $Messages;
        } else {
            if ($debug) {
                array_push($Messages,
                           'LDAP bind successful.',
                           htmlspecialchars("BIND-DN: $ldap_manager_dn"));
            }
        }
    }
    //check the db again, this time so we get the password field returned
    // (use ldap_read() in order to lookup only located dn entry)
    $sr=@ldap_read($ds,$dn,"($ldap_user_field=$username)",array('dn',$ldap_password_field,$ldap_user_field,'objectclass'));
    $info = ldap_get_entries($ds, $sr);
    if ($debug) {
        array_push($Messages,ldap_debug_print_array($info));
    }

    if (isset($info[0][$ldap_password_field][0])) {
        $storedpass = $info[0][$ldap_password_field][0];
    } else {
        array_push($Messages, _("We could not retrieve your old password from the LDAP server."));
        if ($debug) {
            array_push($Messages,
                       htmlspecialchars("Configured password field: $ldap_password_field"));
        }
        return $Messages;
    }

    //this next code tries to identify the correct password type
    //*Note* I've not tested all types here as I cannot reproduce some setups
    //Please let me know if the code does not work for you, or if you have code that will work
    //for a particular type.

    //password types:
    //{crypt} salted passwords of type DES, MD5 and Blowfish
    //{MD5} unsalted password in MD5 format
    //{SHA} unsalted password in SHA format
    //{SMD5} salted md5
    //{SSHA} salted sha

    //lets try to determine the encrytion method of the stored password
    $p=split("}",$storedpass); //split the password
    $ctype=strtolower($p[0]);  //into the {crypttype}
    $lpass=$p[1];              //and the password
    //if the stored password is {crypt} then its salted, but which sub-type?
    switch ($ctype) {
    case "{crypt":
        $pl=strlen($lpass); // We'll look at the length and salt of what's already stored to determine the crypt sub-type
        $stype="DES";       //sensible default if not detected
        if ($pl>=34 and substr($lpass,0,3)=="$1$") $stype="MD5";
        if ($pl>=34 and substr($lpass,0,3)=="$2$") $stype="BLOWFISH";
        if ($debug) array_push($Messages, _("Password type is") . " {crypt}, sub-type $stype");
        $cpass=ldap_crypt_passwd($cp_oldpass,$lpass,$stype); //crypt up our old password so we can check it again
        break;
    case "{md5":
        if ($debug) array_push($Messages, _("Password type is") . " {MD5}" );
        $cpass=ldap_md5_passwd($cp_oldpass);
        break;
    case "{sha":
        if ($debug) array_push($Messages, _("Password type is") . " {SHA}");
        if (!function_exists('sha1') && ! extension_loaded('mhash')) {
            array_push($Messages,
                       _("Unsupported password schema. Insufficient PHP version or PHP mhash extension is not available."));
            return $Messages;
        }
        $cpass=ldap_sha_passwd($cp_oldpass);
        break;
    case "{smd5":
        if ($debug) array_push($Messages, _("Password type is") . " {SMD5}");
        $hash = base64_decode($lpass) ;
        $salt = substr($hash, 16);
        $cpass = base64_encode(pack("H*", md5($cp_oldpass . $salt)).$salt);
        break;
    case "{ssha":
        if ($debug) array_push($Messages, _("Password type is") . " {SSHA}");
        if (!function_exists('sha1') && ! extension_loaded('mhash')) {
            array_push($Messages,
                        _("Unsupported password schema. Insufficient PHP version or PHP mhash extension is not available."));
            return $Messages;
        }
        $hash = base64_decode($lpass) ;
        $salt = substr($hash, 20);
        $cpass=ldap_ssha_passwd($cp_oldpass,$salt);
        break;
    default:                        // Use plain text password
        $cpass=$cp_oldpass;
        $lpass=$storedpass;     // Override $lpass as it is truncated from the original
        break;
    }
    //now check again the stored password against the encrypted version of the supplied old password
    if ($lpass != $cpass) {
        array_push($Messages, _("Your old password is not correct."));
        if ($debug) {
            array_push($Messages,
                       htmlspecialchars("Stored Password: $lpass"),
                       htmlspecialchars("Old Password: $cpass"));
        }
        return $Messages;
    }
    //Make sure the new passwd generation uses the encryption method of the previous password
    switch ($ctype) {
    case "{crypt":
        $newpass="{crypt}".ldap_crypt_passwd($cp_newpass,ldap_generate_salt($stype),$stype);
        break;
    case "{md5":
        $newpass="{MD5}".ldap_md5_passwd($cp_newpass);
        break;
    case "{sha":
        $newpass="{SHA}".ldap_sha_passwd($cp_newpass);
        break;
    case "{smd5":
        $newpass="{SMD5}".ldap_smd5_passwd($cp_newpass);
        break;
    case "{ssha":
        $newpass="{SSHA}".ldap_ssha_passwd($cp_newpass);
        break;
        // more password types should go here ;-) and the functions to drive them below.
    default:                        // Use plain text password
        $newpass=$cp_newpass;
        break;
    }
    if ($debug) {
        array_push($Messages, htmlspecialchars("New Password:  $newpass"));
    }

    $newinfo=array();
    $newinfo[$ldap_password_field][0]=$newpass;

    if (isset($info[0]['objectclass'])) {
        $objects = $info[0]['objectclass'];
        array_walk($objects,'lcp_arraytolower');
    } else {
        // failsafe for missing objectclass data
        $objects = array();
    }
    if ($change_ldapsmb && in_array($ldapsmb_object,$objects) &&
        ($force_smb_sync || sqgetGlobalVar('sync_smb_pass',$sync_smb_pass,SQ_POST))) {
        $exe = "$mkntpwd " . escapeshellarg($cp_newpass) . " 2>&1" ;
        if ($debug) array_push($Messages,$exe);
        $retarray = array();
        $retval = 1;
        $ntString = exec($exe, $retarray, $retval);
        if ($debug) $Messages = array_merge($Messages,$retarray);
        if ( $retval == "0" && preg_match("/^[0-9A-F]+:[0-9A-F]+$/",$ntString )) {
            list($lmPassword, $ntPassword) = explode (":", $ntString);
            $newinfo[$ldapsmb_ntpassword] = $ntPassword;
            $newinfo[$ldapsmb_lmpassword] = $lmPassword;
        } else {  // could not generate ntlm passwords
            array_push($Messages,_("Could not generate NTLM password hashes!"));
            if (!empty($retarray)) {
                $retmsg = implode("\n",$retarray);
                array_push($Messages,sprintf(_("Error: %s"),htmlspecialchars($retmsg)));
            }
            return $Messages;
        }
    }

    if (@ldap_modify($ds,$dn,$newinfo)) {
        $smb=0;
        if ($change_smb &&
            ($force_smb_sync || sqgetGlobalVar('sync_smb_pass',$sync_smb_pass,SQ_POST))) {
            // First we print three lines in stdin
            $exe = 'echo -e ' . escapeshellarg($cp_oldpass . "\\n" . $cp_newpass . "\\n" . $cp_newpass);
            // then pipe to smbpasswd
            $exe.= " | $smb_passwd ";
            // add remote machine name
            if (!empty($smb_host)) {
                $exe.= '-r ' . escapeshellarg($smb_host);
            }
            // add username, get passwords from stdin (-s), redirect output to stdout
            $exe.= " -U " . escapeshellarg($username)
                 . " -s 2>&1";

            // Save used command for debugging
            if ($debug) {
                array_push($Messages,$exe);
            }

            // initial values
            $retarr = array();
            $retval = 1;
            // exec returns last line, but we don't need it. Same line is stored in $retarr
            exec($exe,$retarr,$retval);
            // Check $retval. 0 = success, 1 = failure
            if ($retval) {
                // push $retarr to $Messages
                foreach($retarr as $retline) {
                    array_push($Messages, htmlspecialchars($retline));
                }
            } else {
                $smb=1 ;
                if ($debug) {
                    foreach($retarr as $retline) {
                        array_push($Messages, htmlspecialchars($retline));
                    }
                }
            }
        } else {
            $smb=1;
        }
        if ($smb) {
            // load sqm_baseuri() function for SM 1.4.0-1.4.5
            include_once(SM_PATH . 'functions/display_messages.php');
            $base_uri = sqm_baseuri();
            // Write new cookies for the password
            $onetimepad = OneTimePadCreate(strlen($cp_newpass));
            $_SESSION['onetimepad']=$onetimepad; //do I need to do this now?
            $key = OneTimePadEncrypt($cp_newpass, $onetimepad);
            $_COOKIES['key']=$key;
            setcookie("key", $key, 0, $base_uri);
            // Give feedback if password change was successful.
            if (! $debug) {
                array_push($Messages, _("Password changed successfully"));
                return $Messages ;
            }
            return $Messages;
        } else { //smbpasswd change failed so we must re sync the ldap password back to its original
            $newinfo[$ldap_password_field][0]=$storedpass;
            if (@ldap_modify($ds,$dn,$newinfo)) {
                array_push($Messages, _("SMB Password change was not successful, so LDAP not changed!"));
            } else {
                array_push($Messages, _("Due to numerous password modification errors your LDAP and SMB passwords are out of sync. Please contact your administrator."));
            }
            return $Messages;
        }
    } else {
        array_push($Messages, _("LDAP Password change was not successful!"));
        if ($debug) array_push($Messages, _("LDAP ERROR => " . ldap_error($ds))) ;
        return $Messages;
    }
}

// Generate an unsalted SHA1 pw. This should work withNetscape messaging / directory server 4+
function ldap_sha_passwd($clear_pw) {
    if(function_exists('sha1')) {
        $hash = pack("H*",sha1($clear_pw));
    } else if (function_exists('mHash')) {
        $hash = mHash(MHASH_SHA1, $clear_pw);
    } else {
        echo "Error: You will need php >= 4.3.0 or php compiled with MHASH if you are going to use SHA or SSHA passwords.";
        exit();
    }
    return base64_encode($hash);
}

/**
 * Generate a salted SHA1 pw.
 * @param string $clean_pw
 * @param string $salt
 * @return string
 */
function ldap_ssha_passwd($clear_pw,$salt=null) {
    if (!isset($salt)){
        // set seed for the random number generator
        mt_srand((double)microtime()*1000000);
        $salt = substr(md5(mt_rand()), 4, 8);
    }
    if(function_exists('sha1')) {
        $hash = pack("H*",sha1($clear_pw . $salt));
    } else if (function_exists('mHash')) {
        $hash = mHash(MHASH_SHA1, $clear_pw . $salt);
    } else {
        echo "Error: You will need php >= 4.3.0 or php compiled with MHASH if you are going to use SHA or SSHA passwords.";
        exit();
    }
    return base64_encode($hash . $salt);
}

// Generate an unsalted MD5 pw. This works fine with OpenLDAP.
function ldap_md5_passwd($clear_pw) {
    return base64_encode(pack("H*", md5($clear_pw)));
}

function ldap_smd5_passwd($clear_pw) {
    $salt = myhash_keyge_s2k($clear_pw, 4);
    $new_password = base64_encode(pack("H*", md5($clear_pw . $salt)) . $salt);
    return $new_password ;
}

/**
 * Generates salted crypt passwords
 *
 * @param string $password
 * @param string $salt
 * @param string $stype Password type. MD5, BLOWFISH, DES. If other string is
 * used, since 2.2 function defaults to DES.
 * @return string
 */
function ldap_crypt_passwd($password,$salt,$stype) {
    if ($stype=="MD5")      return crypt($password,substr($salt,0,12)); //MD5 uses 12 chr salt
    if ($stype=="BLOWFISH") return crypt($password,substr($salt,0,16)); //BLOWFISH uses 16 chr salt
    // DES or something else
    return crypt($password,substr($salt,0,2));      //crypt uses 2 chr salt
}

/**
 * Generates salt
 * @param string $stype Salt type. MD5, BLOWFISH or DES. If other string is
 * used, since 2.2 function defaults to DES.
 * @return string Salt string
 */
function ldap_generate_salt($stype) {
    $salt=""; //generate a salt using characters [A-Z][a-z][0-9]./
    $chars="./0123456789abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ";
    for ($i=0;$i<16;$i++) {
        $salt.=$chars[mt_rand(0,strlen($chars))];
    }
    //return salts longer than we need but these will be trimmed in the ldap_crypt_passwd function
    if ($stype=="MD5")      return "$1$".$salt;
    if ($stype=="BLOWFISH") return "$2$".$salt;
    // DES or something else
    return $salt;
}

/**
 * This and md5 implimentation of the Salted S2K algorithm as
 * specified in the OpenPGP document (RFC 2440).
 * basically a non mhash dependant version of mhash_keygen_s2k
 * @param string $pass
 * @param integer $bytes
 * @return string
 */
function myhash_keyge_s2k($pass, $bytes ){
    $salt=substr(pack("h*", md5(mt_rand())), 0, 8);
    return substr(pack("H*", md5($salt . $pass)), 0, $bytes);
}

/**
 * Debug function used to print ldap search results
 * @param array $array
 */
function ldap_debug_print_array($array) {
    $out="<br>--------------------------------------------------------<br>\n";
    $out.=ldap_debug_print_array1($array,"");
    $out.= "<br>--------------------------------------------------------<br>\n";
    return $out;
}

/**
 * Prints array recursively
 * @param mixed $array
 * @param string $out
 * @return string
 */
function ldap_debug_print_array1($array,$out) {
    if(gettype($array)=="array") {
        $out.="<ul>";
        while (list($index, $subarray) = each($array) ) {
            $out.='<li>' . htmlspecialchars($index) . " <code>=&gt;</code>";
            $out=ldap_debug_print_array1($subarray,$out);
            $out.="</li>";
        }
        $out.="</ul>";
    } else {
        $out.=htmlspecialchars($array);
    }
    return $out;
}

/**
 * Translates PHP Crack extension messages
 *
 * cpw_translate_crack_msg() function from NaSMail change_password plugin.
 * @param string $err error from crack_getlastmessage()
 * @return string translated error message or original error
 * @since 2.2
 */
function lcp_translate_crack_msg($err) {
    /**
     * PHP crack 0.4 extension errors
     * from cracklib_fascist_look_ex()
     * - it's WAY too short
     * - it is too short
     * - it is too simplistic/systematic
     * - it looks like a National Insurance number.
     * - it is all whitespace
     * - it does not contain enough DIFFERENT characters
     * - it is based on a dictionary word
     * - it is based on a (reversed) dictionary word
     * from cracklib_fascist_gecos() (checks are not executed on Win32)
     * - you are not registered in the password file
     * - it is based on your username
     * - it's derivable from your password entry
     * - it is derivable from your password entry
     * - it is derived from your password entry
     * - it's derived from your password entry
     * - it is based upon your password entry
     */

    // can't use strcasecmp(). No locale insensitive string comparison functions in PHP.
    // internal implementation will slow things down.
    switch($err) {
    case 'it\'s WAY too short':
    case 'it is too short':
        $err = _("Password is too short.");
        break;
    case 'it is too simplistic/systematic':
        $err = _("New password is too simplistic/systematic.");
        break;
    case 'it looks like a National Insurance number.':
        // password looks like personal identification number used
        // in UK's social security system (aa000000a).
        $err = _("New password looks like a National Insurance number.");
        break;
    case 'it is all whitespace':
        $err = _("New password contains only whitespace.");
        break;
    case 'it does not contain enough DIFFERENT characters':
        $err = _("New password does not contain enough DIFFERENT characters.");
        break;
    case 'it is based on a dictionary word':
        $err = _("New password is based on a dictionary word.");
        break;
    case 'it is based on a (reversed) dictionary word':
        $err = _("New password is based on a (reversed) dictionary word.");
        break;
    case 'you are not registered in the password file':
        $err = _("You are not registered in the password file.");
        break;
    case 'it is based on your username':
        $err = _("New password is based on your username.");
        break;
    case 'it\'s derivable from your password entry':
    case 'it is derivable from your password entry':
    case 'it is derived from your password entry':
    case 'it\'s derived from your password entry':
    case 'it is based upon your password entry':
        // combines five error messages.
        // password is derivable/derived/based upon your password entry
        $err = _("New password is similar to your current password entry.");
        break;
    default:
        // leave $err untranslated
    }
    // 'No obscure checks in this session' - error comes from PHP crack extension
    // should never pop out because we always call crack_check() before crack_getlastmessage()
    return $err;
}

/**
 * Callback function to lowercase array keys and values
 *
 * See ldq_arraytolower in ldapquery plugin.
 * @param string Array value
 * @param string Array key
 * @since 2.2
 */
function lcp_arraytolower(&$value,&$key) {
    $value=strtolower($value);
    $key=strtolower($key);
}
