<?php
/**
 * Change LDAP Password plugin functions
 * @copyright &copy; 2000-2003 Simon Annetts <simon@ateb.co.uk>
 * @copyright &copy; 2006 The SquirrelMail Project Team
 * @license http://opensource.org/licenses/gpl-license.php GNU Public License
 * @version $Id: functions.php,v 1.6 2006/10/07 06:32:01 tokul Exp $
 * @package plugins
 * @subpackage change_ldappass
 */

/**
 * Checks user input
 * @return array
 */
function change_ldappass_check() {
    $cp_oldpass = '';
    $cp_newpass = '';
    $cp_verify = '';
    sqgetGlobalVar('cp_oldpass',$cp_oldpass,SQ_POST);
    sqgetGlobalVar('cp_newpass',$cp_newpass,SQ_POST);
    sqgetGlobalVar('cp_verify',$cp_verify,SQ_POST);
 
    $Messages = array();
    if (! extension_loaded('ldap')) {
        array_push($Messages,_("PHP LDAP extension is not available."));
    }
    if ($cp_oldpass == '') {
        array_push($Messages, _("You must type in your old password."));
    }
    if ($cp_newpass == '') {
        array_push($Messages, _("You must type in a new password."));
    }

    if ($cp_verify == '') {
        array_push($Messages,_("You must also type in your new password in the verify box."));
    }

    if ($cp_newpass != '' && $cp_verify != $cp_newpass) {
        array_push($Messages, _("Your new password doesn't match the verify password."));
    }

    if (!preg_match("/^[A-Za-z0-9\%\^\*\(\)\-\_\+\=\[\]\{\}\:\@\#\~\,\.\?]+$/",$cp_newpass)) {
        // i18n: comma separated list of acceptable characters is listed in next line
        array_push($Messages, 
                   _("Passwords can only contain the following characters:"),
                   "A-Z, a-z, 0-9, %^*()-_+=[]{}:@#~,.?");
    }

    if (count($Messages)) {
        return $Messages;
    }
    return change_ldappass_go($cp_oldpass, $cp_newpass);
}

/**
 * Changes password
 * @param string $cp_oldpass
 * @param string $cp_newpass
 */
function change_ldappass_go($cp_oldpass, $cp_newpass) {
    global $username;

    // Set default values
    $ldap_server = 'localhost';
    $ldap_protocol_version = 2;
    $ldap_base_dn = '';
    $ldap_password_field = 'userpassword';
    $ldap_user_field = 'uid';
    $query_dn = '';
    $query_pw = '';

    $no_bind_as_user = false;

    $ldap_bind_as_manager = false;
    $ldap_manager_dn = '';
    $ldap_manager_pw = '';

    $change_ldapsmb = false;
    $mkntpwd = '';
    $ldapsmb_ntpassword = '';
    $ldapsmb_lmpassword = '';

    $change_smb = false;
    $smb_host = '';
    $smb_passwd = '';

    $debug = false;
    // load config
    include(SM_PATH . 'plugins/change_ldappass/config.php');

    // field names use lower case
    $ldap_password_field = strtolower($ldap_password_field);
    $ldap_user_field = strtolower($ldap_user_field);


    $Messages = array();
    if ($debug) {
        array_push($Messages, 'Connecting to LDAP Server');
    }

    $ds=ldap_connect($ldap_server);
    if (! $ds) {
        array_push($Messages, _("Can't connect to Directory Server, please try later!"));
        return $Messages;
    }

    if (function_exists( 'ldap_set_option')) {
        if (! @ldap_set_option($ds,LDAP_OPT_PROTOCOL_VERSION,$ldap_protocol_version)) {
            array_push($Messages, _("Unable to set LDAP bind protocol version."));
            if ($debug) {
                array_push($Messages, htmlspecialchars("LDAP bind protocol version: $ldap_protocol_version"));
            }
            return $Messages;
        } else {
            if ($debug) {
                array_push($Messages, htmlspecialchars("LDAP protocol version was set to $ldap_protocol_version"));
            }
        }
    }


    // first bind and try to determine the correct dn for the user with uid=$username
    if (! @ldap_bind($ds, $query_dn, $query_pw)) {
        array_push($Messages, _("LDAP bind failed."),
                   sprintf(_("Error: %s"),htmlspecialchars(ldap_err2str(ldap_errno($ds)))));
        if ($debug) {
            array_push($Messages,
                       htmlspecialchars("LDAP server: $ldap_server"),
                       htmlspecialchars('BIND-DN: ' . (! empty($query_dn) ? $query_dn : 'anonymous')));
        }
        return $Messages;
    } else {
        if ($debug) {
            array_push($Messages,
                       'LDAP bind successful.',
                       htmlspecialchars("LDAP server: $ldap_server"),
                       htmlspecialchars('BIND-DN: ' . (! empty($query_dn) ? $query_dn : 'anonymous')));
        }
    }

    // Hide ldap_search notices
    $sr=@ldap_search($ds,$ldap_base_dn,"($ldap_user_field=$username)",array('dn')); //search for uid
    if (! $sr) {
        array_push($Messages, _("LDAP search failed."),
                   sprintf(_("Error: %s"),htmlspecialchars(ldap_err2str(ldap_errno($ds)))));
        if ($debug) {
            array_push($Messages,
                       htmlspecialchars("BASE DN: $ldap_base_dn"),
                       htmlspecialchars("Query: ($ldap_user_field=$username)"));
        }
        return $Messages;
    }

    if (ldap_count_entries($ds,$sr)>1) {
        array_push($Messages, _("Duplicate login entries detected, cannot change password!"));
        if ($debug) {
            array_push($Messages,ldap_debug_print_array(ldap_get_entries($ds,$sr)));
        }
        return $Messages;
    }

    if (ldap_count_entries($ds,$sr)==0) {
        array_push($Messages, _("Your login account was not found in the LDAP database, cannot change password!"));
        return $Messages;
    }

    $info=ldap_get_entries($ds,$sr);
    if ($debug) {
        array_push($Messages,ldap_debug_print_array($info));
    }
    $dn=$info[0]["dn"]; //finally get the full users dn
    
    // now rebind to the database as user to verify password.
    if (! $no_bind_as_user) {
        if (! @ldap_bind($ds,$dn,$cp_oldpass)) { //if we can't bind as the user then the old passwd must be wrong
            array_push($Messages, _("Your old password is not correct."));
            if ($debug) {
                array_push($Messages,
                           'LDAP bind failed.',
                           htmlspecialchars("BIND-DN: $dn"));
            }
            return $Messages;
        } else {
            if ($debug) {
                array_push($Messages, 
                           'LDAP bind successful.',
                           htmlspecialchars("BIND-DN: $dn"));
            }
        }
    } elseif ($ldap_bind_as_manager) {
        // Now, if needed, we rebind as the manager so we can read passwords and make changes.
        if (! @ldap_bind($ds,$ldap_manager_dn,$ldap_manager_pw)) {
            array_push($Messages, _("LDAP bind failed."));
            if ($debug){
                array_push($Messages, htmlspecialchars("BIND-DN: $ldap_manager_dn"));
            }
            return $Messages;
        } else {
            if ($debug) { 
                array_push($Messages, 
                           'LDAP bind successful.',
                           htmlspecialchars("BIND-DN: $ldap_manager_dn"));
            }
        }
    }
    //check the db again, this time so we get the password field returned 
    // (use ldap_read() in order to lookup only located dn entry)
    $sr=@ldap_read($ds,$dn,"($ldap_user_field=$username)",array('dn',$ldap_password_field,$ldap_user_field)); 
    $info = ldap_get_entries($ds, $sr);
    if ($debug) {
        array_push($Messages,ldap_debug_print_array($info));
    }

    if (isset($info[0][$ldap_password_field][0])) {
        $storedpass = $info[0][$ldap_password_field][0];
    } else {
        array_push($Messages, _("We could not retrieve your old password from the LDAP server."));
        if ($debug) {
            array_push($Messages,
                       htmlspecialchars("Configured password field: $ldap_password_field"));
        }
        return $Messages;
    }
    
    //this next code tries to identify the correct password type
    //*Note* I've not tested all types here as I cannot reproduce some setups
    //Please let me know if the code does not work for you, or if you have code that will work
    //for a particular type.
    
    //password types:
    //{crypt} salted passwords of type DES, MD5 and Blowfish
    //{MD5} unsalted password in MD5 format
    //{SHA} unsalted password in SHA format
    //{SMD5} salted md5
    //{SSHA} salted sha
    
    //lets try to determine the encrytion method of the stored password
    $p=split("}",$storedpass); //split the password
    $ctype=strtolower($p[0]);  //into the {crypttype}
    $lpass=$p[1];              //and the password
    //if the stored password is {crypt} then its salted, but which sub-type?
    switch ($ctype) {
    case "{crypt":
        $pl=strlen($lpass); // We'll look at the length and salt of what's already stored to determine the crypt sub-type
        $stype="DES";       //sensible default if not detected
        if ($pl>=34 and substr($lpass,0,3)=="$1$") $stype="MD5"; 
        if ($pl>=34 and substr($lpass,0,3)=="$2$") $stype="BLOWFISH";
        if ($debug) array_push($Messages, _("Password type is") . " {crypt}, sub-type $stype");
        $cpass=ldap_crypt_passwd($cp_oldpass,$lpass,$stype); //crypt up our old password so we can check it again
        break;
    case "{md5":
        if ($debug) array_push($Messages, _("Password type is") . " {MD5}" );
        $cpass=ldap_md5_passwd($cp_oldpass);
        break;
    case "{sha":
        if ($debug) array_push($Messages, _("Password type is") . " {SHA}");
        if (!function_exists('sha1') && ! extension_loaded('mhash')) {
            array_push($Messages,
                       _("Unsupported password schema. Insufficient PHP version or PHP mhash extension is not available."));
            return $Messages;
        }
        $cpass=ldap_sha_passwd($cp_oldpass);
        break;
    case "{smd5":
        if ($debug) array_push($Messages, _("Password type is") . " {SMD5}");
        $hash = base64_decode($lpass) ;
        $salt = substr($hash, 16);
        $cpass = base64_encode(pack("H*", md5($cp_oldpass . $salt)).$salt);
        break;
    case "{ssha":
        if ($debug) array_push($Messages, _("Password type is") . " {SSHA}");
        if (!function_exists('sha1') && ! extension_loaded('mhash')) {
            array_push($Messages,
                        _("Unsupported password schema. Insufficient PHP version or PHP mhash extension is not available."));
            return $Messages;
        }
        $hash = base64_decode($lpass) ;
        $salt = substr($hash, 20);
        $cpass=ldap_ssha_passwd($cp_oldpass,$salt);
        break;
    default:                        // Use plain text password
        $cpass=$cp_oldpass;
        $lpass=$storedpass;     // Override $lpass as it is truncated from the original
        break;
    }
    //now check again the stored password against the encrypted version of the supplied old password
    if ($lpass != $cpass) {
        array_push($Messages, _("Your old password is not correct."));
        if ($debug) {
            array_push($Messages,
                       htmlspecialchars("Stored Password: $lpass"),
                       htmlspecialchars("Old Password: $cpass"));
        }
        return $Messages;
    }
    //Make sure the new passwd generation uses the encryption method of the previous password
    switch ($ctype) {
    case "{crypt":
        $newpass="{crypt}".ldap_crypt_passwd($cp_newpass,ldap_generate_salt($stype),$stype);
        break;
    case "{md5":
        $newpass="{MD5}".ldap_md5_passwd($cp_newpass);
        break;
    case "{sha":
        $newpass="{SHA}".ldap_sha_passwd($cp_newpass);
        break;
    case "{smd5":
        $newpass="{SMD5}".ldap_smd5_passwd($cp_newpass);
        break;
    case "{ssha":
        $newpass="{SSHA}".ldap_ssha_passwd($cp_newpass);
        break;
        // more password types should go here ;-) and the functions to drive them below.
    default:                        // Use plain text password
        $newpass=$cp_newpass;
        break;
    }
    if ($debug) {
        array_push($Messages, htmlspecialchars("New Password:  $newpass"));
    }

    $newinfo=array();
    $newinfo[$ldap_password_field][0]=$newpass;

    if ($change_ldapsmb) {
        $exe = "$mkntpwd " . escapeshellarg($cp_newpass) . " 2>&1" ;
        if ($debug) array_push($Messages,$exe);
        $ntString = exec($exe, $retarray, $retval);
        if ($debug) $Messages = array_merge($Messages,$retarray);
        if ( $retval == "0" && preg_match("/^[0-9A-F]+:[0-9A-F]+$/",$ntString )) {
            list($lmPassword, $ntPassword) = explode (":", $ntString);
            $newinfo[$ldapsmb_ntpassword] = $ntPassword;
            $newinfo[$ldapsmb_lmpassword] = $lmPassword;
        } else {  // could not generate ntlm passwords
            array_push($Messages, _("SMB Password change was not successful, so LDAP not changed!"));
            return $Messages;
        }
    }

    if (@ldap_modify($ds,$dn,$newinfo)) {
        $smb=0;
        if ($change_smb) {
            if ($smb_host) {
                $exe="echo " . escapeshellarg($cp_oldpass)  
                    .  " |$smb_passwd -r " . escapeshellarg($smb_host)
                    . " -U " . escapeshellarg($username) 
                    . " -s " . escapeshellarg($cp_newpass) . " 2>&1" ;
            } else {
                $exe="echo " . escapeshellarg($cp_oldpass)  
                    .  " |$smb_passwd -U " . escapeshellarg($username) 
                    . " -s " . escapeshellarg($cp_newpass) . " 2>&1" ;
            }
            if ($debug) array_push($Messages,$exe);
            $r=exec($exe,$s);
            if ($r==_("Password changed for user") . " $username") $smb=1 ;
            if ($debug) array_push($Messages, $r);
        } else {
            $smb=1;
        }
        if ($smb) {
            // load sqm_baseuri() function for SM 1.4.0-1.4.5
            include_once(SM_PATH . 'functions/display_messages.php');
            $base_uri = sqm_baseuri();
            // Write new cookies for the password
            $onetimepad = OneTimePadCreate(strlen($cp_newpass));
            $_SESSION['onetimepad']=$onetimepad; //do I need to do this now?
            $key = OneTimePadEncrypt($cp_newpass, $onetimepad);
            $_COOKIES['key']=$key;
            setcookie("key", $key, 0, $base_uri);
            // Give feedback if password change was successful.
            if (! $debug) {
                array_push($Messages, _("Password changed successfully"));
                return $Messages ;
                exit(0);
            }
            return $Messages;
        } else { //smbpasswd change failed so we must re sync the ldap password back to its original
            $newinfo[$ldap_password_field][0]=$storedpass;
            if (@ldap_modify($ds,$dn,$newinfo)) {
                array_push($Messages, _("SMB Password change was not successful, so LDAP not changed!"));
            } else {
                array_push($Messages, _("Due to numerous password modification errors your LDAP and SMB passwords are out of sync. Please contact your administrator."));
            }
            return $Messages;
        }
    } else {
        array_push($Messages, _("LDAP Password change was not successful!"));
        if ($debug) array_push($Messages, _("LDAP ERROR => " . ldap_error($ds))) ;
        return $Messages;
    }
    @ldap_close($ds);
}

// Generate an unsalted SHA1 pw. This should work withNetscape messaging / directory server 4+ 
function ldap_sha_passwd($clear_pw) {
    if(function_exists('sha1')) {
        $hash = pack("H*",sha1($clear_pw));
    } else if (function_exists('mHash')) {
        $hash = mHash(MHASH_SHA1, $clear_pw);
    } else {
        echo "Error: You will need php >= 4.3.0 or php compiled with MHASH if you are going to use SHA or SSHA passwords.";
        exit();
    }
    return base64_encode($hash);
}

/**
 * Generate a salted SHA1 pw.
 * @param string $clean_pw
 * @param string $salt
 * @return string
 */
function ldap_ssha_passwd($clear_pw,$salt=null) {
    if (!isset($salt)){
        // set seed for the random number generator
        mt_srand((double)microtime()*1000000);
        $salt = substr(md5(mt_rand()), 4, 8);
    }
    if(function_exists('sha1')) {
        $hash = pack("H*",sha1($clear_pw . $salt));
    } else if (function_exists('mHash')) {
        $hash = mHash(MHASH_SHA1, $clear_pw . $salt);
    } else {
        echo "Error: You will need php >= 4.3.0 or php compiled with MHASH if you are going to use SHA or SSHA passwords.";
        exit();
    }
    return base64_encode($hash . $salt);
}

// Generate an unsalted MD5 pw. This works fine with OpenLDAP.
function ldap_md5_passwd($clear_pw) {
    return base64_encode(pack("H*", md5($clear_pw)));
}

function ldap_smd5_passwd($clear_pw) {
    $salt = myhash_keyge_s2k($clear_pw, 4);
    $new_password = base64_encode(pack("H*", md5($clear_pw . $salt)) . $salt);
    return $new_password ;
}

// Generate salted crypt passwords
function ldap_crypt_passwd($password,$salt,$stype) {
    if ($stype=="MD5")      return crypt($password,substr($salt,0,12)); //MD5 uses 12 chr salt
    if ($stype=="BLOWFISH") return crypt($password,substr($salt,0,16)); //BLOWFISH uses 16 chr salt
    if ($stype=="DES")  return crypt($password,substr($salt,0,2));      //crypt uses 2 chr salt
}

function ldap_generate_salt($stype) {
    $salt=""; //generate a salt using characters [A-Z][a-z][0-9]./
    $chars="./0123456789abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ";
    for ($i=0;$i<16;$i++) {
        @ $salt.=$chars[mt_rand(0,strlen($chars))];
    }
    //return salts longer than we need but these will be trimmed in the ldap_crypt_passwd function
    if ($stype=="MD5")      return "$1$".$salt;
    if ($stype=="BLOWFISH") return "$2$".$salt;
    if ($stype=="DES")      return $salt;
}

/**
 * This and md5 implimentation of the Salted S2K algorithm as 
 * specified in the OpenPGP document (RFC 2440).
 * basically a non mhash dependant version of mhash_keygen_s2k
 * @param string $pass
 * @param integer $bytes
 * @return string
 */
function myhash_keyge_s2k($pass, $bytes ){
    $salt=substr(pack("h*", md5(mt_rand())), 0, 8);
    return substr(pack("H*", md5($salt . $pass)), 0, $bytes);
}

/**
 * Debug function used to print ldap search results
 * @param array $array
 */
function ldap_debug_print_array($array) {
    $out="<br>--------------------------------------------------------<br>\n";
    $out.=ldap_debug_print_array1($array,"");
    $out.= "<br>--------------------------------------------------------<br>\n";
    return $out;
}

/**
 * Prints array recursively
 * @param mixed $array
 * @param string $out
 * @return string
 */
function ldap_debug_print_array1($array,$out) {
    if(gettype($array)=="array") {
        $out.="<ul>";
        while (list($index, $subarray) = each($array) ) {
            $out.='<li>' . htmlspecialchars($index) . " <code>=&gt;</code>";
            $out=ldap_debug_print_array1($subarray,$out);
            $out.="</li>";
        }
        $out.="</ul>";
    } else {
        $out.=htmlspecialchars($array);
    }
    return $out;
}
