<?php

/**
  * SquirrelMail Address Book Grouping and Pagination Plugin
  * Copyright (c) 2008 Paul Lesniewski <paul@squirrelmail.org>,
  * Licensed under the GNU GPL. For full terms see the file COPYING.
  *
  * @package plugins
  * @subpackage abook_group_pagination
  *
  */



/**
  * Validate that this plugin is configured correctly
  *
  * @return boolean Whether or not there was a
  *                 configuration error for this plugin.
  *
  */
function abook_group_pagination_check_configuration_do()
{

   // make sure base config is available
   //
   if (!abook_group_pagination_init())
   {
      do_err('Address Book Grouping and Pagination plugin is missing its main configuration file', FALSE);
      return TRUE;
   }

}



/**
  * Initialize this plugin (load config values)
  *
  * @return boolean FALSE if no configuration file could be loaded, TRUE otherwise
  *
  */
function abook_group_pagination_init()
{

   if (!@include_once (SM_PATH . 'config/config_abook_group_pagination.php'))
      if (!@include_once (SM_PATH . 'plugins/abook_group_pagination/config.php'))
         if (!@include_once (SM_PATH . 'plugins/abook_group_pagination/config.sample.php'))
            return FALSE;

   return TRUE;

/* ----------  This is how to do the same thing using the Compatibility plugin
   return load_config('abook_group_pagination',
                      array('../../config/config_abook_group_pagination.php',
                            'config.php',
                            'config.sample.php'),
                      TRUE, TRUE);
----------- */

}



/**
  * Display user configuration options on display preferences page
  *
  */
function abook_group_pagination_display_options_do()
{

   global $data_dir, $username, $agp_abook_pagination,
          $agp_abook_pagination_allow_override, $agp_grouping,
          $optpage_data;

   abook_group_pagination_init();

   sq_change_text_domain('abook_group_pagination');

   $my_optpage_values = array();


   // pagination is native to 1.5.2+
   //
   if (!check_sm_version(1, 5, 2) && $agp_abook_pagination_allow_override)
   {
      $agp_abook_pagination = getPref($data_dir, $username,
                                     'agp_abook_pagination',
                                     $agp_abook_pagination);
      $my_optpage_values[] = array(
         'name'          => 'agp_abook_pagination',
         'caption'       => _("Paginate Address Book"),
         'type'          => SMOPT_TYPE_STRLIST,
         'initial_value' => $agp_abook_pagination,
         'posvals'       => array(0 => _("No Pagination"), 5 => 5, 10 => 10, 15 => 15, 20 => 20, 25 => 25, 30 => 30, 40 => 40, 50 => 50, 60 => 60, 70 => 70, 80 => 80, 90 => 90, 100 => 100, 125 => 125, 150 => 150, 175 => 175, 200 => 200, 250 => 250, 300 => 300, 400 => 400, 500 => 500),
         'refresh'       => SMOPT_REFRESH_NONE,
      );
   }


   $agp_groups = array();
   if ($agp_grouping)
   {
      $agp_groups = getPref($data_dir, $username,
                            'agp_groups', $agp_groups);
      if (!empty($agp_groups) && is_string($agp_groups))
         $agp_groups = unserialize($agp_groups);
      $my_optpage_values[] = array(
         'name'          => 'agp_groups',
         'caption'       => _("Address Book Groups"),
         'type'          => SMOPT_TYPE_EDIT_LIST,
         //'layout_type' => SMOPT_EDIT_LIST_LAYOUT_SELECT,
         'size'          => SMOPT_SIZE_SMALL,
         'posvals'       => $agp_groups,
         'refresh'       => SMOPT_REFRESH_NONE,
         'save'          => 'agp_save_abook_groups',
      );
   }


   if (!empty($my_optpage_values))
   {  

      // 1.5.2+ has a address book section already - add our stuff to that section
      //
      if (check_sm_version(1, 5, 2))
      {
         $optpage_data['vals'][3] = array_merge($optpage_data['vals'][3], $my_optpage_values);
      }  


      // 1.4.x - add an address book section to the display options
      //
      else
      {  
         $optpage_data['grps']['abook_group_pagination'] = _("Address Book");
         $optpage_data['vals']['abook_group_pagination'] = $my_optpage_values;
      }  

   }  
         
   sq_change_text_domain('squirrelmail');

}



/**
  * Save user address groups - called from within
  * SquirrelMail options save routine.  This function
  * is intended to do ancillary processing and use
  * the internal SquirrelMail facilities to actually
  * save the groups data.
  *
  * Specifically, when a group is removed, the group's
  * contents need to be removed from the user preferences.
  *
  * @param object $option The option class object representing
  *                       the address book group data being
  *                       saved.
  *
  */
function agp_save_abook_groups($option)
{

   // grab a copy of the list before processing
   //
   $original = $option->possible_values;


   // now call into standard SquirrelMail save function
   // where the list will be correctly processed
   // and saved
   //
   save_option($option);


   // finally, we can compare the saved list with the
   // original one and determine which items were 
   // removed and thusly remove the contents of any
   // such groups
   //
   $deleted_groups = array_diff($original, $option->possible_values);
   if (!empty($deleted_groups))
   {

      // get user's address book group contents
      //
      global $username, $data_dir;
      $address_group_contents = getPref($data_dir, $username, 'agp_group_contents', '');
      if (empty($address_group_contents))
         $address_group_contents = array();
      else
         $address_group_contents = unserialize($address_group_contents);


      // remove contents of deleted groups
      //
      foreach ($deleted_groups as $group)
         unset($address_group_contents[$group]);


      // and re-save the prefs
      //
      setPref($data_dir, $username, 'agp_group_contents', serialize($address_group_contents));

   }

}



/**
  * Generates an address book paginator link (used in 1.4.x only).
  *
  * @param int    $backend_number The number of the backend for which
  *                               the link is being built
  * @param int    $page_number    The page number for the link
  * @param string $text           The link text
  * @param int    $page_size      The page size for the pagination
  *                               (OPTIONAL; default not used)
  *
  * @return string The desired paginator link
  *
  */
function get_abook_paginator_link($backend_number, $page_number, $text, $page_size=0)
{

   global $agp_abook_backends, $agp_group_selections, $agp_pagination,
          $agp_unrelated_get_args;


   $link = 'addressbook.php' . $agp_unrelated_get_args;


   // add pagination for OTHER backends and group filters for all backends
   //
   foreach ($agp_abook_backends as $backend)
   {

      // pagination for other backends...
      //
      if ($backend != $backend_number)
      {
         if (!empty($agp_pagination[$backend]['page_number']))
            $link .= 'backend_page_number_' . $backend . '='
                  . $agp_pagination[$backend]['page_number'] . '&amp;';
         if (!empty($agp_pagination[$backend]['page_size']))
            $link .= 'backend_page_size_' . $backend . '='
                  . $agp_pagination[$backend]['page_size'] . '&amp;';
      }

      // group filters for all backends...
      //
      if (!empty($agp_group_selections[$backend])
       && is_array($agp_group_selections[$backend]))
      {
         foreach ($agp_group_selections[$backend] as $group)
            $link .= 'abook_groups_' . $backend . '[]=' . htmlspecialchars($group) . '&amp;';
      }

   }



   // now add pagination for the current backend and build final output
   //
   $result = '<a href="' . $link . 'backend_page_number_'
           . $backend_number . '=' . $page_number
           . (!empty($page_size) ? '&amp;backend_page_size_'
              . $backend_number . '=' . $page_size
              : '')
           . '">' . $text . '</a>';


   return $result;

}



/**
  * Generate pagination string if needed (used in 1.4.x only)
  *
  * Looks in a row for pagination details; if insufficient data is found
  * (page size, page number, total addresses in its abook backend for this
  * user), an empty stirng is returned.
  *
  * @param array $abook_entry An array of address data for one address
  *                           which includes information about its
  *                           parent backend and especially any pagination
  *                           information needed herein
  *
  * @return string The pagination string
  *
  */
function get_abook_paginator_string($abook_entry)
{

   if (empty($abook_entry['page_number'])
    || empty($abook_entry['page_size'])
    || empty($abook_entry['total_addresses']))
      return '';


   global $max_pagination_links_to_show, $small_paginator_text;
   if ($max_pagination_links_to_show < 3) $max_pagination_links_to_show = 3;

   abook_group_pagination_init();


   // calculate some values we need below
   //
   $show_elipses_before = FALSE;
   $show_elipses_after = FALSE;
   $backend_number = $abook_entry['backend'];
   $nbsp = '&nbsp;';
   $page_size = $abook_entry['page_size'];
   $total_pages = ceil($abook_entry['total_addresses'] / $page_size);


   if ($total_pages < 2) return '';


   if ($total_pages <= $max_pagination_links_to_show)
   {
      $start_page = 1;
      $end_page = $total_pages;
   }
   else
   {
      $pages_to_show = ($max_pagination_links_to_show % 2 ? $max_pagination_links_to_show : $max_pagination_links_to_show - 1);
      $end_page = $abook_entry['page_number'] + floor($pages_to_show / 2);
      $start_page = $abook_entry['page_number'] - floor($pages_to_show / 2);
      if (!($max_pagination_links_to_show % 2)) $start_page--;

      if ($start_page < 1)
      {
         $end_page += 1 - $start_page;
         $start_page = 1;
      }
      else if ($end_page > $total_pages)
      {
         $start_page -= $end_page - $total_pages;
         $end_page = $total_pages;
      }


       // do we need to insert elipses?
       //
       if (1 < $start_page)
       {
          $start_page++;
          $show_elipses_before = TRUE;
       }
       if ($total_pages > $end_page)
       {
          $end_page--;
          $show_elipses_after = TRUE;
       }
   }



   // now build the output
   //
   $pagination_str = '';
   if ($small_paginator_text) $pagination_str = '<small>';
   if ($show_elipses_before)
      $pagination_str .= get_abook_paginator_link($backend_number, 1, 1, $page_size)
                      . $nbsp . '...' . $nbsp;
   for ($x = $start_page; $x <= $end_page; $x++)
   {
      if ($x == $abook_entry['page_number'])
         $pagination_str .= $x . $nbsp;
      else
         $pagination_str .= get_abook_paginator_link($backend_number, $x, $x, $page_size) . $nbsp;
   }
   if ($show_elipses_after)
      $pagination_str .= '...' . $nbsp
                      . get_abook_paginator_link($backend_number, $total_pages, $total_pages, $page_size) . $nbsp;
    if ($small_paginator_text) $pagination_str .= '</small>';

    return $pagination_str;

}



/**
  * Generate group selector string if needed
  *
  * If a user has any address groups defined, a selector is built
  * containing them, otherwise, an empty string is returned.
  *
  * @param int $backend The backend number for which the selector
  *                     is being built.
  *
  * @return string The group selector string
  *
  */
function get_abook_group_selector_string($backend)
{

   global $data_dir, $username, $agp_grouping,
          $agp_group_selections, $agp_abook_backends,
          $agp_pagination;
//TODO: note that we could actually put $agp_unrelated_get_args into a hidden form field, but this should not be necessary (really, even saving them when making GET links is overkill)

   abook_group_pagination_init();

   $agp_groups = array();
   if ($agp_grouping)
      $agp_groups = getPref($data_dir, $username,
                            'agp_groups', $agp_groups);
   if (!empty($agp_groups) && is_string($agp_groups))
      $agp_groups = unserialize($agp_groups);

   if (empty($agp_groups))
      return '';


   // save the page number(s) if we can find them (only needed in 1.4.x)
   //
   // note that we only need to push these to the form once,
   // so we keep track of whether or not we've done that yet
   //
   $ret = '';
   if (!check_sm_version(1, 5, 2))
   {
      static $agp_save_pagination_link_uris = FALSE;
      if (!$agp_save_pagination_link_uris)
      {
         $agp_save_pagination_link_uris = TRUE;

         foreach ($agp_abook_backends as $backend)
         {
            if (!empty($agp_pagination[$backend]['page_number']))
               $ret .= '<input type="hidden" name="backend_page_number_'
                    . $backend . '" value="' . $agp_pagination[$backend]['page_number']
                    . '" />';
            if (!empty($agp_pagination[$backend]['page_size']))
               $ret .= '<input type="hidden" name="backend_page_size_'
                    . $backend . '" value="' . $agp_pagination[$backend]['page_size']
                    . '" />';
         }
      }
   }


   // grab groups that are already selected, if any
   //
   if (!empty($agp_group_selections[$backend])
    && is_array($agp_group_selections[$backend]))
      $groups = $agp_group_selections[$backend];
   else
      $groups = array();


   // build the selector
   //
   sq_change_text_domain('abook_group_pagination');
   //$size = min(3, sizeof($agp_groups));
   $size = 4;


   // templated output for 1.5.2+
   //
   if (check_sm_version(1, 5, 2))
   {
      global $oTemplate;
      $oTemplate->assign('size', $size);
      $oTemplate->assign('backend', $backend);
      $oTemplate->assign('all_groups', $agp_groups);
      $oTemplate->assign('selected_groups', $groups);
      $ret .= $oTemplate->fetch('plugins/abook_group_pagination/group_selector.tpl');
   }


   // 1.4.x...
   //
   else
   {
      $ret .= '<table width="1%" cellpadding="0" cellspacing="2"><tr><td>'
           . '<select name="abook_groups_' . $backend . '[]" multiple="multiple" size="' . $size . '">';
      foreach ($agp_groups as $group)
      {
         $ret .= '<option value="' . $group . '"'
              . (in_array($group, $groups) ? ' selected="selected"' : '')
              . '>' . $group . "</option>\n";
      }
      $ret .= '</select></td><td valign="bottom">'
           . '<input type="submit" name="abook_groupdel_' . $backend . '" value="'
           . _("Remove") . '" /><br />'
           . '<input type="submit" name="abook_groupadd_' . $backend . '" value="'
           . _("Add") . '" /><br />'
           . '<input type="submit" name="abook_group_filter_'
           . $backend . '" value="' . _("Go")
           . '" /></td></tr></table>';
   }


   sq_change_text_domain('squirrelmail');

   return $ret;

}



/**
  * Add pagination and group selector to address book table header
  *
  * A copy of the pagination links is placed on the right when there
  * is no group selector to be shown.
  *
  * @param array $abook_entry An array describing one of the address
  *                           book entries in the target address book
  *                           backend, which should contain pagination
  *                           and grouping information in it.
  *
  * @return string The pagination and/or group selectors for the
  *                table header, or an empty string if none needed.
  *
  */
function add_pagination_and_grouping_do($abook_entry)
{

   // 1.5.2 only needs the templated group selection widgets
   //
   if (check_sm_version(1, 5, 2))
   {
      global $current_backend;
      return array('address_book_filter' => get_abook_group_selector_string($current_backend));
   }


   global $color;
   $nbsp = '&nbsp;';

   $paginator = get_abook_paginator_string($abook_entry);

   // play nice with the Select Range plugin
   //
   if (is_plugin_enabled('select_range'))
   {
      global $alist;
      include_once(SM_PATH . 'plugins/select_range/functions.php');
      $links = build_abook_links(sizeof($alist));
      if (!empty($links)) foreach ($links as $link)
         if (!empty($link))
            $paginator .= (empty($paginator) ? '' : '|&nbsp;') . $link . '&nbsp;';
   }

   if (!empty($paginator))
      $pagination_string = html_tag('td', $paginator, 'left', $color[0], 'width="50%"');

   $group_selector = get_abook_group_selector_string($abook_entry['backend']);
   if (!empty($group_selector))
      $grouping_string = html_tag('td', $group_selector, 'right', $color[0], 'width="50%"');

   if (empty($pagination_string) && empty($grouping_string))
      return '';

   if (empty($pagination_string))
      $pagination_string = html_tag('td', $nbsp, '', $color[0], 'width="50%"');

   if (empty($grouping_string))
      $grouping_string = html_tag('td', $paginator, 'right', $color[0], 'width="50%"');

   return $pagination_string . $grouping_string;

}



/**
  * Filter address book search results
  *
  * @param array $args Under 1.4.x, the pre-sorted list of
  *                    addresses to be filtered and paginated.
  *                    Under 1.5.x, an array of the address
  *                    list, the backend number, the page
  *                    number, and the "current page args"
  *                    array
  *
  * @return array The modified list of addresses, or void if the list
  *               was not changed.
  *
  */  
function filter_address_book_listing_do(&$args)
{

   // under 1.5.2, we have to pull out desired hook arguments
   //
   if (check_sm_version(1, 5, 2))
   {
      list($addresses, $current_backend, $page_number, $current_page_args) = $args;
      $addresses = $addresses[$current_backend]['Addresses'];
   }
   else
   {
      $addresses = $args;
      $current_backend = 0;
   }


   if (!is_array($addresses) || empty($addresses))
      return;



   global $data_dir, $username, $agp_abook_pagination,
          $agp_abook_pagination_allow_override, $agp_grouping,
          $agp_abook_backends, $agp_group_selections,
          $agp_pagination, $agp_backends_to_filter,
          $agp_backend_sources;

   abook_group_pagination_init();



   // get config settings and user overrides for
   // pagination (1.4.x only) and grouping functionality
   //
   if (check_sm_version(1, 5, 2))
      $agp_abook_pagination = 0;
   else if ($agp_abook_pagination_allow_override)
      $agp_abook_pagination = getPref($data_dir, $username,
                                     'agp_abook_pagination',
                                     $agp_abook_pagination);
   $agp_groups = array();
   if ($agp_grouping)
      $agp_groups = getPref($data_dir, $username,
                            'agp_groups', $agp_groups);
   if (!empty($agp_groups) && is_string($agp_groups))
      $agp_groups = unserialize($agp_groups);



   // pagination turned off and no groups?  nothing to do here
   //
   if (!$agp_abook_pagination && empty($agp_groups))
      return;



   // process GET/POST arguments for group filters and pagination links
   // all in once place so when building links later (on another hook),
   // we have everything we need
   //
   agp_process_args($addresses, $current_backend);



   // first, process any add/remove group actions and then
   // strip out any addresses not in the groups selected,
   // if any
   //
   // NOTE that abook backends can retrieve address lists by
   // group on their own (much more efficient in most cases
   // if all entries don't have to be gathered first)... if
   // they do so, the backend should put a "grouped" entry
   // in each of the addresses in the list it returns, where
   // the value for that entry is equivalent to boolean TRUE.
   //
   // In calculating what group(s) to retrieve, the backend
   // would need to use the values placed in the page request
   // by this plugin:  "abook_groups_x", where "x" is the
   // backend number.  The value for it will be an array
   // containing the selected group names.
   //
   // The backend will also need to detect and act upon the
   // actions this plugin generates for adding and removing
   // addresses to/from group(s).  It should add any addresses
   // it finds that are appropriate in the page request
   // variable called "sel" when there is a page request
   // variable present called "abook_groupadd_x", where "x"
   // is the backend number.  Likewise, removals should be
   // processed using "sel" and "abook_groupdel_x".  The
   // groups to be added to or removed from are in the same
   // "abook_groups_x" explained above.
   //
   if (!empty($agp_groups))
   {

      if (!sqgetGlobalVar('sel', $sel, SQ_POST))
         $sel = array();
      $grouped_addresses = array();



      // get user's address book group contents
      //
      $address_group_contents = getPref($data_dir, $username, 'agp_group_contents', '');
      if (empty($address_group_contents))
         $address_group_contents = array();
      else
         $address_group_contents = unserialize($address_group_contents);


      // add/remove addresses to/from selected groups?
      //
      $contents_changed = FALSE;
      if (!empty($sel)) foreach ($agp_abook_backends as $backend)
      {

         // add
         //
         if (sqGetGlobalVar('abook_groupadd_' . $backend, $ignore, SQ_POST))
         {
            foreach ($agp_group_selections[$backend] as $group)
            {
               if (empty($address_group_contents[$group]))
                  $address_group_contents[$group] = array();
               foreach ($sel as $selected_id)
                  if (!in_array($selected_id, $address_group_contents[$group]))
                  {
                     $contents_changed = TRUE;
                     $address_group_contents[$group][] = $selected_id;
                  }
            }
            break;  // only one add button can be pressed at a time
         }


         // remove
         //
         if (sqGetGlobalVar('abook_groupdel_' . $backend, $ignore, SQ_POST))
         {
            foreach ($agp_group_selections[$backend] as $group)
            {
               if (empty($address_group_contents[$group]))
                  $address_group_contents[$group] = array();
               foreach ($sel as $selected_id)
               {
                  $contents_changed = TRUE;
                  $address_group_contents[$group] = array_diff($address_group_contents[$group], array($selected_id));
               }
            }
            break;  // only one remove button can be pressed at a time
         }

      }



      // update user's address book group contents if they changed
      //
      if ($contents_changed)
         setPref($data_dir, $username, 'agp_group_contents', serialize($address_group_contents));



      // now filter!
      //
      if (check_sm_version(1, 5, 2))
      { $nickname = 'NickName'; $separator = '_'; }
      else
      { $nickname = 'nickname'; $separator = ':'; }
      foreach ($addresses as $abook_entry)
      {

         // in 1.5.2+ the backend is not specified in the entry
         //
         if (check_sm_version(1, 5, 2))
            $abook_entry['backend'] = $current_backend;

         // we are filtering this entry's backend,
         // so check against selected groups
         //
         if ($agp_backends_to_filter[$abook_entry['backend']]
          && empty($abook_entry['grouped']))
         {
            foreach ($address_group_contents as $group => $contents)
               if (in_array($group, $agp_group_selections[$abook_entry['backend']])
                && in_array($abook_entry['backend'] . $separator . $abook_entry[$nickname], $contents))
               {
                  $grouped_addresses[] = $abook_entry;
                  break; // needed for 1.5.2+
               }
         }

         // not filtering this entry's backend,
         // so it just gets copied in
         //
         else
         {
            $grouped_addresses[] = $abook_entry;
         }

      }


      // insert special message in any backend that is
      // now empty as a result of filtering
      //
      $backend_totals = array();
      foreach ($grouped_addresses as $abook_entry)
      {
         if (empty($backend_totals[$abook_entry['backend']]))
            $backend_totals[$abook_entry['backend']] = 1;
         else
            $backend_totals[$abook_entry['backend']]++;
      }
      foreach ($agp_abook_backends as $backend)
      {
         if (empty($backend_totals[$backend]))
         {
            sq_change_text_domain('abook_group_pagination');
            $grouped_addresses[] = array(
                                         'backend' => $backend,
                                         'source' => $agp_backend_sources[$backend],
                                         'special_message' => _("No addresses found; please relax group filters"),
                                        );
            sq_change_text_domain('squirrelmail');
         }
      }

   }
   else $grouped_addresses = $addresses;



   // now paginate the addresses (each backend seperately)
   //
   // NOTE that abook backends can paginate on their own
   // (much more efficient in most cases if all entries
   // don't have to be gathered first)... if they do so,
   // the backend should put a "paginated" entry in each
   // of the addresses in the list it returns, where the
   // value for that entry is equivalent to boolean TRUE.
   //
   // The backend also needs to populate a "total_addresses" 
   // entry, which contains the total (unpaginated) number
   // of addresses in the backend (used for generating
   // pagination links).
   //
   // In calculating what page size and page number to
   // retrieve, the backend would need to use the values
   // placed in the page request by this plugin, such as
   // "backend_page_number_x" and "backend_page_size_x",
   // where "x" is the backend number.
   //
   // Note that this code only executes under SM 1.4.x
   //
   if ($agp_abook_pagination && !empty($grouped_addresses) && is_array($grouped_addresses))
   {

      $paginated_addresses = array();



      // first, have to iterate the whole list and count the
      // total addresses from each backend
      //
      foreach ($grouped_addresses as $abook_entry)
      {

         // skip entries that are just special messages
         //
         if (!empty($abook_entry['special_message'])) continue;


         if (empty($agp_pagination[$abook_entry['backend']]['total_entries']))
         {

            if (empty($agp_pagination[$abook_entry['backend']]['page_number']))
               $agp_pagination[$abook_entry['backend']]['page_number'] = 1;

            if (empty($agp_pagination[$abook_entry['backend']]['page_size']))
               $agp_pagination[$abook_entry['backend']]['page_size'] = $agp_abook_pagination;

            $agp_pagination[$abook_entry['backend']]['total_entries'] = 1;


            // is this backend already paginated?
            //
            if (!empty($abook_entry['paginated']))
               $agp_pagination[$abook_entry['backend']]['paginated'] = TRUE;
            else
               $agp_pagination[$abook_entry['backend']]['paginated'] = FALSE;

         }
         else
            $agp_pagination[$abook_entry['backend']]['total_entries']++;

      }



      // then we actually paginate, stripping all entries
      // not on the current page(s)
      //
      // This makes the huge assumption that all entries
      // for any one backend are contiguous!  this is the
      // current defined behavior based on how the sort
      // function works: see function alistcmp() in
      // functions/addressbook.php in the SquirrelMail core
      //
      $backends_seen = array();
      foreach ($grouped_addresses as $abook_entry)
      {

         // skip entries that are just special messages
         //
         if (!empty($abook_entry['special_message']))
         {
            $paginated_addresses[] = $abook_entry;
            continue;
         }


         if (empty($backends_seen[$abook_entry['backend']]))
         {

            $backends_seen[$abook_entry['backend']] = 1;

            $page_item_count = 0;
            $page_count = 1;

            // roll back page number if it is beyond max pages
            //
            $total_pages = ceil($agp_pagination[$abook_entry['backend']]['total_entries'] / $agp_pagination[$abook_entry['backend']]['page_size']);
            if ($total_pages < $agp_pagination[$abook_entry['backend']]['page_number'])
               $agp_pagination[$abook_entry['backend']]['page_number'] = $total_pages;

         }

         $page_item_count++;

         if ($page_item_count > $agp_pagination[$abook_entry['backend']]['page_size'])
         {
            $page_count++;
            $page_item_count = 1;
         }

         if ($page_count == $agp_pagination[$abook_entry['backend']]['page_number']
          || $agp_pagination[$abook_entry['backend']]['paginated'])
         {

            // these array keys and their values are needed to calculate
            // and show pagination links elsewhere in this plugin
            //
            $abook_entry['page_size']
               = $agp_pagination[$abook_entry['backend']]['page_size'];
            $abook_entry['page_number']
               = $agp_pagination[$abook_entry['backend']]['page_number'];
            if (empty($abook_entry['total_addresses']))
               $abook_entry['total_addresses']
                  = $agp_pagination[$abook_entry['backend']]['total_entries'];

            $paginated_addresses[] = $abook_entry;

         }

      }

   }
   else $paginated_addresses = $grouped_addresses;



   // in 1.5.2, just modify hook args directly
   //
   if (check_sm_version(1, 5, 2))
   {
      $args[0][$current_backend]['Addresses'] = $paginated_addresses;
      $args[3]['abook_groups_' . $current_backend] = $agp_group_selections[$current_backend];
   }


   return $paginated_addresses;

}



/**
  * Looks at the page request arguments and pulls together
  * any pagination links and group filters and makes
  * them available globally for creating new such links.
  * The following global variables are populated:
  *
  * $agp_group_selections is an array keyed by backend
  * number where values are arrays containing the
  * groups that were selected in the group filter for
  * that backend.
  *
  * $agp_backends_to_filter is an array keyed by
  * backend number where values are boolean, indicating
  * if group filtering for the backend is turned on.
  *
  * $agp_pagination is an array keyed by backend
  * number where values are arrays containing two
  * keys: "page_number" and "page_size", whose
  * corresponding values are 0 (zero) if they were
  * not found in the page request.
  *
  * This function also determines the number of backends
  * and puts them in a global array called $agp_abook_backends.
  *
  * This function also saves off all GET query string
  * arguments that are UNRELATED to this plugin in a
  * global variable called $agp_unrelated_get_args.
  * The contents of that string will end with a "?"
  * or "&amp;" so that it is ready for appending our
  * own arguments to it for use in links produced in
  * this plugin.
  *
  * @param array $addresses The full (unfiltered) list
  *                         of addresses queried for
  *                         the address book page from
  *                         which we determine the
  *                         number of backends.
  * @param int   $backend   The backend number to use
  *                         when only one should be used
  *                         (OPTIONAL; default not used;
  *                         will inspect all backends).
  *
  */
function agp_process_args($addresses, $backend=0)
{

   global $agp_abook_backends, $agp_group_selections,
          $agp_pagination, $agp_unrelated_get_args,
          $agp_backends_to_filter, $agp_backend_sources;


   // figure out how many backends we are dealing with
   //
   if ($backend)
   {
      $agp_abook_backends[] = $backend;
//TODO/FIXME: not sure what to do with this... but for now, this shouldn't be a problem, since this code is intended for 1.5.2+ where this is not used
      $agp_backend_sources[$backend] = 'UNKNOWN';
   }
   else
   {
      $agp_abook_backends = array();
      foreach ($addresses as $address)
         if (!in_array($address['backend'], $agp_abook_backends))
         {
            $agp_abook_backends[] = $address['backend'];
            $agp_backend_sources[$address['backend']] = $address['source'];
         }
   }


   // pull out pagination and group filter details from
   // the page request for each backend and make them
   // globally available
   //
   foreach ($agp_abook_backends as $backend)
   {

      if (!sqGetGlobalVar('abook_groups_' . $backend, $agp_group_selections[$backend], SQ_FORM))
//TODO:FIXME: in 1.5.2, this prevents keeping current group filters when changing backends, since it's one backend at a time... in this if(), we could attempt to search for other SQ_FORM args abook_groups_X with a different number to see if there is one(?) and use it(?)
         $agp_group_selections[$backend] = array();

      // turn filtering on for all backends unless their
      // group list is empty (don't need to use the page
      // request variable called "abook_group_filter_x")
      //
      if (empty($agp_group_selections[$backend]))
         $agp_backends_to_filter[$backend] = FALSE;
      else
         $agp_backends_to_filter[$backend] = TRUE;


      if (!sqGetGlobalVar('backend_page_number_' . $backend, $agp_pagination[$backend]['page_number'], SQ_FORM))
         $agp_pagination[$backend]['page_number'] = 0;

      if (!sqGetGlobalVar('backend_page_size_' . $backend, $agp_pagination[$backend]['page_size'], SQ_FORM))
         $agp_pagination[$backend]['page_size'] = 0;

   }



   // save any GET query string arguments that are not
   // related to this plugin
   //
   $php_self = php_self();
   $args_start_pos = strpos($php_self, '?');
   if ($args_start_pos !== FALSE)
   {
      $agp_unrelated_get_args = htmlspecialchars(substr($php_self, $args_start_pos));

      // remove links for all pagination and group filtering
      //
      $agp_unrelated_get_args = preg_replace(array('/(&amp;|\?)backend_(page_number|page_size)_\d+=\d*/',
                                                   '/(&amp;|\?)abook_group(_filter_\d+|s_\d+\[])=[^?&]*/',
                                                   '/(&amp;|\?)abook_group(add|del)_\d+=[^?&]*/'),
                                             '', $agp_unrelated_get_args);

      if (empty($agp_unrelated_get_args))
         $agp_unrelated_get_args = '?';
      else if ($agp_unrelated_get_args[0] == '&')
         $agp_unrelated_get_args = '?' . substr($agp_unrelated_get_args, 5) . '&amp;';
      else if ($agp_unrelated_get_args[0] != '?')
         $agp_unrelated_get_args = '?' . $agp_unrelated_get_args . '&amp;';
      else
         $agp_unrelated_get_args .= '&amp;';
   }
   else
      $agp_unrelated_get_args = '?';


   return;

}



